/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.table;

import java.util.Locale;
import java.util.Map;

import org.opengion.hayabusa.db.AbstractTableFilter;
import org.opengion.hayabusa.db.DBTableModel;

import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;

/**
 * TableFilter_UPPER は、TableFilter インターフェースを継承した、DBTableModel 処理用の
 * 実装クラスです。
 *
 * ここでは、指定の カラム(KEY_CLMS) に対して、データを大文字化します。
 * カラムは、CSV形式で指定する事とし、DBTableModel に指定のカラムがなければ無視します。
 *
 * パラメータは、tableFilterタグの keys, vals にそれぞれ記述するか、BODY 部にCSS形式で記述します。
 * 【パラメータ】
 *  {
 *       KEY_CLMS : AA,BB,CC ;    データを大文字化したいカラム名を、CSV形式で指定します。
 *  }
 *
 * @og.formSample
 * ●形式：
 *      ① &lt;og:tableFilter classId="UPPER" keys="KEY_CLMS" vals='"TABLE_NAME,CLM"' /&gt;
 *
 *      ② &lt;og:tableFilter classId="UPPER" &gt;
 *               {
 *                   KEY_CLMS : TABLE_NAME,CLM ;
 *               }
 *         &lt;/og:tableFilter&gt;
 *
 * @og.rev 5.5.5.0 (2012/07/28) 新規追加
 * @og.rev 5.6.6.0 (2013/07/05) keys の整合性チェックを追加
 *
 * @version  0.9.0  2000/10/17
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.1,
 */
public class TableFilter_UPPER extends AbstractTableFilter {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.6.6.1 (2013/07/12)" ;

	/**
	 * keys の整合性チェックを行うための初期設定を行います。
	 *
	 * @og.rev 5.6.6.1 (2013/07/12) keys の整合性チェック対応
	 *
	 * @param	keysMap keys の整合性チェックを行うための Map
	 */
	@Override
	protected void init( final Map<String,String> keysMap ) {
		keysMap.put( "KEY_CLMS"  , "大文字化したいカラム名を、CSV形式で指定"	);
	}

	/**
	 * DBTableModel処理を実行します。
	 *
	 * @return 処理結果のDBTableModel
	 */
	public DBTableModel execute() {
		final DBTableModel table = getDBTableModel();		// 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加

		final String[] keyClms = StringUtil.csv2Array( getValue( "KEY_CLMS" ) );

		if ( keyClms == null || keyClms.length == 0 ) { return table; }		// KEY_CLMSが、存在しなかった。

		int[] keyNos = new int[keyClms.length];
		int lastNo = 0;
		for ( int i = 0; i < keyNos.length; i++ ) {
			final int no = table.getColumnNo( keyClms[i], false );	// 存在しない場合は、-1 を返す。
			if( no >=0 ) { keyNos[lastNo++] = no; }				// -1 でなければ、keyNos に設定する。（前づめ）
		}

		if( lastNo == 0 ) { return table; }		// 対象のカラムが、すべて存在しなかった。

		final int rowCnt = table.getRowCount();
		for( int row=0; row<rowCnt; row++ ) {
			try {
				for( int j=0; j<lastNo; j++ ) {
					final int col = keyNos[j];
					String val = table.getValue( row,col );
					if( val != null ) {
						val = val.toUpperCase( Locale.JAPAN );
						table.setValueAt( val,row,col );
					}
				}
			}
			catch( RuntimeException ex ) {
				final ErrorMessage errMessage = makeErrorMessage( "TableFilter_UPPER Error",ErrorMessage.NG );
				errMessage.addMessage( row+1,ErrorMessage.NG,ex.getMessage() );
				errMessage.addMessage( row+1,ErrorMessage.NG,"KEY_CLMS=[" + StringUtil.array2csv( keyClms ) + "]" );
				errMessage.addMessage( row+1,ErrorMessage.NG,"row=[" + row + "]" );
			}
		}

		return table;
	}
}
