/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.view;

import java.util.List;

import org.opengion.fukurou.util.StringUtil;
// import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.html.TableFormatter;

/**
 * JavaScript のツリー階層を持ったテーブル表示を行う、ツリーテーブル表示クラスです。
 *
 * AbstractViewForm により、setter/getterメソッドのデフォルト実装を提供しています。
 * 各HTMLのタグに必要な setter/getterメソッドのみ，追加定義しています。
 *
 * AbstractViewForm を継承している為,ロケールに応じたラベルを出力させる事が出来ます。
 *
 * @og.group 画面表示
 *
 * @version  4.0
 * @author   Hiroki Nakamura
 * @since    JDK5.0,
 */
public class ViewForm_HTMLCustomTreeBOM extends ViewForm_HTMLTable  {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.2.0.1 (2015/03/06)" ;

	private TableFormatter 		headerFormat	;
	private TableFormatter[]	bodyFormats		;
	private int					bodyFormatsCount;

	private static final int BODYFORMAT_MAX_COUNT = 10;

	/**
	 * DBTableModel から HTML文字列を作成して返します。
	 * startNo(表示開始位置)から、pageSize(表示件数)までのView文字列を作成します。
	 * 表示残りデータが pageSize 以下の場合は,残りのデータをすべて出力します。
	 *
	 * @og.rev 4.3.1.0 (2008/09/08) フォーマットが設定されていない場合のエラー追加
	 * @og.rev 6.2.0.0 (2015/02/27) フォーマット系の noDisplay 対応
	 *
	 * @param  stNo     表示開始位置
	 * @param  pgSize   表示件数
	 *
	 * @return  DBTableModelから作成された HTML文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String create( final int stNo, final int pgSize )  {
		// このクラスでは、テーブル全データを使用します。
		if( getRowCount() == 0 ) { return ""; }	// 暫定処置

		// 4.3.1.0 (2008/09/08)
		if( headerFormat == null ) {
			final String errMsg = "ViewTagで canUseFormat() = true の場合、Formatter は必須です。";
			throw new HybsSystemException( errMsg );
		}

		final int startNo = 0;
		final int pageSize = getRowCount();

		final int lastNo = getLastNo( startNo, pageSize );

		final StringBuilder out = new StringBuilder( BUFFER_LARGE );

		headerFormat.makeFormat( getDBTableModel() );
		// 6.2.0.0 (2015/02/27) フォーマット系の noDisplay 対応
		setFormatNoDisplay( headerFormat );

		if( bodyFormatsCount == 0 ) {
			bodyFormats[0] = headerFormat ;
			bodyFormatsCount ++ ;
		}
		else {
			for( int i=0; i<bodyFormatsCount; i++ ) {
				bodyFormats[i].makeFormat( getDBTableModel() );
				// 6.2.0.0 (2015/02/27) フォーマット系の noDisplay 対応
				setFormatNoDisplay( bodyFormats[i] );
			}
		}

		out.append( getHeader() );

		int level;
		boolean isFld;
		for( int row=startNo; row<lastNo; row++ ) {
			// カラム==０は、レベルを指定する。
			level = Integer.parseInt( getValueLabel(row,0) );
			isFld = false;
			if( row+1<lastNo ) {
				final int nextLevel = Integer.parseInt( getValueLabel(row+1,0) );
				isFld = level < nextLevel ;
			}
			out.append( getLevelScript( level,isFld ) );

			// 開始
			for( int i=0; i<bodyFormatsCount; i++ ) {
				final TableFormatter bodyFormat = bodyFormats[i];

				int cl = 0;
				for( ; cl < bodyFormat.getLocationSize(); cl++ ) {
					String fmt = bodyFormat.getFormat(cl);
					final int loc = bodyFormat.getLocation(cl);
					if( ! bodyFormat.isNoClass() && loc >= 0 ) {
//						final StringBuilder newtg = new StringBuilder( BUFFER_LARGE )
//							.append("<td class=\"")
//							.append( getColumnDbType(loc) )
//							.append("\" ");
////						final String tdclass = newtg.toString();
////						fmt = StringUtil.replace( bodyFormat.getFormat(cl) ,"<td", tdclass );
//						fmt = StringUtil.replace( bodyFormat.getFormat(cl) ,"<td", newtg.toString() );
						final String tdclass = "<td class=\"" + getColumnDbType(loc) + "\" ";
						fmt = StringUtil.replace( bodyFormat.getFormat(cl) ,"<td", tdclass );
					}
					out.append( fmt );
					if( loc >= 0 ) {
						switch( bodyFormat.getType(cl) ) {
						case '#' : out.append( getColumnLabel(loc) );		break;
						case '$' : out.append( getRendererValue(row,loc) );	break;
						case '!' : out.append( getValue(row,loc) );			break;
						default  : out.append( getValueLabel(row,loc) );	break;
						}
					}
				}
				out.append( StringUtil.replace( bodyFormat.getFormat(cl), "</tr>", "" ) );
			}
			// 終了

			out.append( "', '', 'gold')" );
			if( level != 0 ) {
				out.append( ')' );		// 6.0.2.5 (2014/10/31) char を append する。
			}
			out.append( CR );
		}
		out.append( getFutter() );

		return out.toString();
	}

	/**
	 * DBTableModel から テーブルのヘッダータグ文字列を作成して返します。
	 * JavaScript の TreeBody では、JavaScriptに関連する定義もこのヘッダーに
	 * 含めます。
	 *
	 * @return  テーブルのヘッダータグ文字列
	 * @og.rtnNotNull
	 */
	@Override
	protected String getHeader() {
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
			.append("<table border=\"0\" cellspacing=\"2\" cellpadding=\"0\"  summary=\"bomTable\" id=\"viewTable\">")	// 3.9.0.1 (2007/12/18)
			.append( CR )
			.append("<script type=\"text/javascript\">")
			.append( CR )
			.append("<!--")
			.append( CR )
			.append("aux0 = gFld('");

		int cl = 0;
		for( ; cl < headerFormat.getLocationSize(); cl++ ) {
			buf.append( StringUtil.replace( headerFormat.getFormat(cl) ,"td","th" ));
			final int loc = headerFormat.getLocation(cl);
			if( loc >= 0 ) { buf.append( getColumnLabel(loc) ); }
			// ヘッダーフォーマット部では、何もしません。
		}
		buf.append( StringUtil.replace( StringUtil.replace( headerFormat.getFormat(cl) ,"td","th" ), "</tr>", "" ) )
			.append("', '', 'gold')")
			.append( CR );

		return buf.toString();
	}

	/**
	 * DBTableModel から テーブルのフッタータグ文字列を作成して返します。
	 * JavaScript の TreeBody では、JavaScriptに関連する定義もこのフッターに
	 * 含めます。
	 *
	 * @return  テーブルのフッタータグ文字列
	 * @og.rtnNotNull
	 */
	protected String getFutter() {
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
			.append("initializeDocument()").append( CR )
			.append("//-->").append( CR )
			.append("</script>").append( CR )
			.append("</table>").append( CR );

		return buf.toString();
	}

	/**
	 * 行のレベルに応じた JavaScript関数のヘッダー部分を返します。
	 *
	 * @og.rev 3.5.2.1 (2003/10/27) JavaScript 内のダブルコーテーションをシングルコーテーションに変更する。
	 *
	 * @param	lvl		ツリーのレベル
	 * @param	isFld	フォルダかどうか[true:フォルダ/false:最下層]
	 *
	 * @return  JavaScript関数のヘッダー部分
	 */
	private String getLevelScript( final int lvl,final boolean isFld ) {

		final String auxX = "\taux" + ( lvl );
		final String auxY = "aux" + ( lvl-1 );

		final String rtn ;
		if( isFld ) {
			rtn = auxX + " = insFld(" + auxY + ", gFld('";
		}
		else {
			rtn = "\tinsFld(" + auxY + ", gLnk('CONTENTS','";
		}

		return rtn;
	}

	/**
	 * フォーマットを設定します。
	 *
	 * @param	list	TableFormatterのリスト
	 */
	@Override
	public void setFormatterList( final List<TableFormatter> list ) {		// 4.3.3.6 (2008/11/15) Generics警告対応
		bodyFormats = new TableFormatter[BODYFORMAT_MAX_COUNT];

		bodyFormatsCount = 0;
		for( int i=0; i<list.size(); i++ ) {
			final TableFormatter format = list.get( i );		// 4.3.3.6 (2008/11/15) Generics警告対応
			switch( format.getFormatType() ) {
			case TYPE_HEAD : headerFormat = format; break;
			case TYPE_BODY : bodyFormats[bodyFormatsCount++] = format; break;
			default : final String errMsg = "FormatterType の定義外の値が指定されました。";
			// 4.3.4.4 (2009/01/01)
					  throw new HybsSystemException( errMsg );
			}
		}

		if( headerFormat == null ) {
			final String errMsg = "og:thead タグの、フォーマットの指定は必須です。";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * フォーマットメソッドを使用できるかどうかを問い合わせます。
	 *
	 * @return  使用可能(true)/ 使用不可能 (false)
	 */
	@Override
	public boolean canUseFormat() {
		return true;
	}

	/**
	 * ビューで表示したカラムの一覧をCSV形式で返します。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規追加
	 * @og.rev 6.2.0.1 (2015/03/06) TableFormatter#getLocation(int)の有効判定
	 *
	 * @return	ビューで表示したカラムの一覧
	 * @og.rtnNotNull
	 */
	@Override
	public String getViewClms() {
		final DBTableModel table = getDBTableModel();
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
		for( int i=0; i<headerFormat.getLocationSize(); i++ ) {
			if( buf.length() > 0 ) { buf.append( ',' ); }
			// 6.2.0.1 (2015/03/06) TableFormatter#getLocation(int)の有効判定
//			buf.append( table.getColumnName( headerFormat.getLocation( i ) ) );
			final int loc = headerFormat.getLocation(i);
			if( loc >= 0 ) { buf.append( table.getColumnName( loc ) ); }
		}
		return buf.toString();
	}
}
