/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import java.math.BigDecimal;
import java.text.NumberFormat;

import org.opengion.fukurou.model.NativeType;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.db.AbstractDBType;
import org.opengion.hayabusa.db.DBTypeCheckUtil;

/**
 * 半角小数付き数字の NUMBER を扱う為の、カラム属性を定義します。
 *
 * '0' ～ '9' ,'-' ,'.'でのみ構成されている数字型カラム属性を定義します。
 * さらに、カンマ','が含まれていても OK とします。ただし、データからは取り除きます。
 *
 * タイプチェックとして、以下の条件を判定します。
 * ・０～９、マイナス(-)および小数点(.)を許可
 * ・整数部の長さチェック
 * ・小数部の長さチェック
 * ・小数点の位置チェック
 * ・符号の位置チェック
 * ・文字パラメータの 正規表現チェック
 *
 * @og.group データ属性
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DBType_R extends AbstractDBType {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.6.0.3 (2012/01/24)" ;

	private static final String DEF_VALUE = "0.00" ;		// データのデフォルト値

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) type 廃止
	 */
	public DBType_R() {
		super( DEF_VALUE );
	}

	/**
	 * NATIVEの型の識別コードを返します。
	 *
	 * @og.rev 3.5.4.7 (2004/02/06) 新規作成
	 * @og.rev 4.1.1.2 (2008/02/28) Enum型(fukurou.model.NativeType)に変更
	 *
	 * @return  NATIVEの型の識別コード(DBType で規定)
	 * @see org.opengion.fukurou.model.NativeType
	 */
	@Override
	public NativeType getNativeType() {
		return NativeType.DOUBLE;
	}

	/**
	 * 半角０文字の固定長でFILL埋めされた文字列を返します。
	 * なお、エラーチェックは行われません。
	 * 実行前に、必ず valueCheck( String value ,int len ) が行われる必要があります。
	 *
	 * @og.rev 3.5.4.5 (2004/01/23) エンコード指定に変更します。
	 *
	 * @param	value	 ＦＩＬＬ埋めする文字列
	 * @param	sizeX	 整数部分の文字列の長さ
	 * @param	sizeY	 小数部分の文字列の長さ
	 * @param   encode   固定長で変換する文字エンコード
	 *
	 * @return	ＦＩＬＬ埋めした新しい文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String valueFill( final String value ,final int sizeX ,final int sizeY,final String encode ) {

		final NumberFormat format = NumberFormat.getInstance();
		format.setMinimumIntegerDigits( sizeX );
		format.setMinimumFractionDigits( sizeY );

		final double dd  = StringUtil.parseDouble( value );
//		final String rtn = format.format( dd );
//
//		return rtn;
		return format.format( dd );				// 6.1.1.0 (2015/01/17) refactoring
	}

	/**
	 * String引数の文字列を＋１した文字列を返します。
	 * これは、英字の場合(A,B,C など)は、B,C,D のように,最終桁の文字コードを
	 * ＋１ します。
	 * 文字列が数字タイプの場合は, 数字に変換して、+1 します。(桁上がりもあり)
	 * 混在タイプの場合は,最後の桁だけを確認して ＋１します。
	 * 引数が null の場合と、ゼロ文字列("")の場合は,物理的初期設定値(String getDefault())
	 * の値を返します。
	 *
	 * @og.rev 3.3.0.0 (2003/06/23) 値を＋１していた箇所を、Double から、BigDecimal に変更。
	 *
	 * @param	value	String引数の文字列
	 *
	 * @return	String引数の文字列を＋１した文字列
	 */
	@Override
	public String valueAdd( final String value ) {
		if( value == null || value.isEmpty() ) { return getDefault(); }
		final BigDecimal val = new BigDecimal( value ).add( BigDecimal.ONE );
		return val.toString();
	}

	/**
	 * String引数の文字列に、第２引数に指定の文字列(数字、日付等)を加算して返します。
	 *
	 * ここでは、小数文字列に、別の数字（小数も可）文字列を加算します。
	 * 内部処理に、BigDecimal を使用しています。
	 *
	 * @og.rev 5.6.0.3 (2012/01/24) ADD に、引数の値を加算する機能を追加します。
	 *
	 * @param   value  String引数
	 * @param   add    加算する数字文字列
	 *
	 * @return  引数の文字列に数字を加算します。
	 */
	@Override
	public String valueAdd( final String value,final String add ) {
		if( value == null || value.isEmpty() ) { return getDefault(); }

		BigDecimal addSu = BigDecimal.ONE;
		if( add != null && !add.isEmpty() ) {
			addSu = new BigDecimal( add );
		}

		final BigDecimal val = new BigDecimal( value ).add( addSu );
		return val.toString();
	}

	/**
	 * エディターで編集されたデータを登録する場合に、データそのものを
	 * 変換して、実登録データを作成します。
	 * 例えば,大文字のみのフィールドなら、大文字化します。
	 * 実登録データの作成は、DBType オブジェクトを利用しますので,
	 * これと Editor とがアンマッチの場合は、うまくデータ変換
	 * されない可能性がありますので、注意願います。
	 *
	 * @og.rev 3.3.3.0 (2003/07/09) 前後のスペースを取り除いておく。
	 * @og.rev 3.3.3.1 (2003/07/18) 後ろスペースを取り除く。(StringUtil#rTrim)
	 *
	 * @param	value	(一般に編集データとして登録されたデータ)
	 *
	 * @return	修正後の文字列(一般にデータベースに登録するデータ)
	 */
	@Override
	public String valueSet( final String value ) {
		return StringUtil.deleteChar( StringUtil.rTrim( value ),',' );
	}

	/**
	 * データが登録可能かどうかをチェックします。
	 * データがエラーの場合は、そのエラー内容を返します。
	 *
	 * @og.rev 2.1.1.2 (2002/11/21) マイナス記号を含む場合の最大桁数の設定を考慮する。
	 * @og.rev 2.3.1.4 (2003/02/18) 属性チェックを強化した。
	 * @og.rev 3.6.0.0 (2004/09/22) dbType パラメータを引数に追加
	 * @og.rev 5.2.2.0 (2010/11/01) 厳密にチェック(isStrict=true)するフラグを追加
	 *
	 * @param   key		キー
	 * @param   value	値
	 * @param   sizeX	整数部分の文字列の長さ
	 * @param   sizeY	小数部分の文字列の長さ
	 * @param   typeParam	dbType パラメータ
	 * @param   isStrict	厳密にチェックするかどうか[true:する/false:標準的]
	 *
	 * @return  エラー内容
	 * @og.rtnNotNull
	 */
	@Override
	public ErrorMessage valueCheck( final String key ,final String value ,
									final int sizeX ,final int sizeY ,final String typeParam ,final boolean isStrict) {

		final ErrorMessage msg = new ErrorMessage();
		if( value == null || value.isEmpty() ) { return msg; }

		String check;

		check = DBTypeCheckUtil.decimalFormatCheck( value );
		if( check != null ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,check );
		}

		check = DBTypeCheckUtil.sizeXCheck( value ,sizeX ,sizeY );
		if( check != null ) {
			// 整数部の長さが指定の長さよりも長いです。
			msg.addMessage( 0, ErrorMessage.NG, "ERR0021", key, value, check, String.valueOf( sizeX ) );
		}

		check = DBTypeCheckUtil.sizeYCheck( value ,sizeX ,sizeY );
		if( check != null ) {
			// 小数部の長さが指定の長さよりも長いです。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0022", key, value, check, String.valueOf( sizeY ) );
		}

		check = DBTypeCheckUtil.decimalPointCheck( value );
		if( check != null ) {
			// 小数点の位置が不正です。
			msg.addMessage( 0, ErrorMessage.NG, "ERR0024", key, check );
		}

		check = DBTypeCheckUtil.decimalCodeCheck( value );
		if( check != null ) {
			// 符号の位置が不正です。
			msg.addMessage( 0, ErrorMessage.NG, "ERR0023", key, check );
		}

		// 3.6.0.0 (2004/09/22) dbType パラメータを使用したマッチチェック
		check = DBTypeCheckUtil.matcheCheck( value,typeParam );
		if( check != null ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,check );
		}

		return msg;
	}
}
