/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import java.util.Map;

import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.db.Transaction;
// import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.resource.ResourceManager;

/**
 * TableFilter は、TableFilterTag で使用される DBTableModelの処理用のインターフェースです。
 * 適切な実装(サブクラス)を TableFilterTag に与えることで，DBTableModelの処理を適宜
 * 切り替えることが出来ます。
 *
 * @og.rev 3.8.5.0 (2006/03/20) 新規作成
 * @og.rev 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加
 *
 * @version  0.9.0  2000/10/17
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.1,
 */
public interface TableFilter {
	/**
	 * DBTableModelの処理を実行します。
	 * 処理自体は, 各サブクラスの実装に依存します。
	 *
	 * @return	DBTableModelオブジェクト
	 */
	DBTableModel execute() ;

	/**
	 * DBTableModel をセットします。
	 *
	 * @param	table DBTableModelオブジェクト
	 */
	void setDBTableModel( DBTableModel table ) ;

	/**
	 * DBTableModel を取得します。
	 *
	 * @og.rev 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加
	 *
	 * @return	内部のDBTableModel
	 */
	DBTableModel getDBTableModel() ;

	/**
	 * データ処理の方法(A:追加 C:更新 D:削除)を指定します。
	 *
	 * 通常は、DBTableModel に自動設定されている modifyType を元に、データ処理方法を
	 * 選別します。(A:追加 C:更新 D:削除)
	 * この場合、行単位で modifyType の値を取得して判別する必要がありますが、一般には
	 * 処理対象は、全件おなじ modifyType である可能性が高いです。
	 * また、selectedAll などで強制的に全件処理対象とする場合は、modifyType に値が
	 * 設定さていません。その様な場合に外部より modifyType を指定します。
	 * 初期値は、自動判定 です。
	 *
	 * @param  type データ処理の方法(A:追加 C:更新 D:削除)
	 */
	void setModifyType( String type ) ;

	/**
	 * データ処理の方法(A:追加 C:更新 D:削除)を取得します。
	 *
	 * 初期値は、自動判定 です。
	 *
	 * @og.rev 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加
	 *
	 * @return  データ処理の方法(A:追加 C:更新 D:削除)
	 */
	String getModifyType() ;

	/**
	 * キーと値のペアの変数配列を受け取ります。
	 *
	 * ここでは、この方式以外に、パラメーターMapを受け取る方法もあります。
	 *
	 * keys,vals とパラメーターMapを同時に指定した場合は、両方とも有効です。
	 * ただし、キーが重複した場合は、不定と考えてください。
	 *
	 * @param   keys キー配列
	 * @param   vals 値配列
	 */
	void setKeysVals( String[] keys , String[] vals ) ;

	/**
	 * 選択された行番号の配列をセットします。
	 *
	 * 表示データの HybsSystem.ROW_SEL_KEY を元に、選ばれた 行を
	 * 処理の対象とします。
	 *
	 * @param   rowNo 行番号の配列(可変長引数)
	 */
//	void setParameterRows( int[] rowNo ) ;
	void setParameterRows( int... rowNo ) ;

	/**
	 * 選択された行番号の配列を取得します。
	 *
	 * 表示データの HybsSystem.ROW_SEL_KEY を元に、選ばれた 行を
	 * 処理の対象とします。
	 *
	 * @og.rev 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加
	 *
	 * @return  行番号の配列
	 */
	int[] getParameterRows() ;

	/**
	 * アクセスログ取得の為,Transactionオブジェクトを設定します。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応(新規追加)
	 *
	 * @param   tran Transactionオブジェクト
	 */
	void setTransaction( Transaction tran ) ;

	/**
	 * アクセスログ取得の為,Transactionオブジェクトを取得します。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応(新規追加)
	 * @og.rev 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加
	 *
	 * @return	Transactionオブジェクト
	 */
	Transaction getTransaction() ;

	/**
	 * DBIDを指定します。
	 *
	 * @og.rev 4.2.4.0 (2008/06/23) 新規追加
	 *
	 * @param dbid 接続先ID
	 */
	void setDbid( String dbid ) ;

	/**
	 * DBIDを取得します。
	 *
	 * @og.rev 4.2.4.0 (2008/06/23) 新規追加
	 * @og.rev 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加
	 *
	 * @return DBID(接続先ID)
	 */
	String getDbid() ;

	/**
	 * ボディー部分のSQLを指定します。
	 *
	 * @og.rev 4.2.4.0 (2008/06/23) 新規追加
	 *
	 * @param sql ボディー部分のSQL
	 */
	void setSql( String sql ) ;

	/**
	 * ボディー部分のSQLを取得します。
	 *
	 * @og.rev 4.2.4.0 (2008/06/23) 新規追加
	 * @og.rev 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加
	 *
	 * @return ボディー部分のSQL
	 */
	String getSql() ;

	/**
	 * パラメーターMapを指定します。
	 *
	 * keys,vals と パラメーターMapを同時に指定した場合は、両方とも有効です。
	 * ただし、キーが重複した場合は、不定と考えてください。
	 *
	 * @og.rev  5.6.5.2 (2013/06/21) 新規追加
	 *
	 * @param paramMap パラメーターMap
	 */
	void setParamMap( Map<String,String> paramMap ) ;

	/**
	 * リソースオブジェクトを指定します。
	 *
	 * @og.rev 4.3.7.4 (2009/07/01) 新規追加
	 *
	 * @param resource リソースオブジェクト
	 */
	void setResource( ResourceManager resource ) ;

	/**
	 * リソースオブジェクトを取得します。
	 *
	 * @og.rev 4.3.7.4 (2009/07/01) 新規追加
	 * @og.rev 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加
	 *
	 * @return リソースオブジェクト
	 */
	ResourceManager getResource() ;

	/**
	 * デバッグ情報を表示するかどうか[true/false]を指定します。
	 * true でデバッグ情報を表示します。
	 *
	 * @param   flag 表示するかどうか [true:表示する/false:表示しない]
	 */
	void setDebug( boolean flag ) ;

	/**
	 * デバッグ情報を表示するかどうか[true/false]を取得します。
	 * true でデバッグ情報を表示します。
	 *
	 * @og.rev 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加
	 *
	 * @return  デバッグ情報 [true:表示する/false:表示しない]
	 */
	boolean isDebug() ;

	/**
	 * エラーコード を取得します。
	 * エラーコード は、ErrorMessage クラスで規定されているコードです。
	 *
	 * @return   エラーコード
	 */
	int getErrorCode() ;

	/**
	 * エラーメッセージオブジェクト を取得します。
	 *
	 * @return   エラーメッセージオブジェクト
	 */
	ErrorMessage getErrorMessage() ;
}
