/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.AbstractEditor;
import org.opengion.hayabusa.db.CellEditor;
import org.opengion.hayabusa.db.SelectionCellEditor;					// 6.2.2.0 (2015/03/27)
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.Selection;
import org.opengion.hayabusa.db.SelectionFactory;
import org.opengion.fukurou.util.StringFormat;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.TagBuffer;

import static org.opengion.fukurou.util.StringUtil.isNull;				// 6.1.1.0 (2015/01/17)

/**
 * カラムの編集パラメーターのＳＱＬ文の実行結果より、datalistを作成して
 * 入力候補となるデータリストを定義する編集用エディタークラスです。
 * datalist は、HTML5 から採用されたタグです。
 *
 * 編集パラメータには、datalistを作成するための、SQL文を記述します。
 * このSQL文は、select KEY,LABEL from xx ･･･ という構文で、KEY部分とLABEL部分が
 * 選択されます。
 * datalist 自身が、HTML5からの新機能なので、現時点では、これ以上の機能はありません。
 * 将来的に、DBMENU などと同様に、第三カラム以降を利用可能になると思いますので、
 * 今は使わないでください。（将来の機能追加時に互換性問題を引き起こすかもしれませんので）
 *
 * 入力フィールドとdatalistタグとの関係付は、カラムIDに、"カラムID.sel" で結びつけます。
 *
 * &lt;input name="カラムID" list="カラムID.sel" /&gt;
 * &lt;div style="display:none;"&gt;
 *   &lt;datalist id="カラムID.sel"&gt;
 *     &lt;option value="KEY1"&gt;LABEL1&lt;/option&gt;
 *     &lt;option value="KEY2"&gt;LABEL2&lt;/option&gt;
 *     &lt;option value="KEY3"&gt;LABEL3&lt;/option&gt;
 *   &lt;/datalist&gt;
 * &lt;/div&gt;
 *
 * divタグは、HTML5 非対応ブラウザを使用した場合、datalist の option がそのまま
 * テキストとして見えてしまうのを避けるためです。
 *
 * 一覧表出力時の getValue( int ,String ) 処理では、Selection オブジェクトの
 * キャッシュ機能を利用して、同一Selection オブジェクトの間は、datalist は、
 * １度しか、出力しない様に制御しています。これにより、共有のdatalist を使用する為、
 * HTMLの出力データ量を抑えることが可能になります。
 * (キャッシュを利用しないと１００行出力すると１００個のdatalistを出力する事になります。)
 * (同様の機能を持つ INDBMENU では、行ごとにプルダウンデータを作成しています。)
 * ただし、行単位にSQLの条件を変える機能(AAA:BBB:CCC:DDD引数)が指定された場合は、
 * 行ごとに出力します。
 *
 * 各カラムの値(value値)に、AAA:BBB:CCC:DDD という値を設定できます。これは、
 * $1,$2,$3,$4 に割り当てなおして、QUERYを実行します。また、$1 は、本来の値として、
 * メニューの初期値設定等に使用します。上記の例では、AAA が値で、それ以降は、
 * 引数になります。
 * 又、$Cには自分自身のカラム名を割り当てます。
 * この機能を使用すれば、動的メニューを行ごとに条件を変えて作成することが
 * 可能になります。
 * 例：select KEY,LABEL from xx where KUBUN='$2' and CDK='$3'
 * さらに、元の文字列"AAA:BBB:CCC:DDD"は、$0 に割り当てられます。割り当てがない
 * 変数は、""(ゼロ文字列)として、扱われます。
 *
 * カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.rev 5.7.4.3 (2014/03/28) 新規作成
 * @og.rev 6.2.2.0 (2015/03/27) SelectionCellEditor I/Fを追加
 * @og.group データ編集(HTML5)
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
// public class Editor_DATALIST extends AbstractEditor {
public class Editor_DATALIST extends AbstractEditor implements SelectionCellEditor {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.2.2.0 (2015/03/27)" ;

	// 5.7.5.0 (2014/04/04) datalist 使用時は、display:none にして、HTML5未対応のブラウザに備える。
	private static final String DIV1 = "<div style=\"display:none;\">" ;
	private static final String DIV2 = "</div>" ;

	private final String query	;
	private final String dbid	;
	private final String lang	;
	private final String useSLabel ;		// 6.2.0.0 (2015/02/27) SLABEL 対応
	private final String addKeyLabel ;		// 6.2.0.0 (2015/02/27) キー:ラベル形式

	private Selection bkSel		;			// 5.7.5.0 (2014/04/04) Selection オブジェクトのキャッシュ機能

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 6.2.0.0 (2015/02/27) SLABEL 対応
	 * @og.rev 6.2.0.0 (2015/02/27) キー:ラベル形式で表示するかどうかを、指定できるようにします。
	 */
	public Editor_DATALIST() {
	//	super();
		query	= null;
		dbid	= null;
		lang	= null;
		useSLabel	= "auto";		// 6.2.0.0 (2015/02/27) SLABEL 対応
		addKeyLabel = null;			// 6.2.0.0 (2015/02/27) キー:ラベル形式
	}

	/**
	 * コンストラクター。
	 *
	 * @og.rev 6.2.0.0 (2015/02/27) SLABEL 対応
	 * @og.rev 6.2.0.0 (2015/02/27) キー:ラベル形式で表示するかどうかを、指定できるようにします。
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	private Editor_DATALIST( final DBColumn clm ) {
		super( clm );
		tagBuffer.add( XHTMLTag.inputAttri( attributes ) );

		query		= clm.getEditorParam();
		dbid		= clm.getDbid();
		lang		= clm.getLang();			// 4.0.0.0 (2006/11/15)
		useSLabel	= clm.getUseSLabel() ;		// 6.2.0.0 (2015/02/27) SLABEL 対応
		addKeyLabel = clm.getAddKeyLabel();		// 6.2.0.0 (2015/02/27) キー:ラベル形式

		// 3.5.5.9 (2004/06/07)
//		if( query == null || query.length() == 0 ) {
		if( query == null || query.isEmpty() ) {
			final String errMsg = "DATALIST Editor では、編集パラメータは必須です。"
					+ " name=[" + name + "]" + CR ;
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellEditorオブジェクト
	 * @og.rtnNotNull
	 */
	public CellEditor newInstance( final DBColumn clm ) {
		return new Editor_DATALIST( clm );
	}

	/**
	 * データの編集用文字列を返します。
	 *
	 * ここでは、AAA:BBB:CCC:DDD という値を、$1,$2,$3,$4 に割り当てなおして、
	 * QUERYを実行します。また、$1 は、本来の値として、メニューの初期値設定等に
	 * 使用します。上記の例では、AAA が値で、それ以降は、引数になります。
	 * さらに、元の文字列"AAA:BBB:CCC:DDD"は、$0 に割り当てられます。割り当てがない
	 * 変数は、""(ゼロ文字列)として、扱われます。
	 * 又、$Cには自分自身のカラム名を割り当てます。
	 *
	 * @og.rev 5.7.5.0 (2014/04/04) datalist 使用時は、display:none にして、HTML5未対応のブラウザに備える。
	 * @og.rev 5.7.6.2 (2014/05/16) list属性とid属性の結びつきを、name+".sel" に変更
	 * @og.rev 6.2.0.0 (2015/02/27) SLABEL 対応
	 *
	 * @param	value 入力値
	 *
	 * @return	データの編集用文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final String value ) {
		// input タグの作成
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		final String intag = new TagBuffer( "input" )
						.add( "name"	, name )										// 4.3.6.0 (2009/04/01)
						.add( "id"		, name , isNull( attributes.get( "id" ) ) )		// 4.3.7.2 (2009/06/15)
						.add( "list"    , name + ".sel" )		// datalistタグとの関係付けるためのキーワード
						.add( "value"   , value )
						.add( "size"    , size1 )
						.add( tagBuffer.makeTag() )
						.makeTag();

//		// input タグの作成
//		final TagBuffer intag = new TagBuffer( "input" );
//		intag.add( "name"    , name );
////		if( attributes.get( "id" ) == null || attributes.get( "id" ).length() == 0 ) {
//		if( attributes.get( "id" ) == null || attributes.get( "id" ).isEmpty() ) {
//			intag.add( "id"      , name );
//		}
////		intag.add( "list"    , "dl_" + name );		// datalistタグとの関係付けるためのキーワード
//		intag.add( "list"    , name + ".sel" );		// datalistタグとの関係付けるためのキーワード
//		intag.add( "value"   , value );
//		intag.add( "size"    , size1 );
//		intag.add( tagBuffer.makeTag() );
////		intag.add( optAttr );

		final boolean useSlbl = "true".equalsIgnoreCase( useSLabel );		// 6.2.0.0 (2015/02/27)

		// datalist タグの作成
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		final String dltag = getOption(
					new TagBuffer( "datalist" )
						.add( "id"	, name + ".sel" )	// inputタグとの関係付けるためのキーワード
					, value
					, false								// キャッシュは使用しない。(つまり、null は返らない)
					, useSlbl							// 6.2.0.0 (2015/02/27) SLABEL 対応
				).makeTag() ;

		// DIV1 の display:none は、datalist の optionのBODY部が、HTML5 以外では表示されてしまうのを防ぐため。
		return intag + CR + DIV1 + dltag + DIV2 + CR;

//		// datalist タグの作成
//		TagBuffer dltag = new TagBuffer( "datalist" );
////		dltag.add( "id"    , "dl_" + name );		// inputタグとの関係付けるためのキーワード
//		dltag.add( "id"    , name + ".sel" );			// inputタグとの関係付けるためのキーワード
//
//		dltag = getOption( dltag,value,false );			// キャッシュは使用しない。
//
//		// display:none は、datalist の optionのBODY部が、HTML5 以外では表示されてしまうのを防ぐため。
//		return intag.makeTag()	+ CR +
//				DIV1 + dltag.makeTag() + DIV2 + CR;
	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し,
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * ここでは、AAA:BBB:CCC:DDD という値を、$1,$2,$3,$4 に割り当てなおして、
	 * QUERYを実行します。また、$1 は、本来の値として、メニューの初期値設定等に
	 * 使用します。上記の例では、AAA が値で、それ以降は、引数になります。
	 * さらに、元の文字列"AAA:BBB:CCC:DDD"は、$0 に割り当てられます。割り当てがない
	 * 変数は、""(ゼロ文字列)として、扱われます。
	 * 又、$Cには自分自身のカラム名を割り当てます。
	 *
	 * @og.rev 5.7.5.0 (2014/04/04) datalist 使用時は、display:none にして、HTML5未対応のブラウザに備える。
	 * @og.rev 5.7.5.0 (2014/04/04) Selection オブジェクトのキャッシュ機能
	 * @og.rev 5.7.6.2 (2014/05/16) list属性とid属性の結びつきを、name+".sel" に変更
	 * @og.rev 6.2.0.0 (2015/02/27) SLABEL 対応
	 *
	 * @param	row   行番号
	 * @param	value 入力値
	 *
	 * @return	データ表示/編集用の文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final int row,final String value ) {
		final String name2 = name + HybsSystem.JOINT_STRING + row ;

		// 5.7.5.0 (2014/04/04) Selection オブジェクトのキャッシュ機能 (true:使用可能)
		final boolean useSelCache = value != null && value.indexOf( ':' ) < 0 ;

		final String listId = useSelCache ? name : name2;		// キャッシュを使用する場合は、共通の name を使う。

		// input タグの作成
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		final String intag = new TagBuffer( "input" )
						.add( "name"	, name2 )										// 4.3.6.0 (2009/04/01)
						.add( "id"		, name2 , isNull( attributes.get( "id" ) ) )	// 4.3.7.2 (2009/06/15)
						.add( "list"    , listId + ".sel" )		// datalistタグとの関係付けるためのキーワード
						.add( "value"   , value )
						.add( "size"    , size2 )
						.add( tagBuffer.makeTag() )
						.makeTag( row,value );

//		// input タグの作成
//		final TagBuffer intag = new TagBuffer( "input" );
//		intag.add( "name"    , name2 );
////		if( attributes.get( "id" ) == null || attributes.get( "id" ).length() == 0 ) {
//		if( attributes.get( "id" ) == null || attributes.get( "id" ).isEmpty() ) {
//			intag.add( "id"      , name2 );
//		}
////		intag.add( "list"    , "dl_" + listId );		// datalistタグとの関係付けるためのキーワード
//		intag.add( "list"    , listId + ".sel" );			// datalistタグとの関係付けるためのキーワード
//		intag.add( "value"   , value );
//		intag.add( "size"    , size2 );
//		intag.add( tagBuffer.makeTag() );
////		intag.add( optAttr );

		final boolean useSlbl = "auto".equalsIgnoreCase( useSLabel ) || "true".equalsIgnoreCase( useSLabel );		// 6.2.0.0 (2015/02/27)

		// datalist タグの作成
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		final TagBuffer dltag = getOption(
					new TagBuffer( "datalist" )
						.add( "id"	, listId + ".sel" )			// inputタグとの関係付けるためのキーワード
					, value
					, useSelCache
					, useSlbl									// 6.2.0.0 (2015/02/27) SLABEL 対応
				);

		// キャッシュが効くと、getOption の戻り値は、null になる。
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		if( dltag != null ) {
			return intag + CR + DIV1 + dltag.makeTag( row,value ) + DIV2 + CR ;
		}
		else {
			return intag + CR ;
		}

//		// datalist タグの作成
//		TagBuffer dltag = new TagBuffer( "datalist" );
////		dltag.add( "id"    , "dl_" + listId );			// inputタグとの関係付けるためのキーワード
//		dltag.add( "id"    , listId + ".sel" );			// inputタグとの関係付けるためのキーワード
//
//		dltag = getOption( dltag,value,useSelCache );

//		// キャッシュが効くと、getOption の戻り値は、null になる。
//		if( dltag != null ) {
//			return intag.makeTag( row,value ) + CR +
//					DIV1 + dltag.makeTag( row,value ) + DIV2 + CR ;
//		}
//		else {
//			return intag.makeTag( row,value ) + CR ;
//		}
	}

	/**
	 * 初期値が選択済みの 選択肢(オプション)をTagBuffer に反映します。
	 * このオプションは、引数の値を初期値とするオプションタグ作成し、TagBuffer
	 * に値を設定して返します。
	 *
	 * 第３引数は、Selection オブジェクトのキャッシュ機能を使用するかどうか指定します。
	 * true で、使用する事を前提に、チェックを行います。
	 * DBMENU など、他のメソッドでは、ラベル(短)の使用有無として使用しているため、異なります。
	 *
	 * ここでは、AAA:BBB:CCC:DDD という値を、$1,$2,$3,$4 に割り当てなおして、
	 * QUERYを実行します。また、$1 は、本来の値として、メニューの初期値設定等に
	 * 使用します。上記の例では、AAA が値で、それ以降は、引数になります。
	 * さらに、元の文字列"AAA:BBB:CCC:DDD"は、$0 に割り当てられます。割り当てがない
	 * 変数は、""(ゼロ文字列)として、扱われます。
	 * 又、$Cには自分自身のカラム名を割り当てます。
	 *
	 * @og.rev 6.2.0.0 (2015/02/27) SLABEL 対応
	 * @og.rev 6.2.0.0 (2015/02/27) キー:ラベル形式で表示するかどうかを、指定できるようにします。
	 *
	 * @param	buf    タグ文字列のバッファー
	 * @param	value  選択されている値
	 * @param   useSelCache Selection オブジェクトのキャッシュ機能を使用するかどうか。
	 * @param   useSlbl ラベル(短)をベースとしたオプション表示を行うかどうか。
	 *
	 * @return	オプションタグ
	 */
//	private TagBuffer getOption( final TagBuffer buf,final String value,final boolean useSelCache ) {
	private TagBuffer getOption( final TagBuffer buf,final String value,final boolean useSelCache,final boolean useSlbl ) {

		final StringFormat format = new StringFormat( query, value, name );
		final String newQuery = format.format();
		final String newValue = format.getValue();

		// 6.2.0.0 (2015/02/27) キー:ラベル形式
//		final Selection selection = SelectionFactory.newDBSelection( newQuery, dbid, lang );
		final Selection selection = SelectionFactory.newDBSelection( newQuery, dbid, lang, addKeyLabel );

		if( useSelCache ) {
			if( selection == bkSel ) { return null; }
			bkSel = selection ;
		}

		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
//		return buf.addBody( selection.getOption( newValue, false ) );
		return buf.addBody( selection.getOption( newValue, false, useSlbl ) );	// 6.2.0.0 (2015/02/27) SLABEL 対応

//		buf.setBody( selection.getOption( newValue, false ) );
//
//		return buf;
	}
}
