/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.db.AbstractRenderer;
import org.opengion.hayabusa.db.CellRenderer;
import org.opengion.hayabusa.db.DBColumn;

/**
 * MONEY レンデラーは、カラムのデータを金額表示する場合に使用するクラスです。
 *
 * マイナス時の表示は、id="minus" をキーに CSSファイルで指定しています。
 * 通貨は、標準では、￥ですが、値:記号 という形式で指定すれば、各値ごとに
 * 通貨を指定できます。(ただし、通貨変換は、サポートしていません。)
 * 負数の場合はspanタグclass="minus"を付けて出力します。
 *
 *  カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.group データ表示
 * @og.rev 5.4.3.6 (2012/01/19) コメント修正
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Renderer_MONEY extends AbstractRenderer {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.2.0.0 (2015/02/27)" ;

	private static final CellRenderer[] dbCell = {
							new Renderer_MONEY(),
							new Renderer_MONEY(1),
							new Renderer_MONEY(2)
	};

	private final int	 minFraction;
	private final String noDisplayVal ;		// 5.6.2.3 (2013/03/22)

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.3.0.0 (2003/06/23) 初期値設定追加。
	 * @og.rev 5.6.2.3 (2013/03/22) noDisplayVal 変数初期化
	 *
	 */
	public Renderer_MONEY() {
//		minFraction  = 0;
//		noDisplayVal = null;		// 5.5.1.0 (2012/04/03)
		this( 0,null );				// 6.0.2.4 (2014/10/17)
	}

	/**
	 * コンストラクター。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.3.0.0 (2003/06/23) 初期値設定追加。NumberFormatクラスは、廃止します。
	 * @og.rev 5.6.2.3 (2013/03/22) noDisplayVal 変数追加
	 *
	 * @param	size  	小数点
	 */
	private Renderer_MONEY( final int size ) {
//		minFraction  = size ;
//		noDisplayVal = null;		// 5.5.1.0 (2012/04/03)
		this( size,null );			// 6.0.2.4 (2014/10/17)
	}

	/**
	 * コンストラクター
	 *
	 * @og.rev 6.0.2.4 (2014/10/17) noDisplayVal 対応漏れのため、追加
	 *
	 * @param	size  		小数点
	 * @param	noDispVal  	非表示文字の設定
	 */
	private Renderer_MONEY( final int size , final String noDispVal ) {
		minFraction  = size ;
		noDisplayVal = noDispVal;		// 5.5.1.0 (2012/04/03)
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.1.2.1 (2003/04/10) synchronized を、削除します。
	 * @og.rev 6.0.4.0 (2014/11/28) 表示は、ViewLength属性を元に行う。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellRendererオブジェクト
	 * @og.rtnNotNull
	 */
	public CellRenderer newInstance( final DBColumn clm ) {

		int size = clm.getSizeY();

		// 6.0.4.0 (2014/11/28) 表示は、ViewLength属性があれば、それを使う。
		final String viewLength = clm.getViewLength();
		if( viewLength != null ) {
			final int ch = viewLength.indexOf( ',' ) ;		// DBColumn で、"." を "," に変換済み
			if( ch > 0 ) {
				size = Integer.parseInt( viewLength.substring( ch+1 ) );
			}
			else {
				size = 0;
			}
		}

		final String noDispVal = clm.getNoDisplayVal();	// 6.0.2.4 (2014/10/17)
//		if( size < dbCell.length ) { return dbCell[size]; }
//		return new Renderer_MONEY( size );
		if( size < dbCell.length && noDispVal == null ) { return dbCell[size]; }
		return new Renderer_MONEY( size,noDispVal );
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * 引数の値が、『数字型文字列:通貨』という値を渡すことで、通貨を
	 * 頭につけて通貨ごとに異なる値を表示させることができる。
	 *
	 * @og.rev 3.1.0.0 (2003/03/20) 内部に、DBColumn オブジェクトをキープしないように変更
	 * @og.rev 3.3.0.0 (2003/06/23) NumberFormatクラスは、廃止します。
	 * @og.rev 5.6.2.3 (2013/03/22) noDisplayVal 変数追加
	 * @og.rev 6.0.4.0 (2014/11/28) ロジックの共通化
	 *
	 * @param	value 入力値(『数字型文字列』 または『数字型文字列:通貨』)
	 *
	 * @return	データの表示用文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final String value ) {
		return getValue( value , true );

//		// 5.6.2.3 (2013/03/22) noDisplayVal 変数追加
//		if( noDisplayVal != null && noDisplayVal.equalsIgnoreCase( value ) ) { return "" ; }
//
//		String rtn ;
//		if( value == null || (value.trim()).length() == 0 ) {
//			rtn = "0" ;
//		}
//		else {
//			rtn = value;
//		}
//
//		boolean minus = false ;
//		if( rtn.charAt(0) == '-' ) {
//			minus = true;
//			rtn   = rtn.substring( 1 );
//		}
//
//		String tuuka = "￥";
//		int taniPos = rtn.indexOf( ':' );
//		if( taniPos >= 0 ) {
//			tuuka = rtn.substring( taniPos+1 );
//			rtn   = rtn.substring( 0,taniPos );
//		}
//
//		rtn = tuuka + StringUtil.numberFormat( rtn,minFraction ) + "-";
//
//		if( minus ) {
//			rtn = "<span class=\"minus\">-" + rtn + "</span>";
//		}
//		return rtn;
	}

	/**
	 * データ出力用の文字列を作成します。
	 * ファイル等に出力する形式を想定しますので、HTMLタグを含まない
	 * データを返します。
	 * 基本は、#getValue( String ) をそのまま返します。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) データ出力用のレンデラー
	 * @og.rev 6.2.0.0 (2015/02/27) そのまま数値化できるように、数値部のみを返します。
	 *
	 * @param	value 入力値(『数字型文字列』 または『数字型文字列:通貨』)
	 *
	 * @return  データ出力用の文字列(数字型文字列 のみ)
	 * @og.rtnNotNull
	 * @see		#getValue( String )
	 */
	@Override
	public String getWriteValue( final String value ) {
		return getValue( value , false );
	}

	/**
	 * データ表示用/出力用の文字列を作成します。
	 * 第二引数の isView == true で、データ表示用文字列を、false で
	 * データ出力用の文字列を作成します。
	 * 処理の共通化を行うためのメソッドです。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) ロジックの共通化
	 * @og.rev 6.2.0.0 (2015/02/27) マイナス金額を、"-" ではなく、"▲" にします。
	 *
	 * @param	value 入力値(『数字型文字列』 または『数字型文字列:通貨』)
	 * @param   isView データ表示用かどうか(true:表示用/false:出力用)
	 *
	 * @return  データ表示用/出力用の文字列
	 * @og.rtnNotNull
	 * @see		#getValue( String )
	 */
	private String getValue( final String value , final boolean isView ) {
		// 5.6.2.3 (2013/03/22) noDisplayVal 変数追加
		if( noDisplayVal != null && noDisplayVal.equalsIgnoreCase( value ) ) { return "" ; }

//		String rtn ;
//		if( value == null || (value.trim()).length() == 0 ) {
		if( value == null || value.trim().isEmpty() ) {			// 6.1.0.0 (2014/12/26) refactoring
//			rtn = "0" ;
			return "0" ;
		}
//		else {
//			rtn = value;
//		}

		String rtn = value;

		String tuuka = "￥";
		final int taniPos = rtn.indexOf( ':' );
		if( taniPos >= 0 ) {
			rtn   = rtn.substring( 0,taniPos );
			tuuka = rtn.substring( taniPos+1 );
		}

		if( !isView ) { return rtn; }							// 6.2.0.0 (2015/02/27) マイナス記号のまま

		boolean minus = false ;
//		if( rtn.charAt(0) == '-' ) {
		if( StringUtil.startsChar( rtn , '-' ) ) {				// 6.2.0.0 (2015/02/27) １文字 String.startsWith
			minus = true;
			rtn   = rtn.substring( 1 );							// マイナス記号は取っ払っておきます。
		}

		// 6.2.0.0 (2015/02/27) マイナス金額を、"-" ではなく、"▲" にします。
//		rtn = tuuka + StringUtil.numberFormat( rtn,minFraction ) + "-";
		rtn = tuuka + (minus ? "▲" : "" ) + StringUtil.numberFormat( rtn,minFraction ) + "-";

		if( minus ) {
//		if( isView && minus ) {
//			rtn = "<span class=\"minus\">-" + rtn + "</span>";
			rtn = "<span class=\"minus\">" + rtn + "</span>";
		}
		return rtn;
	}
}
