/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.io;

import java.io.PrintWriter;

import org.odftoolkit.odfdom.OdfFileDom;
import org.odftoolkit.odfdom.doc.OdfSpreadsheetDocument;
import org.odftoolkit.odfdom.doc.office.OdfOfficeAutomaticStyles;
import org.odftoolkit.odfdom.doc.office.OdfOfficeSpreadsheet;
import org.odftoolkit.odfdom.doc.table.OdfTable;
import org.odftoolkit.odfdom.doc.table.OdfTableCell;
import org.odftoolkit.odfdom.doc.table.OdfTableRow;
import org.odftoolkit.odfdom.doc.text.OdfTextParagraph;
import org.odftoolkit.odfdom.dom.attribute.office.OfficeValueTypeAttribute;
import org.opengion.fukurou.model.NativeType;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;
import org.w3c.dom.Node;

/**
 * Calcファイルの書き出しクラスです。
 *
 * DefaultTableWriter を継承していますので，ラベル，名前，データの出力部のみ
 * オーバーライドして，OpenOfficeのCalcファイルの出力機能を実現しています。
 *
 * @og.group ファイル出力
 *
 * @version  5.0
 * @author	 Hiroki Nakamura
 * @since    JDK6.0,
 */
public class TableWriter_Calc extends TableWriter_Default {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.2.4.2 (2015/05/29)" ;

	protected OdfSpreadsheetDocument	wb					;
	protected OdfTable					sheet				;
	protected OdfFileDom				contentDom			;
	protected OdfOfficeSpreadsheet		officeSpreadsheet	;
	protected OdfOfficeAutomaticStyles	contentAutoStyles	;

	protected boolean				useNumber			= true;

	private String					sheetName			= "Sheet1";
//	private String					refSheetName		= null;
	private String					filename			;
//	private String					refFilename			= null;

	/**
	 * DBTableModel から 各形式のデータを作成して,PrintWriter に書き出します。
	 * このメソッドは、Calc 書き出し時に使用します。
	 *
	 * @see #isExcel()
	 */
	@Override
	public void writeDBTable() {
		if( !createDBColumn() ) { return; }

		useNumber = isUseNumber();

		if( filename == null ) {
			final String errMsg = "ファイルが指定されていません。";
			throw new HybsSystemException( errMsg );
		}

		/* 以下は未実装 ***********************************************/
		if( isAppend() ) {
			final String errMsg = "Calcの場合はAppend利用できません。";
			throw new HybsSystemException( errMsg );
		}

	 	// 6.0.1.2 (2014/08/08) 元々、EXCEL以外には実装していない為、削除。
//		if( ( refFilename != null && refFilename.length() > 0 ) || ( refSheetName != null && refSheetName.length() >= 0 ) ) {
//		final String errMsg = "Calcの場合は,refFilename,refSheetName利用できません。";
//			throw new HybsSystemException( errMsg );
//		}

	//	if( fontName != null && fontName.length() > 0 ) {
	//	final String errMsg = "Calcの場合はfontNameは、利用できません。";
	//		throw new HybsSystemException( errMsg );
	//		System.err.println( errMsg );
	//	}

	//	if( fontPoint >= 0 ) {
	//	final String errMsg = "Calcの場合はfontPointは、利用できません。";
	//		throw new HybsSystemException( errMsg );
	//		System.err.println( errMsg );
	//	}
		/* *************************************************************/

		// 新規の場合
		try {
			wb = OdfSpreadsheetDocument.newSpreadsheetDocument();
			// コンテンツのルートとDOMを取得
			contentDom = wb.getContentDom();
			officeSpreadsheet = wb.getContentRoot();
			contentAutoStyles = contentDom.getOrCreateAutomaticStyles();

			// デフォルトで用意されているノードを削除
			Node childNode = officeSpreadsheet.getFirstChild();
			while( childNode != null ) {
				officeSpreadsheet.removeChild( childNode );
				childNode = officeSpreadsheet.getFirstChild();
			}
		}
		catch( Exception ex ) {
			final String errMsg = "Calcの文書宣言ができません。";
			throw new HybsSystemException( errMsg, ex );
		}

		//デフォルトで用意されているStyles調整
		resetAutoStylesAndMasterStyles();

		sheet = new OdfTable( contentDom );
		sheet.setTableNameAttribute( sheetName );

		super.writeDBTable( null );

		officeSpreadsheet.appendChild( sheet );

		try {
			wb.save( filename );
			wb.close();
		}
		catch( Exception ex ) {
			final String errMsg = "Calcの文書saveができません。";
			throw new HybsSystemException( errMsg, ex );
		}
		finally {
			if( null != sheet )	{ sheet = null; }
			if( null != wb )	{ wb    = null; }
		}
	}

//	/**
//	 * DBTableModel から データを作成して,PrintWriter に書き出します。
//	 *
//	 * @og.rev 6.0.1.2 (2014/08/08) 親クラスで同様の処理は定義済みのため、削除。
//	 *
//	 * @param	writer PrintWriterオブジェクト
//	 */
//	@Override
//	public void writeDBTable( final PrintWriter writer )  {
//	final String errMsg = "このクラスでは実装されていません。";
//		throw new UnsupportedOperationException( errMsg );
//	}

	/**
	 * PrintWriter に DBTableModelのラベル情報を書き込みます。
	 * 第一カラム目は、ラベル情報を示す "#Label" を書き込みます。
	 * この行は、出力形式に無関係に、TableWriter.TAB_SEPARATOR で区切られます。
	 *
	 * @og.rev 6.0.1.2 (2014/08/08) カラム飛ばしできる機能を追加
	 * @og.rev 6.2.4.2 (2015/05/29) StringUtil#tagCut(String) をラベルに適用します。
	 *
	 * @param	table  DBTableModelオブジェクト
	 * @param	writer PrintWriterオブジェクト
	 */
	@Override
	protected void writeLabel( final DBTableModel table, final PrintWriter writer ) {
		final OdfTableRow row = new OdfTableRow( contentDom );

		if( useNumber ) {
			row.appendCell( createTextCell( contentDom, "#Label", false, false ) );
		}

		for( int i = 0; i < numberOfColumns; i++ ) {
			final int clm = clmNo[i];
			if( clm < 0 ) {			// 6.0.1.2 (2014/08/08) カラム飛ばし
				row.appendCell( createTextCell( contentDom, "", false, false ) );
				continue;
			}

//			String val = dbColumn[clm].getLabel();
			String val = StringUtil.tagCut( dbColumn[clm].getLabel() );		// 6.2.4.2 (2015/05/29)
			if( i == 0 && !useNumber ) {
				val = "#" + val;
			}
			row.appendCell( createTextCell( contentDom, val, false, false ) );
		}
		row.setStyleName( "ro1" );
		sheet.appendRow( row );
	}

	/**
	 * PrintWriter に DBTableModelの項目名情報を書き込みます。
	 * 第一カラム目は、項目名情報を示す "#Name" を書き込みます。
	 * この行は、出力形式に無関係に、TableWriter.TAB_SEPARATOR で区切られます。
	 *
	 * @og.rev 6.0.1.2 (2014/08/08) カラム飛ばしできる機能を追加
	 *
	 * @param	table  DBTableModelオブジェクト
	 * @param	writer PrintWriterオブジェクト
	 */
	@Override
	protected void writeName( final DBTableModel table, final PrintWriter writer ) {
		final OdfTableRow row = new OdfTableRow( contentDom );

		if( useNumber ) {
			row.appendCell( createTextCell( contentDom, "#Name", false, false ) );
		}

		for( int i = 0; i < numberOfColumns; i++ ) {
			final int clm = clmNo[i];
			if( clm < 0 ) {			// 6.0.1.2 (2014/08/08) カラム飛ばし
				row.appendCell( createTextCell( contentDom, "", false, false ) );
				continue;
			}

			String val = table.getColumnName( clm );
			if( i == 0 && !useNumber ) {
				val = "#" + val;
			}
			row.appendCell( createTextCell( contentDom, val, false, false ) );
		}
		row.setStyleName( "ro1" );
		sheet.appendRow( row );
	}

	/**
	 * PrintWriter に DBTableModelのサイズ情報を書き込みます。
	 * 第一カラム目は、サイズ情報を示す "#Size" を書き込みます。
	 * この行は、出力形式に無関係に、TableWriter.TAB_SEPARATOR で区切られます。
	 *
	 * @og.rev 6.0.1.2 (2014/08/08) カラム飛ばしできる機能を追加
	 *
	 * @param	table  DBTableModelオブジェクト
	 * @param	writer PrintWriterオブジェクト
	 */
	@Override
	protected void writeSize( final DBTableModel table, final PrintWriter writer ) {
		final OdfTableRow row = new OdfTableRow( contentDom );

		if( useNumber ) {
			row.appendCell( createTextCell( contentDom, "#Size", false, false ) );
		}

		for( int i = 0; i < numberOfColumns; i++ ) {
			final int clm = clmNo[i];
			if( clm < 0 ) {			// 6.0.1.2 (2014/08/08) カラム飛ばし
				row.appendCell( createTextCell( contentDom, "", false, false ) );
				continue;
			}

			String val = String.valueOf( dbColumn[clm].getTotalSize() );
			if( i == 0 && !useNumber ) {
				val = "#" + val;
			}
			row.appendCell( createTextCell( contentDom, val, true, false ) );
		}
		row.setStyleName( "ro1" );
		sheet.appendRow( row );
	}

	/**
	 * PrintWriter に DBTableModelのクラス名情報を書き込みます。
	 * 第一カラム目は、サイズ情報を示す "#Class" を書き込みます。
	 * この行は、出力形式に無関係に、TableWriter.TAB_SEPARATOR で区切られます。
	 *
	 * @og.rev 6.0.1.2 (2014/08/08) カラム飛ばしできる機能を追加
	 *
	 * @param	table  DBTableModelオブジェクト
	 * @param	writer PrintWriterオブジェクト
	 */
	@Override
	protected void writeClass( final DBTableModel table, final PrintWriter writer ) {
		final OdfTableRow row = new OdfTableRow( contentDom );

		if( useNumber ) {
			row.appendCell( createTextCell( contentDom, "#Class", false, false ) );
		}

		for( int i = 0; i < numberOfColumns; i++ ) {
			final int clm = clmNo[i];
			if( clm < 0 ) {			// 6.0.1.2 (2014/08/08) カラム飛ばし
				row.appendCell( createTextCell( contentDom, "", false, false ) );
				continue;
			}

			String val = dbColumn[clm].getClassName();
			if( i == 0 && !useNumber ) {
				val = "#" + val;
			}
			row.appendCell( createTextCell( contentDom, val, false, false ) );
		}
		row.setStyleName( "ro1" );
		sheet.appendRow( row );
	}

	/**
	 * PrintWriter に セパレーターを書き込みます。
	 * 第一カラム目は、サイズ情報を示す "#----" を書き込みます。
	 * この行は、出力形式に無関係に、TableWriter.TAB_SEPARATOR で区切られます。
	 *
	 * @param	table  DBTableModelオブジェクト
	 * @param	writer PrintWriterオブジェクト
	 */
	@Override
	protected void writeSeparator( final DBTableModel table, final PrintWriter writer ) {
		final String sep = "----";

		final OdfTableRow row = new OdfTableRow( contentDom );
		row.appendCell( createTextCell( contentDom, "#----", false, false ) );
		for( int i = 0; i < numberOfColumns; i++ ) {
			if( i == 0 && !useNumber ) {
				continue;
			}
			row.appendCell( createTextCell( contentDom, sep, false, false ) );
		}
		sheet.appendRow( row );
	}

	/**
	 * PrintWriter に DBTableModelのテーブル情報を書き込みます。
	 * このクラスでは，データを ダブルコーテーション(")で囲みます。
	 * PrintWriter に DBTableModelのテーブル情報を書き込みます。
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) useRenderer 対応
	 * @og.rev 6.0.1.2 (2014/08/08) カラム飛ばしできる機能を追加
	 * @og.rev 6.0.4.0 (2014/11/28) データ出力用のレンデラー
	 *
	 * @param	table  DBTableModelオブジェクト
	 * @param	writer PrintWriterオブジェクト
	 */
	@Override
	protected void writeData( final DBTableModel table,final PrintWriter writer ) {
		final int numberOfRows =	table.getRowCount();

		boolean[] nvar = new boolean[numberOfColumns];
		boolean[] cellType = new boolean[numberOfColumns];
		for( int i=0; i<numberOfColumns; i++ ) {
			final int clm = clmNo[i];
			if( clm < 0 ) { continue; }			// 6.0.1.2 (2014/08/08) カラム飛ばし

			final NativeType nativeType = dbColumn[clm].getNativeType();
			switch( nativeType ) {
				case INT    :
				case LONG   :
				case DOUBLE :
					cellType[i] = true ;
					break;
				case STRING :
				case CALENDAR :
				default :
					cellType[i] = false ;
					break;
			}
			nvar[i] = "NVAR".equals( dbColumn[clm].getDbType() );
		}
		final boolean useRenderer = isUseRenderer();	// 5.2.1.0 (2010/10/01)

		for( int r = 0; r < numberOfRows; r++ ) {
			final OdfTableRow row = new OdfTableRow( contentDom );

			if( useNumber ) {
				row.appendCell( createTextCell( contentDom, String.valueOf( r + 1 ), true, true ) );
			}

			for( int i = 0; i < numberOfColumns; i++ ) {
				final int clm = clmNo[i];
				if( clm < 0 ) {			// 6.0.1.2 (2014/08/08) カラム飛ばし
					row.appendCell( createTextCell( contentDom, "", false, false ) );
					continue;
				}

				String val = table.getValue( r, clm );
				if( nvar[i] ) {
					val = StringUtil.getReplaceEscape( val );
				}
				// 5.2.1.0 (2010/10/01) useRenderer 対応
				else if( useRenderer ) {
					// 6.0.4.0 (2014/11/28) データ出力用のレンデラー
//					val = StringUtil.spanCut( dbColumn[clm].getRendererValue( val ) );
					val = dbColumn[clm].getWriteValue( val );
				}
				row.appendCell( createTextCell( contentDom, val, cellType[i], false ) );
			}
			row.setStyleName( "ro1" );
			sheet.appendRow( row );
		}
	}

	/**
	 * テキストコンテンツ用のセルを生成する。
	 *
	 * @param	contentDom	OdfFileDomオブジェクト
	 * @param	content		コンテンツ
	 * @param	isCellTypeNumber	[true:数字型/false:文字型]
	 * @param	isNumberList		[true:数字リスト=999/false:通常]
	 *
	 * @return	OpenOfficeのセルテーブルオブジェクト
	 */
	protected OdfTableCell createTextCell(	final OdfFileDom contentDom,
											final String content,
											final Boolean isCellTypeNumber,
											final Boolean isNumberList ) {
		final OdfTableCell cell = new OdfTableCell( contentDom );
		if( isCellTypeNumber ) {
			cell.setOfficeValueAttribute( Double.parseDouble( content ) );
			cell.setOfficeValueTypeAttribute( OfficeValueTypeAttribute.Value.FLOAT.toString() );
		}
		else {
			cell.setOfficeValueTypeAttribute( OfficeValueTypeAttribute.Value.STRING.toString() );
		}
		final OdfTextParagraph paragraph = new OdfTextParagraph( contentDom, null, content );
		cell.appendChild( paragraph );
		return cell;
	}

	/**
	 * デフォルトで用意されているStylesを調整します。
	 * ※ここでは何もしません。
	 *
	 */
	protected void resetAutoStylesAndMasterStyles(){
		// Document empty method 対策
	}

	/**
	 * このクラスが、Calc対応機能(=Excel対応機能)を持っているかどうかを返します。
	 *
	 * Calc対応機能とは、シート名のセット、雛型参照ファイル名のセット、
	 * 書き込み元ファイルのFileオブジェクト取得などの、特殊機能です。
	 * 本来は、インターフェースを分けるべきと考えますが、taglib クラス等の
	 * 関係があり、問い合わせによる条件分岐で対応します。
	 *
	 * @return	Calc対応機能(=Excel対応機能)を持っているかどうか(true 固定)
	 */
	@Override
	public boolean isExcel() {
		return true;
	}

	/**
	 * 出力先ディレクトリとファイル名をセットします。
	 * これは、Calc追加機能として実装されています。
	 * EXCELで使用することを主にしているため、ここでは、すぐに合成しておきます。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) ディレクトリとファイルを分けて管理します。
	 *
	 * @param   directory 出力先ディレクトリ名
	 * @param   filename Calc雛型参考ファイル名
	 */
	@Override
//	public void setFilename( final String filename ) {
	public void setFilename( final String directory , final String filename ) {
//		this.filename  = filename;
		this.filename  = StringUtil.urlAppend( directory,filename );
	}

	/**
	 * DBTableModelのデータとして読み込むときのシート名を設定します。
	 * 初期値は、"Sheet1" です。
	 * これは、Calc追加機能として実装されています。
	 *
	 * @param   sheetName シート名
	 */
	@Override
	public void setSheetName( final String sheetName ) {
		if( sheetName != null ) {
			this.sheetName = sheetName;
		}
	}

//	/**
//	 * Calc雛型参考ファイル名をセットします。(DIR + Filename)
//	 * これは、Calc追加機能として実装されています。
//	 *
//	 * @og.rev 6.0.1.2 (2014/08/08) 元々、EXCEL以外には実装していない為、削除。
//	 * @og.rev 6.0.2.0 (2014/09/19) 親クラスで同様の処理は定義済みのため、削除。
//	 *
//	 * @param   filename Calc雛型参考ファイル名
//	 */
//	@Override
//	public void setRefFilename( final String filename ) {
////		refFilename = filename;
//	final String errMsg = "Calcの場合はrefFilenameは、利用できません。refFilename=[" + filename + "]";
//		System.err.println( errMsg );
//	}

//	/**
//	 * Calc雛型参考ファイルのシート名を設定します。
//	 * これは、Calc追加機能として実装されています。
//	 *
//	 * Calcファイルを書き出す時に、雛型として参照するシート名を指定します。
//	 * これにより、複数の形式の異なるデータを順次書き出したり(appendモードを併用)する
//	 * ことや、シートを指定して新規にCalcを作成する場合にフォームを設定する事が可能になります。
//	 * 初期値は、null(第一シート) です。
//	 *
//	 * @og.rev 6.0.1.2 (2014/08/08) 元々、EXCEL以外には実装していない為、削除。
//	 * @og.rev 6.0.2.0 (2014/09/19) 親クラスで同様の処理は定義済みのため、削除。
//	 *
//	 * @param   sheetName シート名
//	 */
//	@Override
//	public void setRefSheetName( final String sheetName ) {
////		if( sheetName != null ) {
////			refSheetName = sheetName;
////		}
//	final String errMsg = "Calcの場合はrefSheetNameは、利用できません。refSheetName=[" + sheetName + "]";
//		System.err.println( errMsg );
//	}

//	/**
//	 * Calc出力時のデフォルトフォント名を設定します。
//	 * Calcの場合はfontNameは、利用できません。
//	 *
//	 * Calcファイルを書き出す時に、デフォルトフォント名を指定します。
//	 * フォント名は、Calcのフォント名をそのまま使用してください。
//	 * に設定されます。
//	 * 初期値は、システムリソース の TABLE_WRITER_DEFAULT_FONT_NAME です。
//	 *
//	 * @og.rev 	5.5.2.6 (2012/05/25) findbugs対応
//	 * @og.rev 6.0.2.0 (2014/09/19) 親クラスで同様の処理は定義済みのため、削除。
//	 *
//	 * @param   fontName デフォルトフォント名
//	 */
//	@Override
//	public void setFontName( final String fontName ) {
//	final String errMsg = "Calcの場合はfontNameは、利用できません。fontName=[" + fontName + "]";
//		System.err.println( errMsg );
//	}

//	/**
//	 * Calc出力時のデフォルトフォントポイント数を設定します。
//	 * Calcの場合はfontPointは、利用できません。
//	 *
//	 * Calcファイルを書き出す時に、デフォルトポイント数を指定します。
//	 * に設定されます。
//	 * 初期値は、システムリソース の TABLE_WRITER_DEFAULT_FONT_POINTS です。
//	 *
//	 * @og.rev 	5.5.2.6 (2012/05/25) findbugs対応
//	 * @og.rev 6.0.2.0 (2014/09/19) 親クラスで同様の処理は定義済みのため、削除。
//	 *
//	 * @param   point デフォルトフォントポイント数
//	 */
//	@Override
//	public void setFontPoint( final short point ) {
//	final String errMsg = "Calcの場合はfontPointは、利用できません。fontPoint=[" + point + "]";
//		System.err.println( errMsg );
//	}
}
