/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.StringUtil ;
import org.opengion.fukurou.util.ToString;
import org.opengion.fukurou.model.POIUtil;

import static org.opengion.fukurou.util.StringUtil.nval ;

import java.util.List;
import java.util.ArrayList;
import java.io.File;
import java.io.IOException;

/**
 * ExcelFileTag は、各種パラメータを指定して、Excelファイルの操作をおこなうタグです。
 *
 * 主に、ExcelModel、POIUtil 、JacobUtil の各機能をJSP上で使用できるようにしました。
 * 入力の file1 を加工して、file2 を出力するというのが、基本パターンになります。
 * file2 を、指定しないか、または、file1 と、file2 を同じ値に指定した場合は、
 * file1 に上書きされます。
 *
 * ※ (readPrinter,saveAs,toPrint)属性は、JACOB(Java COM Bridge)の使用が前提です。
 *    これは、JavaからのCOMオートメーションコンポーネント呼び出しを可能とする 
 *    JAVA-COMブリッジです。COMライブラリのネイティブな呼び出しにJNIを使います。
 *    JACOBは、32bitおよび64bitのJVMをサポートするX86およびx64環境で動作します。 
 *    http://sourceforge.jp/projects/sfnet_jacob-project/  (日本語プロジェクト)
 *    http://sourceforge.net/projects/jacob-project/       (本家)
 *
 *    設定：
 *       jacob-1.18-M2.zip をダウンロードし、
 *         ①jacob-1.18-M2-x64.dll または、jacob-1.18-M2-x86.dll を、
 *           Windowsのシステムディレクトリにコピーします。 （例：C:\Windows\System32）
 *         ②jacob.jar を、クラスパスに登録します。
 *           ここでは、名称を、jacob-1.18-M2.jar に変更し、jre\lib\ext にコピーしています。
 *
 * @og.formSample
 * ●形式：&lt;og:excelFile action="…" fileURL="…" &gt;･･･&lt;/og:file&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:excelFile
 *       fileURL            【TAG】操作するファイルのディレクトリを指定します (初期値:FILE_URL[=filetemp/])
 *       file1            ○【TAG】入力ファイル名を指定します(必須)
 *       file2              【TAG】出力ファイル名を指定します(指定がない場合で、file2 が必要な場合は、file1 が使われます)
 *       sheetName          【TAG】EXCELファイルを読み込むときのシート名を設定します(初期値:指定なし)
 *       sheetNos           【TAG】EXCELファイルを読み込むときのシート番号を複数設定できます(初期値:0)
 *       sheetConstKeys     【TAG】EXCELファイルを読み込むときの固定値となるカラム名(CSV形式)
 *       sheetConstAdrs     【TAG】EXCELファイルを読み込むときの固定値となるアドレス(行-列,行-列,・・・)
 *       useCellStyle       【TAG】EXCEL出力時に、データを書き込んだ範囲に罫線を入れるかどうかを指定します(初期値:false)
 *       useAutoCellSize    【TAG】EXCEL出力時に、セルの幅をデータの幅に自動的に合わせるかどうかを指定します(初期値:false)
 *       useActiveWorkbook  【TAG】EXCEL出力時に、セルの有効範囲を設定するかどうかを指定します(初期値:false)
 *       hyperLinkColumn    【TAG】EXCEL出力時に、Hyperlinkを作成するキーとなるカラム名と値となるカラム名を指定します
 *       addTitleSheet      【TAG】EXCEL出力時に、存在するSheet名一覧を作成する場合に、そのSheet名を指定します。
 *       addImageFile       【TAG】画像ファイルを挿入します(画像ファイル名 シート番号 行 列)
 *       valueType          【TAG】sheetConstXXX,readXXXX のパラメータに登録する方法を指定します(CSV,List,Map)
 *       readText           【TAG】ファイルを読み込んで、内容を 引数の変数にセットします。
 *       readSheet          【TAG】ファイルを読み込んで、シート一覧を 引数の変数にセットします。
 *       readName           【TAG】ファイルを読み込んで、名前一覧を 引数の変数にセットします。
 *       readStyle          【TAG】ファイルを読み込んで、スタイル名を 引数の変数にセットします。
 *   ※  readPrinter        【TAG】サーバー環境から使用可能なプリンタ名の一覧を引数の変数にセットします。
 *   ※  saveAs             【TAG】file1 を指定のファイルの拡張子に合わせた変換を行って保存します(xls,xlsx,pdf)。
 *   ※  toPrint            【TAG】file1 を指定のプリンタに印刷します。
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null) 5.7.7.2 (2014/06/20)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null) 5.7.7.2 (2014/06/20)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:true) 5.7.7.2 (2014/06/20)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:true) 5.7.7.2 (2014/06/20)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *    ・EXCEL を読み取って、シート名の一覧の目次を、先頭シートに追加します。
 *        &lt;og:excelFile  file1="input.xls" addTitleSheet="Title" /&gt;
 *
 * @og.group その他部品
 * @og.rev 6.2.6.0 (2015/06/19) 新規作成
 *
 * @version  6.2
 * @author	 Kazuhiko Hasegawa
 * @since    JDK8.0,
 */
public class ExcelFileTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.2.6.0 (2015/06/19)" ;

	private static final long serialVersionUID = 626020150619L ;

	private String	fileURL 	= HybsSystem.sys( "FILE_URL" );
	private String	file1		= "";
	private String	file2		;

	private String	sheetName			;			// EXCELファイルを読み込むときのシート名を設定します(初期値:指定なし)
	private String	sheetNos			;			// EXCELファイルを読み込むときのシート番号を複数設定できます(初期値:0)
	private String	sheetConstKeys		;			// EXCELファイルを読み込むときの固定値となるカラム名(CSV形式)
	private String	sheetConstAdrs		;			// EXCELファイルを読み込むときの固定値となるアドレス(行-列,行-列,・・・)
	private boolean	useCellStyle		;			// EXCEL出力時に、データを書き込んだ範囲に罫線を入れるかどうかを指定します(初期値:false)
	private boolean	useAutoCellSize		;			// EXCEL出力時に、セルの幅をデータの幅に自動的に合わせるかどうかを指定します(初期値:false)
	private boolean	useActiveWorkbook	;			// EXCEL出力時に、セルの有効範囲を設定するかどうかを指定します(初期値:false)
	private String	hyperLinkColumn		;			// EXCEL出力時に、Hyperlinkを作成するキーとなるカラム名と値となるカラム名を指定します
	private String	addTitleSheet		;			// EXCEL出力時に、存在するSheet名一覧を作成する場合に、そのSheet名を指定します。
	private String	addImageFile		;			// 画像ファイルを挿入します(画像ファイル名 シート番号 行 列)
	private String	valueType			= "CSV";	// sheetConstXXX,readXXXX のパラメータに登録する方法を指定します(CSV,List,Map)
	private String	readText			;			// ファイルを読み込んで、内容を 引数の変数にセットします。
	private String	readSheet			;			// ファイルを読み込んで、シート一覧を 引数の変数にセットします。
	private String	readName			;			// ファイルを読み込んで、名前一覧を 引数の変数にセットします。
	private String	readStyle			;			// ファイルを読み込んで、スタイル名を 引数の変数にセットします。
	private String	readPrinter			;			// サーバー環境から使用可能なプリンタ名の一覧を引数の変数にセットします。
	private String	saveAs				;			// file1 を指定のファイルの拡張子に合わせた変換を行って保存します(xls,xlsx,pdf)。
	private String	toPrint				;			// file1 を指定のプリンタに印刷します。

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 6.2.6.0 (2015/06/19) 新規作成
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)




		return EVAL_PAGE ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 6.2.6.0 (2015/06/19) 新規作成
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		fileURL 			= HybsSystem.sys( "FILE_URL" );
		file1				= "";
		file2				= null;
		sheetName			= null;		// EXCELファイルを読み込むときのシート名を設定します(初期値:指定なし)
		sheetNos			= null;		// EXCELファイルを読み込むときのシート番号を複数設定できます(初期値:0)
		sheetConstKeys		= null;		// EXCELファイルを読み込むときの固定値となるカラム名(CSV形式)
		sheetConstAdrs		= null;		// EXCELファイルを読み込むときの固定値となるアドレス(行-列,行-列,・・・)
		useCellStyle		= false;	// EXCEL出力時に、データを書き込んだ範囲に罫線を入れるかどうかを指定します(初期値:false)
		useAutoCellSize		= false;	// EXCEL出力時に、セルの幅をデータの幅に自動的に合わせるかどうかを指定します(初期値:false)
		useActiveWorkbook	= false;	// EXCEL出力時に、セルの有効範囲を設定するかどうかを指定します(初期値:false)
		hyperLinkColumn		= null;		// EXCEL出力時に、Hyperlinkを作成するキーとなるカラム名と値となるカラム名を指定します
		addTitleSheet		= null;		// EXCEL出力時に、存在するSheet名一覧を作成する場合に、そのSheet名を指定します。
		addImageFile		= null;		// 画像ファイルを挿入します(画像ファイル名 シート番号 行 列)
		valueType			= "CSV";	// sheetConstXXX,readXXXX のパラメータに登録する方法を指定します(CSV,List,Map)
		readText			= null;		// ファイルを読み込んで、内容を 引数の変数にセットします。
		readSheet			= null;		// ファイルを読み込んで、シート一覧を 引数の変数にセットします。
		readName			= null;		// ファイルを読み込んで、名前一覧を 引数の変数にセットします。
		readStyle			= null;		// ファイルを読み込んで、スタイル名を 引数の変数にセットします。
		readPrinter			= null;		// サーバー環境から使用可能なプリンタ名の一覧を引数の変数にセットします。
		saveAs				= null;		// file1 を指定のファイルの拡張子に合わせた変換を行って保存します(xls,xlsx,pdf)。
		toPrint				= null;		// file1 を指定のプリンタに印刷します。
	}

	/**
	 * 【TAG】操作するファイルのディレクトリを指定します
	 *		(初期値:FILE_URL[={@og.value org.opengion.hayabusa.common.SystemData#FILE_URL}])。
	 *
	 * @og.tag
	 * この属性で指定されるディレクトリのファイルを操作します。
	 * 指定方法は、通常の fileURL 属性と同様に、先頭が、'/' (UNIX) または、２文字目が、
	 * ":" (Windows)の場合は、指定のURLそのままのディレクトリに、そうでない場合は、
	 * (初期値:システム定数のFILE_URL[={@og.value org.opengion.hayabusa.common.SystemData#FILE_URL}])。
	 *
	 * @og.rev 6.2.6.0 (2015/06/19) 新規作成
	 *
	 * @param	url ファイルURL
	 * @see		org.opengion.hayabusa.common.SystemData#FILE_URL
	 */
	public void setFileURL( final String url ) {
		String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			final char ch = furl.charAt( furl.length()-1 );
			if( ch != '/' && ch != '\\' ) { furl = furl + "/"; }
			fileURL = StringUtil.urlAppend( fileURL,furl );
		}
	}

	/**
	 * 【TAG】基準となるファイル名を指定します(コマンドの左辺のファイル名です)。
	 *
	 * @og.tag
	 * コマンドの左辺のファイル名です。
	 *
	 * @og.rev 6.2.6.0 (2015/06/19) 新規作成
	 *
	 * @param	fname ファイル名１
	 */
	public void setFile1( final String fname ) {
		file1 = nval( getRequestParameter( fname ),file1 );
	}

	/**
	 * 【TAG】処理結果となるファイル名を指定します(コマンドの右辺のファイル名です)。
	 *
	 * @og.tag
	 * コマンドの右辺のファイル名です。
	 * ただし、action="list" の時は、file2 を、フィルターとして使用します。
	 *
	 * @og.rev 6.2.6.0 (2015/06/19) 新規作成
	 *
	 * @param	fname ファイル名２
	 */
	public void setFile2( final String fname ) {
		file2 = nval( getRequestParameter( fname ),file2 );
	}

	/**
	 * 【TAG】EXCELファイルを読み込むときのシート名を設定します(初期値:指定なし)。
	 *
	 * @og.tag
	 * EXCELファイルを読み込む時に、シート名を指定します。これにより、複数の形式の
	 * 異なるデータを順次読み込むことや、シートを指定して読み取ることが可能になります。
	 * sheetNos と sheetName が同時に指定された場合は、sheetNos が優先されます。エラーにはならないのでご注意ください。
	 * 初期値は、指定なしです。
	 *
	 * @og.rev 6.2.6.0 (2015/06/19) 新規作成
	 *
	 * @param   sheet EXCELファイルのシート名
	 * @see		#setSheetNos( String ) 
	 */
	public void setSheetName( final String sheet ) {
		sheetName = nval( getRequestParameter( sheet ),sheetName );
	}

	/**
	 * 【TAG】EXCELファイルを読み込むときのシート番号を指定します(初期値:0)。
	 *
	 * @og.tag
	 * EXCEL読み込み時に複数シートをマージして取り込みます。
	 * シート番号は、0 から始まる数字で表します。
	 * ヘッダーは、最初のシートのカラム位置に合わせます。（ヘッダータイトルの自動認識はありません。）
	 * よって、指定するシートは、すべて同一レイアウトでないと取り込み時にカラムのずれが発生します。
	 * 
	 * シート番号の指定は、CSV形式で、複数指定できます。また、N-M の様にハイフンで繋げることで、
	 * N 番から、M 番のシート範囲を一括指定可能です。また、"*" による、全シート指定が可能です。
	 * これらの組み合わせも可能です。（ 0,1,3,5-8,10-* ）
	 * ただし、"*" に関しては例外的に、一文字だけで、すべてのシートを表すか、N-* を最後に指定するかの
	 * どちらかです。途中には、"*" は、現れません。
	 * シート番号は、重複(1,1,2,2)、逆転(3,2,1) での指定が可能です。これは、その指定順で、読み込まれます。
	 * sheetNos と sheetName が同時に指定された場合は、sheetNos が優先されます。エラーにはならないのでご注意ください。
	 * 
	 * 初期値は、0（第一シート） です。
	 *
	 * @og.rev 6.2.6.0 (2015/06/19) 新規作成
	 *
	 * @param   sheet EXCELファイルのシート番号（0から始まる）
	 * @see		#setSheetName( String ) 
	 */
	public void setSheetNos( final String sheet ) {
		sheetNos = nval( getRequestParameter( sheet ),sheetNos );
		if( sheetNos != null && sheetNos.length() > 0 ) {
			boolean errFlag = false;
			for( int i=0; i<sheetNos.length(); i++ ) {
				final char ch = sheetNos.charAt(i);
				if( ch == '-' || ch == ',' ) { continue; }
				if( ch == '*' && ( i==0 || i==sheetNos.length()-1 ) ) { continue; }
				if( ch < '0' || ch > '9' ) { errFlag = true; break; }
			}
			if( errFlag ) {
				final String errMsg = "sheetNos の指定を見直してください。sheetNos=[" + sheetNos + "]";
				throw new HybsSystemException( errMsg );
			}
		}
	}

	/**
	 * 【TAG】EXCELファイルを読み込むときのシート単位の固定値を設定するためのカラム名を指定します。
	 *
	 * @og.tag
	 * カラム名は、CSV形式で指定します。
	 * これにより、シートの一か所に書かれている情報を、DBTableModel のカラムに固定値として
	 * 設定することができます。
	 * 例として、DB定義書で、テーブル名をシートの全レコードに設定したい場合などに使います。
	 * このメソッドは、isExcel() == true の場合のみ利用されます。
	 *
	 * @og.rev 6.2.6.0 (2015/06/19) 新規作成
	 *
	 * @param   constKeys 固定値となるカラム名 (CSV形式)
	 * @see		#setSheetConstAdrs( String ) 
	 */
	public void setSheetConstKeys( final String constKeys ) {
		sheetConstKeys = nval( getRequestParameter( constKeys ),null );
	}

	/**
	 * 【TAG】EXCELファイルを読み込むときのシート単位の固定値を設定するためのカラム名に対応するアドレスを指定します。
	 *
	 * @og.tag
	 * アドレスは、EXCEL上の行-列をCSV形式で指定します。
	 * 行列は、EXCELオブジェクトに準拠するため、０から始まる整数です。
	 * 0-0 ⇒ A1 , 1-0 ⇒ A2 , 0-1 ⇒ B1 になります。
	 * これにより、シートの一か所に書かれている情報を、DBTableModel のカラムに固定値として
	 * 設定することができます。
	 * 例として、DB定義書で、テーブル名をシートの全レコードに設定したい場合などに使います。
	 * このメソッドは、isExcel() == true の場合のみ利用されます。
	 *
	 * 5.7.6.3 (2014/05/23) より、
	 *   ①EXCEL表記に準拠した、A1,A2,B1 の記述も処理できるように対応します。
	 *     なお、A1,A2,B1 の記述は、必ず、英字1文字＋数字 にしてください。(A～Zまで)
	 *   ②処理中のEXCELシート名をカラムに割り当てるために、"SHEET" という記号に対応します。
	 * 例えば、sheetConstKeys="CLM,LANG,NAME" とし、sheetConstAdrs="0-0,A2,SHEET" とすると、
	 * NAMEカラムには、シート名を読み込むことができます。
	 * これは、内部処理の簡素化のためです。
	 *
	 * ちなみに、EXCELのセルに、シート名を表示させる場合の関数は、下記の様になります。
	 * =RIGHT(CELL("filename",$A$1),LEN(CELL("filename",$A$1))-FIND("]",CELL("filename",$A$1)))
	 *
	 * @og.rev 6.2.6.0 (2015/06/19) 新規作成
	 *
	 * @param   constAdrs 固定値となるアドレス (行-列,行-列,・・・)
	 * @see		#setSheetConstKeys( String ) 
	 */
	public void setSheetConstAdrs( final String constAdrs ) {
		sheetConstAdrs = nval( getRequestParameter( constAdrs ),null );
	}

	/**
	 * 【TAG】EXCEL出力時に、データを書き込んだ範囲に罫線を入れるかどうかを指定します(初期値:false)。
	 *
	 * @og.tag
	 * データを書き込んでEXCELを作成しても、ノーマルのセルに値がセットされている
	 * だけなので、ある程度加工が必要です。
	 * そこで、データのセットされたセルに罫線を入れることで、それなりのデータが
	 * 出力された感じになります。
	 * この設定と、useAutoCellSize="true" で、セルの幅を自動調整すれば、見栄えが良くなります。
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.2.6.0 (2015/06/19) 新規作成
	 *
	 * @param	useStyle 罫線追記 [true:する/false:しない]
	 * @see		#setUseAutoCellSize( String )
	 */
	public void setUseCellStyle( final String useStyle ) {
		useCellStyle = nval( getRequestParameter( useStyle ),useCellStyle );
	}

	/**
	 * 【TAG】EXCEL出力時に、セルの幅をデータの幅に自動的に合わせるかどうかを指定します(初期値:false)。
	 *
	 * @og.tag
	 * データを書き込んでEXCELを作成しても、ノーマルのセルに値がセットされている
	 * だけなので、ある程度加工が必要です。
	 * そこで、データのセットされたセルの幅を自動調整することで、それなりのデータが
	 * 出力された感じになります。
	 * この設定と、useCellStyle="true" で、セルの罫線を自動設定すれば、見栄えが良くなります。
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.2.6.0 (2015/06/19) 新規作成
	 *
	 * @param	useAutoSize 幅の自動的調整 [true:する/false:しない]
	 * @see		#setUseCellStyle( String )
	 */
	public void setUseAutoCellSize( final String useAutoSize ) {
		useAutoCellSize = nval( getRequestParameter( useAutoSize ),useAutoCellSize );
	}

	/**
	 * 【TAG】EXCEL出力時に、セルの有効範囲を設定するかどうかを指定します(初期値:false)。
	 *
	 * @og.tag
	 * セルの有効範囲というのは、EXCELでの 空行、空列の存在しない範囲を指します(初期値:false)。
	 * 通常、空行でも、データとして残っている場合は、EXCELのセルオブジェクトは存在します。
	 * ここで、useActiveWorkbook="true" とすると、空行、空列を削除します。
	 * 
	 * 雛形を使用した場合は、データより多めに設定した計算などは、この処理で
	 * 削除されますので、データサイズにフィットさせることができます。
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.2.6.0 (2015/06/19) 新規作成
	 *
	 * @param	useActWB 有効範囲の設定 [true:する/false:しない]
	 */
	public void setUseActiveWorkbook( final String useActWB ) {
		useActiveWorkbook = nval( getRequestParameter( useActWB ),useActiveWorkbook );
	}

	/**
	 * 【TAG】EXCEL出力時に、Hyperlinkを作成するキーとなる「カラム名:値」形式のカラム名を指定します。
	 *
	 * @og.tag
	 * ここで、作成するハイパーリンクは、EXCELのシートに対するハイパーリンクです。
	 * それ以外のリンク(本当のURLやファイル等)のリンクは(今は)作成できません。
	 * ハイパーリンクを作成するには、①作成するカラム と ②作成する値 が必要です。
	 * このメソッドで設定するのは、「①:②」という形式でカラム名を指定します。
	 * ②がなければ、①と同じとします。
	 * ②の値のシートの存在有無は、無視します。ハイパーリンクを作成するシートを作成する前に
	 * ハイパーリンクを作成するケースが存在します。
	 * (例えば、各シートへのリンクを持った一覧を作成してから、明細の各シートを作成する様なケース)
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.2.6.0 (2015/06/19) 新規作成
	 *
	 * @param	hlColumn リンク作成カラム名 (「カラム名:値」形式)
	 */
	public void setHyperLinkColumn( final String hlColumn ) {
		hyperLinkColumn = nval( getRequestParameter( hlColumn ),hyperLinkColumn );
	}

	/**
	 * 【TAG】EXCEL出力時に、Sheet一覧を先頭Sheetに作成する場合のSheet名を指定します。
	 *
	 * @og.tag
	 * これは、Workbook に含まれる Sheet 一覧を作成する場合に、利用可能です。
	 * なお、この機能は、TableWriter_Excel のみに実装します。
	 *
	 * @og.rev 6.2.6.0 (2015/06/19) 新規作成
	 *
	 * @param	sheetName タイトルシート名
	 */
	public void setAddTitleSheet( final String sheetName ) {
		addTitleSheet = nval( getRequestParameter( sheetName ),addTitleSheet );
	}

	/**
	 * 【TAG】画像ファイルを挿入します(画像ファイル名 シート番号 行 列)。
	 *
	 * @og.tag
	 *
	 * @og.rev 6.2.6.0 (2015/06/19) 新規作成
	 *
	 * @param	imgFile (画像ファイル名 シート番号 行 列)
	 */
	public void setAddImageFile( final String imgFile ) {
		addImageFile = nval( getRequestParameter( imgFile ),addImageFile );
	}

	/**
	 * 【TAG】sheetConstXXX,readXXXX のパラメータに登録する方法を指定します(CSV,List,Map)。
	 *
	 * @og.tag
	 *
	 * @og.rev 6.2.6.0 (2015/06/19) 新規作成
	 *
	 * @param	type 保管方法(CSV/List/Map)
	 */
	public void setUseListSet( final String type ) {
		valueType = nval( getRequestParameter( type ),valueType );
	}

	/**
	 * 【TAG】ファイルを読み込んで、内容を 引数の変数にセットします。
	 *
	 * @og.tag
	 *
	 * @og.rev 6.2.6.0 (2015/06/19) 新規作成
	 *
	 * @param	read 書き込む変数名
	 */
	public void setReadText( final String read ) {
		readText = nval( getRequestParameter( read ),readText );
	}

	/**
	 * 【TAG】ファイルを読み込んで、シート一覧を 引数の変数にセットします。
	 *
	 * @og.tag
	 *
	 * @og.rev 6.2.6.0 (2015/06/19) 新規作成
	 *
	 * @param	read 書き込む変数名
	 */
	public void setReadSheet( final String read ) {
		readSheet = nval( getRequestParameter( read ),readSheet );
	}

	/**
	 * 【TAG】ファイルを読み込んで、名前一覧を 引数の変数にセットします。
	 *
	 * @og.tag
	 *
	 * @og.rev 6.2.6.0 (2015/06/19) 新規作成
	 *
	 * @param	read 書き込む変数名
	 */
	public void setReadName( final String read ) {
		readName = nval( getRequestParameter( read ),readName );
	}

	/**
	 * 【TAG】ファイルを読み込んで、スタイル名を 引数の変数にセットします。
	 *
	 * @og.tag
	 *
	 * @og.rev 6.2.6.0 (2015/06/19) 新規作成
	 *
	 * @param	read 書き込む変数名
	 */
	public void setReadStyle( final String read ) {
		readStyle = nval( getRequestParameter( read ),readStyle );
	}

	/**
	 * 【TAG】サーバー環境から使用可能なプリンタ名の一覧を引数の変数にセットします。
	 *
	 * @og.tag
	 * JACOB(Java COM Bridge)の使用が前提です。
	 *
	 * @og.rev 6.2.6.0 (2015/06/19) 新規作成
	 *
	 * @param	read 書き込む変数名
	 */
	public void setReadPrinter( final String read ) {
		readPrinter = nval( getRequestParameter( read ),readPrinter );
	}

	/**
	 * 【TAG】file1 を指定のファイルの拡張子に合わせた変換を行って保存します(xls,xlsx,pdf)。
	 *
	 * @og.tag
	 * JACOB(Java COM Bridge)の使用が前提です。
	 *
	 * @og.rev 6.2.6.0 (2015/06/19) 新規作成
	 *
	 * @param	name セーブするファイル名
	 */
	public void setSaveAs( final String name ) {
		saveAs = nval( getRequestParameter( name ),saveAs );
	}

	/**
	 * 【TAG】file1 を指定のプリンタに印刷します。
	 * JACOB(Java COM Bridge)の使用が前提です。
	 *
	 * @og.tag
	 *
	 * @og.rev 6.2.6.0 (2015/06/19) 新規作成
	 *
	 * @param	printer プリンタ名
	 */
	public void setToPrint( final String printer ) {
		toPrint = nval( getRequestParameter( printer ),toPrint );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"				,VERSION			)
				.println( "fileURL"				,fileURL			)
				.println( "file1"				,file1				)
				.println( "file2"				,file2				)
				.println( "sheetName"			,sheetName			)
				.println( "sheetNos"			,sheetNos			)
				.println( "sheetConstKeys"		,sheetConstKeys		)
				.println( "sheetConstAdrs"		,sheetConstAdrs		)
				.println( "useCellStyle"		,useCellStyle		)
				.println( "useAutoCellSize"		,useAutoCellSize	)
				.println( "useActiveWorkbook"	,useActiveWorkbook	)
				.println( "hyperLinkColumn"		,hyperLinkColumn	)
				.println( "addTitleSheet"		,addTitleSheet		)
				.println( "addImageFile"		,addImageFile		)
				.println( "valueType"			,valueType			)
				.println( "readText"			,readText			)
				.println( "readSheet"			,readSheet			)
				.println( "readName"			,readName			)
				.println( "readStyle"			,readStyle			)
				.println( "saveAs"				,saveAs				)
				.println( "toPrint"				,toPrint			)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
