/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.table;

import org.opengion.fukurou.db.DBUtil;
import org.opengion.fukurou.db.Transaction;			// 5.5.2.6 (2012/05/25)
import org.opengion.fukurou.model.Formatter;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.db.AbstractTableFilter;
import org.opengion.hayabusa.db.DBTableModel;

import java.util.Map;

/**
 * TableFilter_DBSELECT は、TableFilter インターフェースを継承した、DBTableModel 処理用の
 * 実装クラスです。
 *
 * ここでは、Body部にかかれたSQLを実行した結果を、テーブルモデルにセットします。
 * SQL文から取得されるカラム名とテーブルモデルのカラム名は一致している必要があります。
 * 検索結果のカラムがテーブルモデルに存在していない場合はエラーとなります。
 * 以下の属性を指定しないと、データが存在しない場合はNULLがセットされます。
 * また、2行以上検索された場合でも、1行目のデータのみをセットします。
 *
 * パラメータは、tableFilterタグの keys, vals にそれぞれ記述するか、BODY 部にCSS形式で記述します。
 * ただし、このフィルターは、BODY部に、SQL文も記述する為、注意が必要です。
 * ルール的には、一番最初に見つけた "{" と、"}" の間の文字列がパラメータと認識します。
 * よって、SQL文を記述する場合は、このパラメータの後ろに記述するか、keys,vals を利用ください。
 * 【パラメータ】
 *  {
 *       INNER_JOIN : データが存在しない場合、テーブルの該当行を削除します。(初期値:false)
 *       APPEND     : 2行以上検索された場合、データをアペンドします。       (初期値:false)
 *       SEPARATOR  : APPENDする場合の区切り文字を指定します。              (初期値:" "スペース)
 *  }
 *
 * Body部にかかれたSQLには[XXXX]形式の変数が指定できます。
 *
 * @og.formSample
 * ●形式：
 *      ① &lt;og:tableFilter classId="DBSELECT" &gt;
 *                
 *         &lt;/og:tableFilter&gt;
 *
 *      ② &lt;og:tableFilter classId="DBSELECT" &gt;
 *               {
 *                   INNER_JOIN : true ;
 *               }
 *               select AAA,BBB,CCC from XXX
 *         &lt;/og:tableFilter&gt;
 *
 * @og.rev 5.6.6.0 (2013/07/05) keys の整合性チェックを追加
 *
 * @version  0.9.0  2000/10/17
 * @author   Hiroki Nakamura
 * @since    JDK1.1,
 */
public class TableFilter_DBSELECT extends AbstractTableFilter {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.6.6.1 (2013/07/12)" ;

	private DBTableModel table		;		// 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加

	/**
	 * keys の整合性チェックを行うための初期設定を行います。
	 *
	 * @og.rev 5.6.6.1 (2013/07/12) keys の整合性チェック対応
	 *
	 * @param	keysMap keys の整合性チェックを行うための Map
	 */
	@Override
	protected void init( final Map<String,String> keysMap ) {
		keysMap.put( "INNER_JOIN"	, "データが存在しない場合、テーブルの該当行を削除します(初期値:false)"			);
		keysMap.put( "APPEND"		, "2行以上検索された場合、データをアペンドします       (初期値:false)"			);
		keysMap.put( "SEPARATOR"	, "APPENDする場合の区切り文字を指定します              (初期値:\" \"スペース)"	);
	}

	/**
	 * DBTableModel処理を実行します。
	 *
	 * @og.rev 4.3.7.0 (2009/06/01) 実装大幅変更
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応
	 * @og.rev 5.5.2.6 (2012/05/25) protected変数を、private化したため、getterメソッドで取得するように変更
	 *
	 * @return 処理結果のDBTableModel
	 */
	public DBTableModel execute() {
		table = getDBTableModel();		// 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加

		final boolean innerJoin = StringUtil.nval( getValue("INNER_JOIN"), false );
		final boolean isAppend  = StringUtil.nval( getValue("APPEND"), false );
		final String separator  = StringUtil.nval( getValue("SEPARATOR"), " " );

		final Formatter format = new Formatter( table );
		format.setFormat( getSql() );				// 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加
		final int[] sqlClmNo = format.getClmNos();
		final String query = format.getQueryFormatString();

		String[] data = null;
		String[] param = null;
		int[] clmNo = null;
		final Transaction tran = getTransaction();	// 5.5.2.6 (2012/05/25)
		final String      dbid = getDbid();			// 5.5.2.6 (2012/05/25)
		final int[] rowNo = getParameterRows();		// 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加
		final int rowCount = rowNo.length;
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );		// 6.1.0.0 (2014/12/26) refactoring
		for ( int row = rowCount - 1; row >= 0; row-- ) {
			try {
				param = getTableModelData( rowNo[row], sqlClmNo );

				final String[][] dbData;
				if ( row == rowCount - 1 ) {
					final String[][] rtn = DBUtil.dbExecute( query, param, tran, dbid, true );		// 5.1.9.0 (2010/08/01) Transaction 対応
					clmNo = getTableColumnNo( rtn[0] );
					dbData = new String[rtn.length - 1][rtn[0].length];
					System.arraycopy( rtn, 1, dbData, 0, dbData.length );
				}
				else {
					dbData = DBUtil.dbExecute( query, param, tran, dbid, false );		// 5.1.9.0 (2010/08/01) Transaction 対応
				}

				data = table.getValues( rowNo[row] );
				if ( dbData != null && dbData.length > 0 && dbData[0] != null && dbData[0].length > 0 ) {
					for ( int i = 0; i < clmNo.length; i++ ) {
						if( isAppend ) {
//							final StringBuilder buf = new StringBuilder();	// 6.1.0.0 (2014/12/26) refactoring
							buf.setLength(0);
							for( int j = 0; j < dbData.length; j++ ) {
								if( j > 0 ) {
									buf.append( separator );
								}
								buf.append( dbData[j][i] );
							}
							data[clmNo[i]] = buf.toString();
						}
						else {
							data[clmNo[i]] = dbData[0][i]; // 副作用を及ぼしています。注意
						}
					}
				}
				else if( innerJoin ) {
					table.removeValue(rowNo[row]);
				}
			}
			catch ( RuntimeException ex ) {
				final ErrorMessage errMessage = makeErrorMessage( "TableFilter_DBSELECT Error", ErrorMessage.NG );
				errMessage.addMessage( rowNo[row] + 1, ErrorMessage.NG, ex.getMessage() );
				errMessage.addMessage( rowNo[row] + 1, ErrorMessage.NG, StringUtil.array2csv( data ) );
				errMessage.addMessage( rowNo[row] + 1, ErrorMessage.NG, "SQL=[" + getSql() + "]" );
				errMessage.addMessage( rowNo[row] + 1, ErrorMessage.NG, StringUtil.array2csv( param ) );
			}
		}

		return table;
	}

	/**
	 * 指定の行番号の、カラムNo配列(int[])に対応した値の配列を返します。
	 *
	 * 表示データの HybsSystem.ROW_SEL_KEY を元に、選ばれた 行を
	 * 処理の対象とします。
	 *
	 * @param	row   行番号
	 * @param	clmNo カラムNo配列(可変長引数)
	 *
	 * @return	行番号とカラムNo配列に対応した、値の配列
	 */
//	private String[] getTableModelData( final int row, final int[] clmNo ) {
	private String[] getTableModelData( final int row, final int... clmNo ) {
		String[] values = new String[clmNo.length];
		for( int i = 0; i < values.length; i++ ) {
			values[i] = table.getValue( row, clmNo[i] );
		}
		return values;
	}
}
