/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.table;

import org.opengion.hayabusa.db.AbstractTableFilter;
import org.opengion.hayabusa.db.DBTableModel;

import org.opengion.hayabusa.resource.ResourceFactory;
import org.opengion.hayabusa.resource.ResourceManager;

import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;

import java.util.Map;

/**
 * TableFilter_LABEL は、TableFilter インターフェースを継承した、DBTableModel 処理用の
 * 実装クラスです。
 *
 * ここでは、キーのCLM,LBL と、LANG より、ラベルリソースに変換した値を設定します。
 *
 * パラメータは、tableFilterタグの keys, vals にそれぞれ記述するか、BODY 部にCSS形式で記述します。
 * 【パラメータ】
 *  {
 *       SYSTEM_ID  : ラベルリソースの作成システムIDを指定します。無指定時は、ログイン時のリソースになります。
 *       LANG       : ラベルリソースの言語を指定します。無指定時は、日本語になります。
 *       CLM        : ラベルリソースのキーとなる値が設定されているカラム名を指定します。
 *       LBL        : CLMで指定されたカラム名から値を取り出し、ラベルリソースに変換した結果をセットします。
 *  }
 *
 * 5.7.8.0 (2014/07/04) 解除
 * CLM と LBL に同じカラムを指定すると、このフィルターを通過するたびに、変換後のラベルが
 * 再びキーとして変換しようと試みることになります。不測の事態に陥る可能性があるため、
 * その様な設定は出来なくなっています。
 *
 * 5.7.8.0 (2014/07/04) CLM と LBL に同じカラムを指定する事ができます。
 *
 * また、CLM または LBL で指定したカラムが DBTableModel に存在しない場合は、処理そのものを
 * 無視します。その場合は、警告も出力されませんので、ご注意ください。
 *
 * @og.formSample
 * ●形式：
 *      ① &lt;og:tableFilter classId="LABEL" keys="CLM,LBL" vals="COLUMN,LABEL_CLM" /&gt;
 *
 *      ② &lt;og:tableFilter classId="LABEL" &gt;
 *               {
 *                      SYSTEM_ID  : GF ;
 *                      LANG       : ja ;
 *                      CLM        : COLUMN ;
 *                      LBL        : LABEL_CLM ;
 *               }
 *         &lt;/og:tableFilter&gt;
 *
 * @og.rev 5.6.6.0 (2013/07/05) keys の整合性チェックを追加
 *
 * @version  0.9.0  2000/10/17
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.1,
 */
public class TableFilter_LABEL extends AbstractTableFilter {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.7.8.0 (2014/07/04)" ;

	/**
	 * keys の整合性チェックを行うための初期設定を行います。
	 *
	 * @og.rev 5.6.6.1 (2013/07/12) keys の整合性チェック対応
	 *
	 * @param	keysMap keys の整合性チェックを行うための Map
	 */
	@Override
	protected void init( final Map<String,String> keysMap ) {
		keysMap.put( "SYSTEM_ID" 	, "システムIDを指定します(初期値:ログイン時のリソース)"		);
		keysMap.put( "LANG"			, "言語を指定(初期値:日本語)"								);
		keysMap.put( "CLM"			, "キーとなる値が設定されているカラム名を指定"				);
		keysMap.put( "LBL"			, "ラベルリソースに変換した結果をセットするカラム名を指定"	);
	}

	/**
	 * DBTableModel処理を実行します。
	 *
	 * @og.rev 5.5.2.2 (2012/05/09) SYSTEM_ID を受け取るように変更
	 * @og.rev 5.5.2.6 (2012/05/25) protected変数を、private化したため、getterメソッドで取得するように変更
	 * @og.rev 5.7.8.0 (2014/07/04) CLM と LBL に同じカラムを指定できるように変更
	 *
	 * @return 処理結果のDBTableModel
	 */
	public DBTableModel execute() {
		final DBTableModel table = getDBTableModel();		// 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加

		final String systemId = getValue( "SYSTEM_ID" );		// 5.5.2.2 (2012/05/09)
		final String lang		= getValue( "LANG" );
		final ResourceManager resource = ResourceFactory.newInstance( systemId,lang,false );		// 5.5.2.2 (2012/05/09)

		final String clm = getValue( "CLM" );
		final String lbl = getValue( "LBL" );

		final int clmNo  = table.getColumnNo( clm,false );	// 存在しない場合は、-1 を返す。
		final int lblNo  = table.getColumnNo( lbl,false );

		// 5.7.8.0 (2014/07/04) CLM と LBL に同じカラムを指定できるように変更
//		if( clmNo == lblNo && clmNo >= 0 ) {
//		final String errMsg = "CLM と LBL に同じカラムを指定することは出来ません。" ;
//			ErrorMessage errMessage = makeErrorMessage( "TableFilter_LABEL Error",ErrorMessage.NG );
//			errMessage.addMessage( 0,ErrorMessage.NG,errMsg );
//			errMessage.addMessage( 0,ErrorMessage.NG,"CLM=[" + clm + "],LBL=[" + lbl + "]" );
//		}
//		else if( clmNo >= 0 && lblNo >= 0 ) {
		if( clmNo >= 0 && lblNo >= 0 ) {
			String[] data  = null;
			final int rowCnt = table.getRowCount();
			for( int row=0; row<rowCnt; row++ ) {
				String clmVal = null;
				try {
					data   = table.getValues( row );
					clmVal = data[clmNo];		// 変換する元のカラム名
					data[lblNo] = resource.getLabel( clmVal ) ;		// 副作用を及ぼしています。注意
				}
				catch( RuntimeException ex ) {
					final ErrorMessage errMessage = makeErrorMessage( "TableFilter_LABEL Error",ErrorMessage.NG );
					errMessage.addMessage( row+1,ErrorMessage.NG,ex.getMessage() );
					errMessage.addMessage( row+1,ErrorMessage.NG,StringUtil.array2csv( data ) );
					errMessage.addMessage( row+1,ErrorMessage.NG,"CLM=[" + clm + "],VAL=[" + clmVal + "]" );
				}
			}
		}

		return table;
	}
}
