/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.db;

import java.io.IOException;
import java.io.Reader;
import java.sql.Clob;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Types;
import java.sql.Date;
import java.sql.Timestamp;
import java.util.Locale;

import org.opengion.fukurou.util.Closer;
import org.opengion.fukurou.util.HybsDateUtil;
import static org.opengion.fukurou.util.HybsConst.CR;				// 6.1.0.0 (2014/12/26) refactoring

/**
 * ResultSet のデータ処理をまとめたクラスです。
 * ここでは、ResultSetMetaData から、カラム数、カラム名(NAME列)、
 * Type属性を取得し、ResultSet で、値を求める時に、Object型の
 * 処理を行います。
 * Object型としては、CLOB、ROWID、TIMESTAMP 型のみ取り扱っています。
 *
 * @og.rev 6.0.4.0 (2014/11/28) 新規作成
 * @og.group ＤＢ制御
 *
 * @version  6.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK6.0,
 */
public class ResultSetValue {

//	/** システム依存の改行記号をセットします。4.0.0.0(2007/10/17) */
//	private static final String CR = System.getProperty( "line.separator" );

	private final ResultSet	resultSet ;			// 内部で管理する ResultSet オブジェクト
	private final int		clmSize ;			// カラムサイズ
	private final String[]	names ;				// カラム名(ResultSetMetaData#getColumnLabel(int) の toUpperCase)
	private final int[]		type ;				// java.sql.Types の定数定義
	private final boolean	useObj ;			// オブジェクト型の Type が存在するかどうか。

	private final int[]		size ;				// カラムサイズ(ResultSetMetaData#getColumnDisplaySize(int))
	private final boolean[]	isWrit ;			// 書き込み許可(ResultSetMetaData#isWritable(int))

	/**
	 * ResultSet を引数にとるコンストラクタ
	 *
	 * ここで、カラムサイズ、カラム名、java.sql.Types の定数定義 を取得します。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) 新規作成
	 *
	 * @param	res 内部で管理する ResultSet オブジェクト
	 * @throws	java.sql.SQLException データベース・アクセス・エラーが発生した場合
	 */
	public ResultSetValue( final ResultSet res ) throws SQLException {
		resultSet = res;

		final ResultSetMetaData metaData  = resultSet.getMetaData();
		clmSize = metaData.getColumnCount();
		names	= new String[clmSize];
		type	= new int[clmSize];
		size	= new int[clmSize];
		isWrit	= new boolean[clmSize];

		boolean tempUseObj = false;						// そもそも、オブジェクト系のカラムがあるかどうか
		for( int i=0; i<clmSize; i++ ) {
			final int tp = metaData.getColumnType( i+1 );
			type[i] = tp ;
			if( tp == Types.CLOB || tp == Types.ROWID || tp == Types.TIMESTAMP ) { tempUseObj = true; }
			names[i]  = metaData.getColumnLabel(i+1).toUpperCase(Locale.JAPAN) ;
			size[i]	  = metaData.getColumnDisplaySize(i+1) ;
			isWrit[i] = metaData.isWritable(i+1) ;
		}
		useObj = tempUseObj ;
	}

	/**
	 * ResultSetMetaData で求めた、カラム数を返します。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) 新規作成
	 *
	 * @return  カラム数(データの列数)
	 */
	public int getColumnCount() { return clmSize ; }

	/**
	 * カラム名配列を返します。
	 *
	 * 配列は、0から始まり、カラム数-1 までの文字型配列に設定されます。
	 * カラム名は、ResultSetMetaData#getColumnLabel(int) を toUpperCase した
	 * 大文字が返されます。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) 新規作成
	 *
	 * @return	カラム名配列
	 */
	public String[] getNames() {
		return names.clone();
	}

	/**
	 * 指定のカラム番号のカラム名を返します。
	 *
	 * カラム名を取得する、カラム番号は、0から始まり、カラム数-1 までの数字で指定します。
	 * データベース上の、1から始まる番号とは、異なります。
	 * カラム名は、ResultSetMetaData#getColumnLabel(int) を toUpperCase した
	 * 大文字が返されます。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) 新規作成
	 *
	 * @param   clmNo  カラム番号。0から始まり、カラム数-1 までの数字です。
	 * @return  指定のカラム番号のカラム名
	 */
	public String getColumnName( final int clmNo ) {
		return names[clmNo];
	}

//	/**
//	 * java.sql.Types の定数定義配列を返します。
//	 *
//	 * 配列は、0から始まり、カラム数-1 までのint型配列に設定されます。
//	 * Types の定数定義は、ResultSetMetaData#getColumnType(int) の値です。
//	 *
//	 * @og.rev 6.0.4.0 (2014/11/28) 新規作成
//	 *
//	 * @return	カラム(0～カラム数-1)に対応した、Types定数定義配列
//	 */
//	public int[] getSqlTypes() {
//		return type.clone();
//	}

	/**
	 * 指定のカラム番号のjava.sql.Types の定数定義を返します。
	 *
	 * 配列は、0から始まり、カラム数-1 までのint型配列に設定されます。
	 * Types の定数定義は、ResultSetMetaData#getColumnType(int) の値です。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) 新規作成
	 *
	 * @param   clmNo  カラム番号。0から始まり、カラム数-1 までの数字です。
	 * @return	指定のカラム番号のTypes定数
	 */
	public int getSqlType( final int clmNo ) {
		return type[clmNo];
	}

//	/**
//	 * オブジェクト型の Type が存在するかどうか[true:存在する/false:存在しない]を返します。
//	 *
//	 * 基本的には、値を求める処理の内部ロジックでのみ使用します。
//	 * Types.CLOB 、Types.ROWID、Types.TIMESTAMP が、Types定義に含まれている場合、
//	 * true を返します。
//	 * 含まれない場合は、false です。
//	 * オブジェクト型 が存在する場合、値を求める処理に、専用のメソッドが必要になります。
//	 * 含まれない場合は、String.valueOf( Object ) で求めることが可能です。
//	 * この値を参考に、呼び出すメソッドを変えることで、処理の高速化を図ろうと
//	 * 考えています。(効果があるかどうかは、判りません。)
//	 *
//	 * @og.rev 6.0.4.0 (2014/11/28) 新規作成
//	 *
//	 * @return  オブジェクト型の Type が存在するかどうか[true:存在する/false:存在しない]
//	 */
//	public boolean useObjectType() {
//		return useObj;
//	}

	/**
	 * カラムのサイズのint配列を返します。
	 *
	 * 配列は、0から始まり、カラム数-1 までのint型配列に設定されます。
	 * カラムのサイズは、ResultSetMetaData#getColumnDisplaySize(int) の値です。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) 新規作成
	 *
	 * @return	カラムのサイズのint配列
	 */
	public int[] getColumnDisplaySizes() {
		return size.clone();
	}

	/**
	 * 指定のカラム番号のサイズを返します。
	 *
	 * カラムのサイズは、ResultSetMetaData#getColumnDisplaySize(int) の値です。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) 新規作成
	 *
	 * @param   clmNo  カラム番号。0から始まり、カラム数-1 までの数字です。
	 * @return	指定のカラム番号のサイズ
	 */
	public int getColumnDisplaySize( final int clmNo ) {
		return size[clmNo];
	}

	/**
	 * カラムの書き込み可能かどうかのboolean配列を返します。
	 *
	 * 配列は、0から始まり、カラム数-1 までのint型配列に設定されます。
	 * カラムの書き込み可能かどうかは、ResultSetMetaData#isWritable(int) の値です。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) 新規作成
	 *
	 * @return	書き込み可能かどうかのboolean配列
	 */
	public boolean[] isWritable() {
		return isWrit.clone();
	}

	/**
	 * 指定の書き込み可能かどうかを返します。
	 *
	 * カラムの書き込み可能かどうかは、ResultSetMetaData#isWritable(int) の値です。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) 新規作成
	 *
	 * @param   clmNo  カラム番号。0から始まり、カラム数-1 までの数字です。
	 * @return	書き込み可能かどうか
	 */
	public boolean isWritable( final int clmNo ) {
		return isWrit[clmNo];
	}

	/**
	 * カーソルを現在の位置から順方向に1行移動します。
	 *
	 * ResultSet#next() を呼び出しています。
	 * 結果は,すべて文字列に変換されて格納されます。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) 新規作成
	 *
	 * @return  新しい現在の行が有効である場合はtrue、行がそれ以上存在しない場合はfalse
	 * @see		java.sql.ResultSet#next()
	 * @throws	java.sql.SQLException データベース・アクセス・エラーが発生した場合、またはこのメソッドがクローズされた結果セットで呼び出された場合
	 */
	public boolean next() throws SQLException {
		return resultSet.next() ;
	}

	/**
	 * 現在のカーソル位置にあるレコードのカラム番号のデータを取得します。
	 *
	 * ResultSet#getObject( clmNo+1 ) を呼び出しています。
	 * 引数のカラム番号は、0から始まりますが、ResultSet のカラム順は、1から始まります。
	 * 指定は、0から始まるカラム番号です。
	 * 結果は,すべて文字列に変換されて返されます。
	 * また、null オブジェクトの場合も、ゼロ文字列に変換されて返されます。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) 新規作成: org.opengion.hayabusa.db.DBUtil#getValue( ResultSet , int , int ) から移動
	 *
	 * @param   clmNo  カラム番号。0から始まり、カラム数-1 までの数字です。
	 * @return  現在行のカラム番号のデータ(文字列)
	 * @throws	java.sql.SQLException データベース・アクセス・エラーが発生した場合
	 */
	public String getValue( final int clmNo ) throws SQLException {
		final String val ;
		final Object obj = resultSet.getObject( clmNo+1 );
		if( obj == null ) {
			val = "";
		}
		else if( useObj ) {
			switch( type[clmNo] ) {
				case Types.CLOB : 		val = getClobData( (Clob)obj ) ;
										break;
				case Types.ROWID: 		val = resultSet.getString(clmNo+1);
										break;
				case Types.TIMESTAMP : 	val = HybsDateUtil.getDate( ((Timestamp)obj).getTime() , "yyyyMMddHHmmss" );
										break;
				default : 				val = String.valueOf( obj );
										break;
			}
		}
		else {
			val = String.valueOf( obj );
		}

		return val ;
	}

	/**
	 * 現在のカーソル位置にあるレコードの全カラムデータを取得します。
	 *
	 * #getValue( clmNo ) を、0から、カラム数-1 まで呼び出して求めた文字列配列を返します。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) 新規作成
	 *
	 * @return  現在行の全カラムデータの文字列配列
	 * @throws	java.sql.SQLException データベース・アクセス・エラーが発生した場合
	 */
	public String[] getValues() throws SQLException {
		final String[] vals = new String[clmSize];

		for( int i=0; i<clmSize; i++ ) {
			vals[i] = getValue( i );
		}

		return vals ;
	}

	/**
	 * タイプに応じて変換された、Numberオブジェクトを返します。
	 *
	 * 条件に当てはまらない場合は、null を返します。
	 * org.opengion.hayabusa.io.HybsJDBCCategoryDataset2 から移動してきました。
	 * これは、検索結果をグラフ化する為の 値を取得する為のメソッドですので、
	 * 数値に変換できない場合は、エラーになります。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) 新規作成: org.opengion.hayabusa.db.DBUtil#getNumber( int , Object ) から移動
	 *
	 * @param   clmNo  カラム番号。0から始まり、カラム数-1 までの数字です。
	 * @return	Numberオブジェクト(条件に当てはまらない場合は、null)
	 * @see		java.sql.Types
	 * @throws	java.sql.SQLException データベース・アクセス・エラーが発生した場合
	 * @throws	RuntimeException 数字変換できなかった場合。
	 */
	public Number getNumber( final int clmNo ) throws SQLException {
		final Object obj = resultSet.getObject( clmNo+1 );
		Number value = null;

		switch( type[clmNo] ) {
			case Types.TINYINT:
			case Types.SMALLINT:
			case Types.INTEGER:
			case Types.BIGINT:
			case Types.FLOAT:
			case Types.DOUBLE:
			case Types.DECIMAL:
			case Types.NUMERIC:
			case Types.REAL: {
				value = (Number)obj;
				break;
			}
			case Types.DATE:
			case Types.TIME:  {
				value = Long.valueOf( ((Date)obj).getTime() );
				break;
			}
			// 5.6.2.1 (2013/03/08) Types.DATE と Types.TIMESTAMP で処理を分けます。
			case Types.TIMESTAMP: {
				value = Long.valueOf( ((Timestamp)obj).getTime() );
				break;
			}
			case Types.CHAR:
			case Types.VARCHAR:
			case Types.LONGVARCHAR: {
				final String str = (String)obj;
				try {
					value = Double.valueOf(str);
				}
				catch (NumberFormatException ex) {
					final String errMsg = "数字変換できませんでした。in=" + str
									+ CR + ex.getMessage() ;
					throw new RuntimeException( errMsg,ex );
					// suppress (value defaults to null)
				}
				break;
			}
			default:
				// not a value, can't use it (defaults to null)
				break;
		}

		return value;
	}

	/**
	 * カラムのタイプを表現する文字列値を返します。
	 *
	 * この文字列を用いて、CCSファイルでタイプごとの表示方法を
	 * 指定することができます。
	 * 現時点では、VARCHAR2,LONG,NUMBER,DATE,CLOB,NONE のどれかにあてはめます。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) 新規作成: org.opengion.hayabusa.db.DBUtil#type2ClassName( int ) から移動
	 *
	 * @param   clmNo  カラム番号。0から始まり、カラム数-1 までの数字です。
	 * @return	カラムのタイプを表現する文字列値
	 * @see		java.sql.Types
	 */
	public String getClassName( final int clmNo ) {
		final String rtn ;

		switch( type[clmNo] ) {
			case Types.CHAR:
			case Types.VARCHAR:
			case Types.BIT:
				rtn = "VARCHAR2"; break;
			case Types.LONGVARCHAR:
				rtn = "LONG"; break;
			case Types.TINYINT:
			case Types.SMALLINT:
			case Types.INTEGER:
			case Types.NUMERIC:
			case Types.BIGINT:
			case Types.FLOAT:
			case Types.DOUBLE:
			case Types.REAL:
			case Types.DECIMAL:
				rtn = "NUMBER"; break;
			case Types.DATE:
			case Types.TIME:
			case Types.TIMESTAMP:
				rtn = "DATE"; break;
			case Types.CLOB:
				rtn = "CLOB"; break;
			default:
				rtn = "NONE"; break;
		}

		return rtn;
	}

	/**
	 * Clob オブジェクトから文字列を取り出します。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) 新規作成: org.opengion.hayabusa.db.DBUtil#getClobData( Clob ) から移動
	 *
	 * @param	clobData Clobオブジェクト
	 * @return	Clobオブジェクトから取り出した文字列
	 * @throws	SQLException データベースアクセスエラー
	 * @throws	RuntimeException 入出力エラーが発生した場合
	 */
	private String getClobData( final Clob clobData ) throws SQLException {
		if( clobData == null ) { return ""; }

		final StringBuilder buf = new StringBuilder( 10000 );

		Reader reader = null;
		try {
			reader = clobData.getCharacterStream();
			final char[] ch = new char[10000];
			int  len ;
			while( (len = reader.read( ch )) >= 0 ) {
				buf.append( ch,0,len );
			}
		}
		catch( IOException ex ) {
			final String errMsg = "CLOBデータの読み込みに失敗しました。"
								+ ex.getMessage() ;
			throw new RuntimeException( errMsg,ex );
		}
		finally {
			Closer.ioClose( reader );
		}
		return buf.toString();
	}
}
