/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.transfer;

import java.io.BufferedReader;
import java.io.File;
import java.util.ArrayList;
import java.util.List;

import org.opengion.fukurou.db.Transaction;
import org.opengion.fukurou.util.ApplicationInfo;
import org.opengion.fukurou.util.Closer;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.LogWriter;
import org.opengion.fukurou.util.StringUtil;

/**
 * 伝送要求に対して、テキストファイルからデータを読取します。
 * 但し、読取されるデータについては、旧伝送システムの形式と互換性を持たせるため、
 * 31Byteから430Byteまでの400Byteを取得します。
 *
 * 読取するファイル名は、読取対象で指定します。
 * ファイル名は絶対パスで指定する必要があります。
 *
 * 読込及びその後の実行処理が正常終了した場合は、読取ファイルは削除されます。
 * 但し、読取パラメーターに"UNDEL"という文字を設定した場合は、正常終了した場合でも
 * ファイルは削除されません。
 *
 * また、読取するテキストファイルのエンコードは読取パラメーターが指定することができます。
 * 指定しない場合、システムリソースの"DB_ENCODE"で指定された値が適用されます。
 *
 * @og.group 伝送システム
 *
 * @version  5.0
 * @author   Hiroki.Nakamura
 * @since    JDK1.6
 */
public class TransferRead_SAMCB implements TransferRead {

	// 読取ファイルオブジェクト
	private String fileName;

	// 完了時に読取ファイルを削除するかどうか
	private boolean fileDel = true;

	/**
	 * ファイルからデータを読み取ります。
	 *
	 * @param config 伝送設定オブジェクト
	 * @param tran トランザクションオブジェクト
	 *
	 * @return 読み取りしたデータ(配列)
	 */
	@Override
	public String[] read( final TransferConfig config, final Transaction tran ) {
		fileName = config.getReadObj();
		final File fileRead = new File( fileName );
		if( !fileRead.exists() ) { return new String[0]; }

		String readPrm = config.getReadPrm();
		if( readPrm != null && readPrm.indexOf( "UNDEL" ) >= 0 ) {
			fileDel = false;
			readPrm = readPrm.replace( "UNDEL", "" ).trim();
		}

		// 読取ファイルのエンコード
		String fileEncode = readPrm;
		if( fileEncode == null || fileEncode.isEmpty() ) {
			fileEncode = "UTF-8";
		}

		final List<String> valList = new ArrayList<String>();
		final BufferedReader reader = FileUtil.getBufferedReader( fileRead, fileEncode );
		String line = null;
		try {
			while( ( line = reader.readLine() ) != null ) {
				line = StringUtil.stringFill( line, 500, fileEncode );
				final byte[] bytes = StringUtil.makeByte( line, fileEncode );
				line = StringUtil.makeString( bytes, 30, 400, fileEncode );
				valList.add( line );
			}
		}
		catch( Throwable ex ) {
			LogWriter.log( ex );
			final String errMsg = "ファイル読取時にエラーが発生しました。[LINE=" + line + "]";
			throw new RuntimeException( errMsg, ex );
		}
		finally {
			Closer.ioClose( reader );
		}

		return valList.toArray( new String[valList.size()] );
	}

	/**
	 * 更新(削除)対象のファイル名(配列)を返します。
	 *
	 * @return ファイル名(配列)
	 */
	@Override
	public String[] getKeys() {
		return new String[] { fileName };		// 5.5.2.4 (2012/05/16)
	}

	/**
	 * 更新(削除)対象のファイル名をセットします。
	 *
	 * @param keys ファイル名配列(可変長引数)
	 */
	@Override
//	public void setKeys( final String[] keys ) {
	public void setKeys( final String... keys ) {
		if( keys == null || keys.length == 0 ) { return; }
		fileName = keys[0];
	}

	/**
	 * 読取したデータに対して完了処理を行います。
	 * 具体的には、読取したテキストファイルを削除します。
	 *
	 * @param config 伝送設定オブジェクト
	 * @param tran トランザクションオブジェクト
	 */
	@Override
	public void complete( final TransferConfig config, final Transaction tran ) {
		if( !fileDel ) { return; }

		final File fileRead = new File( fileName );
		if( !fileRead.exists() ) {
			return;
		}

		final boolean rtn = fileRead.delete();
		if( !rtn ) {
			final String errMsg = "ファイルの削除に失敗しました。[FILE=" + fileRead.getAbsolutePath() + "]";
			throw new RuntimeException( errMsg );
		}
	}

	/**
	 * 読取したデータに対してエラー処理を行います。
	 * (ここでは何も処理しません)
	 *
	 * @param config 伝送設定オブジェクト
	 * @param appInfo DB接続情報
	 */
	@Override
//	public void error( final TransferConfig config, final ApplicationInfo appInfo ) {}
	public void error( final TransferConfig config, final ApplicationInfo appInfo ) {
		System.err.println( " Error:config=" + config + " , appInfo=" + appInfo );
	}
}
