/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.util;


/**
 * 個々のエラーメッセージを持っておくためのオブジェクトです。
 * このオブジェクトは、不変オブジェクトです。
 * 内部には、行番号とIDと結果と、メッセージリソースで使用される引数を持っています。
 * 結果は 0:正常 1:警告 2:異常 8:EXCEPTION 9:ORACLEエラーで、
 * ErrorMessageクラスの public static 変数で定義されている値を使用します。
 *
 * @og.group エラー処理
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class ErrMsg {
	/** バッファの初期容量を通常より多い目に設定します。  {@value}  */
	public static final int BUFFER_MIDDLE = 200;							// 5.1.9.0 (2010/08/01) 追加

	private final int no;			// 行番号
	private final int kekka;		// 結果 0:正常 1:警告 2:異常 8:EXCEPTION 9:ORACLEエラー
	private final String id;		// メッセージID
	private final String[] args ;	// メッセージの引数配列
	private final String pg;		// PG名 3.8.9.5 (2007/09/12)
	private final String step;		// ステップ名 3.8.9.5 (2007/09/12)

	/**
	 *  必要な引数をすべて設定して、ErrMsg オブジェクトを構築します。
	 *
	 * @param	no	行番号
	 * @param	kekka	結果 0:正常 1:警告 2:異常 8:EXCEPTION 9:ORACLEエラー
	 * @param    pg PG名
	 * @param    step STEP名
	 * @param	id	メッセージID
	 * @param    inArgs String... メッセージIDの引数
	 */
	public ErrMsg( final int no,final int kekka,
					final String pg,final String step,
					final String id,final String... inArgs) {
		this.no = no;
		this.kekka = kekka;
		this.id = id;
		final int cnt = inArgs.length;
		args = new String[cnt];
		for( int i=0; i<cnt; i++ ) {
			args[i] = argIn( inArgs[i] );
		}
		this.pg = pg;		// 3.8.9.5 (2007/09/12)
		this.step = step;	// 3.8.9.5 (2007/09/12)
	}

	/**
	 *  行番号のみ異なる、新しい ErrMsg オブジェクトを作成します。
	 *
	 * @param    newNo 行番号
	 *
	 * @return   行番号のみ異なる、新しい ErrMsgオブジェクト
	 */
	public ErrMsg copy( final int newNo ) {
		return new ErrMsg( newNo,kekka,pg,step,id,args );
	}

	/**
	 *  行番号を返します。
	 *
	 * @return   行番号
	 */
	public int getNo() { return no; }

	/**
	 *  結果を返します。
	 *
	 * @return   結果 0:正常 1:警告 2:異常 8:EXCEPTION 9:ORACLEエラー
	 */
	public int getKekka() { return kekka; }

	/**
	 *  メッセージIDを返します。
	 *
	 * @return   メッセージID
	 */
	public String getId() { return id; }

	/**
	 *  メッセージ引数を返します。
	 *
	 * @param    no 引数アドレス
	 *
	 * @return   メッセージ引数
	 */
	public String getArg( final int no ) { return args[no]; }

	/**
	 *  メッセージ引数配列を返します。
	 *
	 * @return   メッセージ引数配列
	 */
	public String[] getArgs() { return args.clone(); }

	/**
	 * 引数について、{#XXXX} 文字列の場合、#XXXX に変換します。
	 * この、{#XXXX} 文字列は、メッセージリソースの引数に、XXXX の
	 * ラベルリソースを適用させる場合に使用します。
	 * この処理は、メッセージリソースでの処理で互換性を確保する為です。
	 * なお、この変更のために、普通に、#YYYY と指定した文字列も、
	 * ラベルキーとして処理されるようになりますので、ご注意ください。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規作成
	 *
	 * @param    arg	入力引数
	 *
	 * @return   変換後引数
	 */
	private String argIn( final String arg ) {
		if( arg != null && arg.startsWith( "{#" ) ) {
			return arg.substring( 1,arg.length()-1 ) ;
		}
		else {
			return arg ;
		}
	}

	/**
	 *  PG名を返します。
	 *
	 * @og.rev  3.8.9.5 (2007/09/12) 新規作成
	 *
	 * @return   PG名
	 */
	public String getPg() { return pg; }

	/**
	 *  ステップ名を返します。
	 *
	 * @og.rev  3.8.9.5 (2007/09/12) 新規作成
	 *
	 * @return   ステップ名
	 */
	public String getStep() { return step; }

	/**
	 *  デバッグ用のオブジェクトの内部表現の文字列を返します。
	 *
	 * @og.rev 4.0.0.0 (2004/12/31) 内部表現を文字のみとする。
	 *
	 * @return   内部表現の文字列
	 */
	@Override
	public String toString() {
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
			.append( "id=[" ).append( id ).append( "] no=[" ).append( no ).append( ']' )		// 6.0.2.5 (2014/10/31) char を append する。
			.append( " kekka=[" ).append( kekka ).append( "] values={" );
		for( int i=0; i<args.length; i++ ) {
			if( args[i] != null ) {
				buf.append( ' ' ).append( args[i] );		// 6.0.2.5 (2014/10/31) char を append する。
			}
		}
		buf.append( " }" );
		return buf.toString();
	}
}
