/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.util;

import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Cell;

/**
 * POI による、Excel(xls/xlsx)の読み取りクラスです。
 *
 * xls/xlsx形式のEXCELを、DOM方式でテキストデータを読み取ります。
 * このクラスでは、ss.usermodel.Workbook を使用したオブジェクトを処理する為、
 * xls/xlsx両形式のファイルを、ExcelReaderEvent を介したイベントで読み取ります。
 * ExcelReaderEventイベントは、openGion形式のファイル読み取りに準拠した方法をサポートします。
 * ※ openGion形式のEXCELファイルとは、#NAME 列に、カラム名があり、#で始まる
 *    レコードは、コメントとして判断し、読み飛ばす処理の事です。
 *
 * @og.rev 6.0.3.0 (2014/11/13) 新規作成
 * @og.group その他
 *
 * @version  6.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK7.0,
 */
public final class ExcelReader_SS {
	/** このプログラムのVERSION文字列を設定します。 {@value} */
	private static final String VERSION = "6.0.3.0 (2014/11/13)" ;

	private static final String CR = System.getProperty("line.separator");

	/**
	 * 引数ファイル(Excel)を、Workbook を使用してテキスト化します。
	 *
	 * ExcelReaderEventは、EXCEL読み取り処理用の統一されたイベント処理クラスです。
	 * openGion特有のEXCEL処理方法(#NAME , 先頭行#コメントなど)を実装しています。
	 *
	 * これは、HSSFやXSSFの処理を、統一的なイベントモデルで扱うためです。
	 * SSモデルが良いのですが、巨大なXSSF(.xlsx)ファイルを解析すると、OutOfMemoryエラーが
	 * 発生する為、個々に処理する必要があります。
	 * あくまで、読み取り限定であれば、こちらのイベントモデルで十分です。
	 *
	 * @og.rev 6.0.3.0 (2014/11/13) 新規作成
	 *
	 * @param	fname 入力ファイル名
	 * @param	event イベント処理するオブジェクト
	 */
	public static void excelReader( final String fname , final ExcelReaderEvent event ) {
		Workbook wkbook = POIUtil.createWorkbook( fname );				// 6.0.2.3 (2014/10/10) メソッド利用
		for (int x=0; x<wkbook.getNumberOfSheets(); x++) {
			Sheet sheet = wkbook.getSheetAt( x );
			String sName = sheet.getSheetName();
			event.startSheet( sName,x );								// イベント発生
			int stR = sheet.getFirstRowNum();
			int edR = sheet.getLastRowNum();
			for (int y=stR; y<=edR; y++) {								// getLastRowNum は、含む
				Row rowObj = sheet.getRow( y );
				if( rowObj != null ) {
					int stC = rowObj.getFirstCellNum();
					int edC = rowObj.getLastCellNum();
					for (int z=stC; z<edC; z++) {						// getLastCellNum は、含まない
						if( event.isSkip( y ) ) { continue; }			// イベント発生
						Cell colObj = rowObj.getCell( z );
						if( colObj != null && colObj.getCellType() != Cell.CELL_TYPE_BLANK ) { 
							String text = POIUtil.getValue( colObj );
							event.value( text,y,z );					// Cellテキスト, Row番号, Cell番号
						}
					}
				}
			}
			event.endSheet( x );										// イベント発生
		}
	}

	/**
	 * アプリケーションのサンプルです。
	 *
	 * 入力ファイル名 は必須で、第一引数固定です。
	 *
	 * Usage: java org.opengion.fukurou.util.ExcelReader_SS 入力ファイル名
	 *
	 * @og.rev 6.0.3.0 (2014/11/13) 新規作成
	 *
	 * @param	args	コマンド引数配列
	 */
	public static void main( final String[] args ) {
		final String usageMsg = "Usage: java org.opengion.fukurou.util.ExcelReader_SS 入力ファイル名" ;
		if( args.length == 0 ) {
			System.err.println( usageMsg );
			return ;
		}

		String fname = args[0];

		ExcelReader_SS.excelReader(
			fname,
			new ExcelReaderEvent() {
				public boolean startSheet( final String shtNm,final int shtNo ) {
					System.out.println( "S[" + shtNo + "]=" + shtNm );
					return super.startSheet( shtNm,shtNo );
				}

		//		public void columnNames( final String[] names ) {
		//			System.out.println( "NM=" + java.util.Arrays.toString( names ) );
		//		}

		//		public void values( final String[] vals,final int rowNo ) {
		//			System.out.println( "V[" + rowNo + "]=" + java.util.Arrays.toString( vals ) );
		//		}

		//		public boolean isSkip( final int rowNo ) {
		//			super.isSkip( rowNo );
		//			return false;
		//		}

				public boolean value( final String val,final int rowNo,final int colNo ) {
					System.out.println( "R[" + rowNo + "],C[" + colNo + "]=" + val );
					return super.value( val,rowNo,colNo );
				}
			}
		);
	}
}
