/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.resource;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import static org.opengion.fukurou.util.HybsConst.CR ;				// 6.1.0.0 (2014/12/26)
import static org.opengion.fukurou.util.HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring

import java.util.Calendar;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap ;
import java.util.BitSet ;

/**
 * 事業所(CDJGS) 毎の休日カレンダデータオブジェクトです。
 *
 * カレンダデータは、指定の事業所に関して、すべての休日情報を持っています。
 * 元のカレンダテーブル(GE13)の １日(DY1)～３１日(DY31)までの日付け欄に対して、
 * 休日日付けの 年月日 に対する、休日かどうかを判断できるだけの情報を保持します。
 * 具体的には、休日の年月日に対する List を持つことになります。
 * このクラスは、パッケージプライベートになっています。このオブジェクトを作成するのは、
 * CalendarFactory#getCalendarData( String ) で行います。引数は、事業所コード(cdjgs)です。
 * このカレンダオブジェクトを使用するには、事業所カレンダテーブル(GE13) を使用する
 * ことを許可しておく必要があります。
 * 許可は、システムパラメータ の USE_CALENDAR_DATABASE 属性を true に
 * 設定します(初期値は、互換性優先の false です。)
 * この、カレンダテーブルは、GE13 固定です。他のテーブルを使用する場合は、
 * ビュー等を作成する必要があります。
 * カレンダテーブル は、通常 DEFAULT DBIDを使用しますが、RESOURCE_CALENDAR_DBID
 * を設定することで、他のデータベースから読み取ることが可能になります。
 *
 * @og.rev 3.6.0.0 (2004/09/17) 新規作成
 * @og.group リソース管理
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
class CalendarDBData implements CalendarData {
	private final SortedMap<String,BitSet> ymMap  = new TreeMap<String,BitSet>() ;	// 休日日付けデータを年月をキーに持ちます。

	/**
	 * コンストラクタ
	 *
	 * 配列文字列のデータを元に、CalendarDBDataオブジェクトを構築します。
	 * このコンストラクタは、他のパッケージから呼び出せないように、
	 * パッケージプライベートにしておきます。
	 * 年月データは、連続である必要があります。
	 * 途中に抜けがあるかどうかのチェックを行います。
	 *
	 * @param	data	データベース検索データ
	 * @param	isFlat	縦持ち(false)か横持ち(true)の区別
	 */
	CalendarDBData( final String[][] data,final boolean isFlat ) {
		if( isFlat ) {
			callFlatTable( data );
		}
		else {
			callVerticalTable( data );
		}
	}

	/**
	 * 横持ち(フラット)配列文字列のデータを元に、日付情報を構築します。
	 * 年月データは、連続である必要があります。
	 * 途中に抜けがあるかどうかのチェックを行います。
	 *
	 * @og.rev 3.8.8.0 (2007/12/22) 休日が全くない場合でも、日付設定する。
	 *
	 * @param data String[][]
	 *                     [0]:年月(YYYYMM) 200406 という形式
	 *                     [1]～[31] 1日～31日のデータ
	 *                           0:平日 / その他:休日
	 */
	private void callFlatTable( final String[][] data ) {
		for( int ym=0; ym<data.length; ym++ ) {
			final String yyyymm = data[ym][0] ;
			if( yyyymm.length() != 6 ) {
				final String errMsg = "年月(YYYYMM)は、YYYYMM(例：200406) という形式で指定して下さい。"
							+ " YYYYMM [" + yyyymm + "]" ;
				throw new HybsSystemException( errMsg );
			}

			final Calendar month = HybsSystem.getCalendar( yyyymm + "01" );	// 当月
			final int lastDay = month.getActualMaximum( Calendar.DAY_OF_MONTH );	// 月末日

			final BitSet ymData = new BitSet( lastDay+1 );
			for( int d=1; d<=lastDay; d++ ) {
				if( ! "0".equals( data[ym][d] ) ) {
					// 日付けのビットを立てます。
					ymData.set( d );
				}
			}

			// 日付けのキーがあり、休日設定があれば、有効月とみなし、セットします。
			// 3.8.8.0 (2007/12/22) 休日が全くない場合でも、日付設定する。
			ymMap.put( yyyymm,ymData );
		}
	}

	/**
	 * 縦持ち(バーティカル)配列文字列のデータを元に、日付情報を構築します。
	 * 年月データは、連続である必要があります。
	 * 途中に抜けがあるかどうかのチェックを行います。
	 *
	 * @og.rev 3.8.8.0 (2007/12/22) 休日が全くない場合でも、日付設定する。
	 *
	 * @param data String[][]
	 *                     [0]:年月(YYYYMMDD) 20040601 という形式
	 *                     [1] 0:平日 / その他:休日
	 */
	private void callVerticalTable( final String[][] data ) {
		String bk_yyyymm = null;
		BitSet ymData = null;
		for( int ymd=0; ymd<data.length; ymd++ ) {
			if( data[ymd][0].length() != 8 ) {
				final String errMsg = "年月日(YYYYMMDD)は、YYYYMMDD(例：20040601) という形式で指定して下さい。"
							+ " YYYYMMDD [" + data[ymd][0] + "]" ;
				throw new HybsSystemException( errMsg );
			}

			final String yyyymm = data[ymd][0].substring( 0,6 ) ;
			if( ! yyyymm.equals( bk_yyyymm ) ) {	// 月のブレイク
				// 日付けのキーがあり、休日設定があれば、有効月とみなし、セットします。
				// 3.8.8.0 (2007/12/22) 休日が全くない場合でも、日付設定する。
				if( ymData != null && bk_yyyymm != null ) {
					ymMap.put( bk_yyyymm,ymData );
				}
				bk_yyyymm = yyyymm;

				final Calendar month = HybsSystem.getCalendar( yyyymm + "01" );	// 当月
				final int lastDay = month.getActualMaximum( Calendar.DAY_OF_MONTH );	// 月末日

				ymData = new BitSet( lastDay+1 );
			}

			if( ! "0".equals( data[ymd][1] ) ) {
				// 日付けのビットを立てます。
				final int dt = Integer.parseInt( data[ymd][0].substring( 6 ) );
				ymData.set( dt );
			}
		}
		// 日付けのキーがあり、休日設定があれば、有効月とみなし、セットします。
		// 3.8.8.0 (2007/12/22) 休日が全くない場合でも、日付設定する。
		if( ymData != null && bk_yyyymm != null ) {
			ymMap.put( bk_yyyymm,ymData );
		}
	}

	/**
	 * 指定の日付けが、休日かどうかを返します。
	 * 指定の日付けが、データベースに設定されていない場合は、すべて休日を
	 * 返すことで、存在しない事を示します。
	 *
	 * @param  day Calendar 指定の日付け
	 *
	 * @return	休日：true それ以外：false
	 *
	 */
	public boolean isHoliday( final Calendar day ) {
		final String yyyymm = cal2YM( day );
		final BitSet ymData = ymMap.get( yyyymm );

		if( ymData != null ) {
			return ymData.get( day.get( Calendar.DATE ) );
		}
		else {
			return true;	// DB 上に登録されていない場合は、すべて休日とする。
		}
	}

	/**
	 * 指定の日付けから、範囲の間に、本日を含むかどうかを返します。
	 * 指定の日付けが、キャッシュしているデータの最大と最小の間に
	 * 存在しない場合は、常に false になります。
	 * 判定は、年月日の項目のみで比較し、時分秒は無視します。
	 *
	 * @og.rev 3.7.1.1 (2005/05/31) 新規追加
	 * @og.rev 3.8.8.6 (2007/04/20) today を毎回求めます。(キャッシュ対策)
	 *
	 * @param  day Calendar 指定の開始日付け
	 * @param	scope	範囲の日数
	 *
	 * @return	本日：true それ以外：false
	 */
	public boolean isContainedToday( final Calendar day,final int scope ) {
		final boolean rtnFlag;

		final Calendar today = Calendar.getInstance();
		today.set( Calendar.HOUR_OF_DAY ,12 );	// 昼にセット
		today.set( Calendar.MINUTE ,0 );
		today.set( Calendar.SECOND ,0 );

		if( scope == 1 ) {
			// false の確率の高い方から、比較します。
			rtnFlag = day.get( Calendar.DATE )  == today.get( Calendar.DATE  ) &&
						day.get( Calendar.MONTH ) == today.get( Calendar.MONTH ) &&
						day.get( Calendar.YEAR )  == today.get( Calendar.YEAR  ) ;
		}
		else {
			final Calendar next = (Calendar)day.clone();
			next.add( Calendar.DATE,scope );
			rtnFlag = day.before( today ) && next.after( today ) ;
		}
		return rtnFlag ;
	}

	/**
	 * 指定の開始、終了日の期間に、平日(稼働日)が何日あるか求めます。
	 * 調査月が、データベース上に存在していない場合は、エラーとします。
	 * 開始と終了が同じ日の場合は、１を返します。
	 *
	 * @param  start Calendar 開始日付け(稼働日に含めます)
	 * @param  end   Calendar 終了日付け(稼働日に含めます)
	 *
	 * @return	稼働日数
	 *
	 */
	public int getKadoubisu( final Calendar start,final Calendar end ) {

		final String stYM = cal2YM( start );
		final String edYM = cal2YM( end );

		final SortedMap<String,BitSet> subMap = ymMap.subMap( stYM,edYM+"\0" ) ;	// 終了キーはそのままでは含まれないため。

		@SuppressWarnings("rawtypes")
		final Map.Entry[] entrys = subMap.entrySet().toArray( new Map.Entry[subMap.size()] );

		int holidaySu = 0;
		for( int i=0; i<entrys.length; i++ ) {
			final String ym = (String)entrys[i].getKey();
			BitSet bt ;
			if( stYM.equals( ym ) ) {
				bt = ((BitSet)entrys[i].getValue()).get( start.get( Calendar.DATE ),31 );
			}
			else if( edYM.equals( ym ) ) {
				bt = ((BitSet)entrys[i].getValue()).get( 1,end.get( Calendar.DATE )+1 );
			}
			else {
				bt = (BitSet)entrys[i].getValue();
			}
			holidaySu += bt.cardinality() ;
		}

		final long diff = start.getTimeInMillis() - end.getTimeInMillis() ;
		final int dayCount = (int)(diff/(1000*60*60*24)) + 1;	// end も含むので＋１必要

		return dayCount - holidaySu ;
	}

	/**
	 * 指定の開始日に平日のみ期間を加算して求められる日付けを返します。
	 * これは、実稼働日計算に使用します。
	 * 例えば、start=20040810 , span=5 で、休日がなければ、10,11,12,13,14 となり、
	 * 20040815 を返します。
	 * 指定の日付けや、期間加算後の日付けが、キャッシュしているデータの
	 * 最大と最小の間に存在しない場合は、エラーとします。
	 *
	 * @param  start Calendar   開始日付け(YYYYMMDD 形式)
	 * @param	span 	稼動期間
	 *
	 * @return Calendar 開始日から稼動期間を加算した日付け(当日を含む)
	 *
	 */
	public Calendar getAfterDay( final Calendar start,final int span ) {
		final Calendar today = (Calendar)(start.clone());
		int suSpan = span ;
		while( suSpan > 0 ) {
			if( ! isHoliday( today ) ) { suSpan--; }
			today.add(Calendar.DATE, 1);		// 日にちを進める。
		}
		return today ;
	}

	/**
	 * カレンダオブジェクトより、年月を YYYYMM 形式にした 文字列を返します。
	 *
	 * @return  カレンダのYYYYMM 文字列
	 */
	private String cal2YM( final Calendar cal ) {
		return String.valueOf(
					cal.get( Calendar.YEAR ) * 100 +
					cal.get( Calendar.MONTH ) + 1 );
	}

	/**
	 * オブジェクトの識別子として，詳細なカレンダ情報を返します。
	 *
	 * @return  詳細なカレンダ情報
	 */
	public String toString() {
		final StringBuilder rtn = new StringBuilder( BUFFER_MIDDLE );
		rtn.append( "CLASS   : ").append( getClass().getName() ).append( CR );	// クラス名

		@SuppressWarnings("rawtypes")
		final Map.Entry[] entrys = ymMap.entrySet().toArray( new Map.Entry[ymMap.size()] );

		for( int i=0; i<entrys.length; i++ ) {
			final String yyyymm = (String)entrys[i].getKey();

			rtn.append( yyyymm );
			rtn.append( ':' );						// 6.0.2.5 (2014/10/31) char を append する。
			rtn.append( (BitSet)entrys[i].getValue() );
			rtn.append( CR );
		}

		return rtn.toString();
	}
}
