/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.hayabusa.db.AbstractDBType;
import org.opengion.hayabusa.db.DBTypeCheckUtil;

/**
 * 半角文字＋半角カタカナを扱う為の、カラム属性を定義します。
 *
 * 半角文字＋半角カタカナとは、X 属性に半角カタカナを加えた、
 * 「c &lt; 0x20 || ( c &gt; 0x7e &amp;&amp; c &lt; 0xff64 ) || ( c &gt;= 0xffa0 ) 以外」
 * でのみ構成された文字列のことです。
 *
 * タイプチェックとして、以下の条件を判定します。
 * ・文字列長は、Byte換算での文字数との比較
 * ・半角文字＋半角カタカナチェック
 * ・文字パラメータの 正規表現チェック
 * ・クロスサイトスクリプティングチェック
 *
 * @og.group データ属性
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DBType_XH extends AbstractDBType {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.2.2.0 (2010/11/01)" ;

	/**
	 * データが登録可能かどうかをチェックします。
	 * データがエラーの場合は、そのエラー内容を返します。
	 *
	 * @og.rev 3.0.1.3 (2003/03/11) DBTypeCheckUtilクラスを利用するように修正
	 * @og.rev 3.6.0.0 (2004/09/22) dbType パラメータを引数に追加
	 * @og.rev 3.6.1.0 (2005/01/05) 半角カタカナに、『､』を含めます。(0xff65 以上 → 0xff64以上)
	 * @og.rev 5.2.2.0 (2010/11/01) 厳密にチェック(isStrict=true)するフラグを追加
	 *
	 * @param   key		キー
	 * @param   value	値
	 * @param   sizeX	整数部分の文字列の長さ
	 * @param   sizeY	少数部分の文字列の長さ
	 * @param   typeParam	dbType パラメータ
	 * @param   isStrict	厳密にチェックするかどうか[true:する/false:標準的]
	 *
	 * @return  エラー内容
	 */
	@Override
	public ErrorMessage valueCheck( final String key ,final String value ,
									final int sizeX ,final int sizeY ,final String typeParam ,final boolean isStrict) {

		ErrorMessage msg = new ErrorMessage();
		if( value == null || value.isEmpty() ) { return msg; }

		final int len = (sizeY == 0) ? sizeX : sizeX + sizeY + 1;
		String check = DBTypeCheckUtil.byteLengthCheck( value,len );
		if( check != null ) {
			// 文字列の長さが指定の長さよりも長いです。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0006",key,value,check,String.valueOf( len ) );
		}

		final StringBuilder val = new StringBuilder( BUFFER_MIDDLE );
		boolean isError = false;
		for( int i=0; i<value.length(); i++ ) {
			final char ch = value.charAt( i );
			if( ch < 0x20 || ( ch > 0x7e && ch < 0xff64 ) || ( ch >= 0xffa0 ) ) {
				val.append( "<span class=\"NG\">" ).append( ch ).append( "</span>" );
				isError = true;
			}
			else {
				val.append( ch );
			}
		}
		if( isError ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,val.toString() );
		}

		// 3.6.0.0 (2004/09/22) dbType パラメータを使用したマッチチェック
		check = DBTypeCheckUtil.matcheCheck( value,typeParam );
		if( check != null ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,check );
		}

		// クロスサイトスクリプティング対策：'<', '>' は登録させない。
		msg = xssCheck( key ,value, msg );

		return msg;
	}
}
