/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.process;

import org.opengion.fukurou.util.Argument;
import org.opengion.fukurou.util.HybsConst;				// 6.1.0.0 (2014/12/26) refactoring

import java.util.Map;

/**
 * AbstractProcess は、ChainProcess インターフェースを実装した、Abstract クラスです。
 * ChainProcess を用いて、順次、バッチプロセスを実行することができます。
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
abstract public class AbstractProcess implements HybsProcess {

	/** システムの改行コードを設定します。*/
	protected static final String CR		 = HybsConst.CR;			// 6.1.0.0 (2014/12/26) refactoring
	/** StringBilderなどの初期値を設定します。	{@value} */
	protected static final int BUFFER_MIDDLE = HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring
	/** StringBilderなどの初期値を設定します。	{@value} */
	protected static final int BUFFER_LARGE  = HybsConst.BUFFER_LARGE;	// 6.1.0.0 (2014/12/26) refactoring

	/** タブセパレータ  */
	public static final char TAB = '\t';	// 6.0.2.5 (2014/10/31) タブ区切り文字を char 化します。

	private final Argument argments ;
	private LoggerProcess logger	;

	/**
	 * コンストラクター
	 *
	 * @param	name			このクラス(サブクラス)のクラス名称
	 * @param   mustProparty	必須チェックMap
	 * @param   usableProparty	整合性チェックMap
	 */
	public AbstractProcess( final String name , final Map<String,String> mustProparty ,final Map<String,String> usableProparty ) {
		argments = new Argument( name ) ;
		argments.setMustProparty( mustProparty );
		argments.setUsableProparty( usableProparty );
	}

	/**
	 * 引数形式を解析する 引数オブジェクトに、引数を設定します。
	 * Argument の文字列から、引数かプロパティをセットします。
	 * ［プロパティ］のキー部の大文字･小文字は、厳格に判定しています。
	 * Argument の文字列には、タイプがあります。
	 *
	 * ［コメント］  ： # で始まる引数で、使用されません。(登録もされません。)
	 * ［引数］      ： #,-,= 以外で始まる通常の文字列。登録の順番が指定されます。
	 * ［プロパティ］： - で始まり、キーと値を=で区切っているパラメータです。順序は無関係。
	 *
	 * @param   arg	引数
	 */
	public void putArgument( final String arg ) {
		argments.putArgument( arg ) ;
	}

	/**
	 * Argument の文字列から、プロパティをセットします。
	 * ［プロパティ］のキー部の大文字･小文字は、厳格に判定しています。
	 * このメソッドは、引数 や コメントの判断を行いません。プロパティ のみ
	 * 設定されるものとして、処理します。
	 * プロパティの key=val が初めから分割されている場合の簡易メソッドです。
	 *
	 * @param   key キー
	 * @param   val 値
	 */
	public void putArgument( final String key,final String val ) {
		argments.putArgument( key,val ) ;
	}

	/**
	 * 引数形式を解析する 引数オブジェクトを返します。
	 *
	 * @return	引数オブジェクト
	 */
	public Argument getArgument() {
		return argments ;
	}

	/**
	 * ディスプレイにメッセージを表示します。
	 *
	 * @param	msg	表示するメッセージ
	 */
	public void println( final String msg ) {
		if( logger != null ) {
			logger.println( msg ) ;
		}
	}

	/**
	 * ディスプレイにメッセージを表示します。
	 *
	 * @param	msg	表示するメッセージ
	 */
	public void logging( final String msg ) {
		if( logger != null ) {
			logger.logging( msg ) ;
		}
	}

	/**
	 * ディスプレイ出力する LoggerProcess オブジェクトをセットします。
	 *
	 * @param logger LoggerProcessオブジェクト
	 */
	public final void setLoggerProcess( final LoggerProcess logger ) {
		this.logger = logger ;
	}

	/**
	 * プロセスの内容表示を行います。
	 * Argument#toString() を呼び出しています。
	 *
	 * @return 内容表示
	 */
	@Override
	public String toString() {
		return argments.toString();
	}
}
