/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.filter;

import javax.servlet.ServletRequest;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.ServletException;
import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;

import java.io.File;												// 5.7.3.2 (2014/02/28) Tomcat8 対応
import java.io.PrintWriter;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.UnsupportedEncodingException;

import org.opengion.hayabusa.common.HybsSystem;						// 6.2.4.1 (2015/05/22)

import org.opengion.fukurou.util.Closer;
import org.opengion.fukurou.util.HybsDateUtil;
import org.opengion.fukurou.util.FileUtil;							// 6.2.0.0 (2015/02/27)
import static org.opengion.fukurou.util.HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring

/**
 * AccessStopFilter は、Filter インターフェースを継承した アクセス制御クラスです。
 * web.xml で filter 設定することにより、Webアプリケーションへのアクセスを制御できます。
 * また、SYSTEM ユーザーは、このフィルターを常に通過します。
 *
 * フィルターに対してweb.xml でパラメータを設定します。
 *   ・startTime:停止開始時刻
 *   ・stopTime :停止終了時刻
 *   ・filename :停止時メッセージ表示ファイル名
 *
 * 【WEB-INF/web.xml】
 *     &lt;filter&gt;
 *         &lt;filter-name&gt;AccessStopFilter&lt;/filter-name&gt;
 *         &lt;filter-class&gt;org.opengion.hayabusa.filter.AccessStopFilter&lt;/filter-class&gt;
 *         &lt;init-param&gt;
 *             &lt;param-name&gt;startTime&lt;/param-name&gt;
 *             &lt;param-value&gt;070000&lt;/param-value&gt;
 *         &lt;/init-param&gt;
 *         &lt;init-param&gt;
 *             &lt;param-name&gt;stopTime&lt;/param-name&gt;
 *             &lt;param-value&gt;070000&lt;/param-value&gt;
 *         &lt;/init-param&gt;
 *         &lt;init-param&gt;
 *             &lt;param-name&gt;filename&lt;/param-name&gt;
 *             &lt;param-value&gt;jsp/custom/stopFile.html&lt;/param-value&gt;
 *         &lt;/init-param&gt;
 *     &lt;/filter&gt;
 *
 *     &lt;filter-mapping&gt;
 *         &lt;filter-name&gt;AccessStopFilter&lt;/filter-name&gt;
 *         &lt;url-pattern&gt;/jsp/*&lt;/url-pattern&gt;
 *     &lt;/filter-mapping&gt;
 *
 * @og.group フィルター処理
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class AccessStopFilter implements Filter {

	private static boolean stopFilter	;
	private String startTime	= "230000";		// 停止開始時刻
	private String stopTime		= "070000";		// 停止終了時刻
	private String filename		;				// 停止時メッセージ表示ファイル名
	private int    startStop	;

	/**
	 * フィルター処理本体のメソッドです。
	 *
	 * @og.rev 3.1.3.0 (2003/04/10) UTF-8 決め打ちで、stopFile.html を返送する。
	 * @og.rev 3.1.8.0 (2003/05/16) 文字エンコードが、UTF-8 になっていないのを修正。
	 * @og.rev 6.2.0.0 (2015/02/27) new BufferedReader … を、FileUtil.getBufferedReader … に変更。
	 *
	 * @param	request		ServletRequestオブジェクト
	 * @param	response	ServletResponseオブジェクト
	 * @param	chain		FilterChainオブジェクト
	 * @throws IOException 入出力エラーが発生した場合、throw されます。
	 * @throws ServletException サーブレット関係のエラーが発生した場合、throw されます。
	 */
	public void doFilter( final ServletRequest request,
							final ServletResponse response,
							final FilterChain chain)
								throws IOException, ServletException {

		if( isStop( request ) ) {
			BufferedReader in = null ;
			try {
				response.setContentType( "text/html; charset=UTF-8" );
				final PrintWriter out = response.getWriter();
				in = FileUtil.getBufferedReader( new File( filename ), "UTF-8" );		// 6.2.0.0 (2015/02/27)
				String str ;
				while( (str = in.readLine()) != null ) {
					out.println( str );
				}
				out.flush();
			}
			catch( UnsupportedEncodingException ex ) {
				final String errMsg = "指定されたエンコーディングがサポートされていません。[UTF-8]" ;
				throw new RuntimeException( errMsg,ex );
			}
			catch( IOException ex ) {
				final String errMsg = "ストリームがオープン出来ませんでした。[" + filename + "]" ;
				throw new RuntimeException( errMsg,ex );
			}
			finally {
				Closer.ioClose( in );
			}
			return;
		}

		chain.doFilter(request, response);
	}

	/**
	 * フィルターの初期処理メソッドです。
	 *
	 * フィルターに対してweb.xml で初期パラメータを設定します。
	 *   ・startTime:停止開始時刻
	 *   ・stopTime :停止終了時刻
	 *   ・filename :停止時メッセージ表示ファイル名
	 *
	 * @og.rev 5.7.3.2 (2014/02/28) Tomcat8 対応。getRealPath( "/" ) の互換性のための修正。
	 * @og.rev 6.2.4.1 (2015/05/22) REAL_PATH 対応。realPath は、HybsSystem経由で、取得する。
	 *
	 * @param	filterConfig	FilterConfigオブジェクト
	 */
	public void init(final FilterConfig filterConfig) {

		startTime = filterConfig.getInitParameter("startTime");
		stopTime  = filterConfig.getInitParameter("stopTime");
		filename  = HybsSystem.getRealPath() + filterConfig.getInitParameter("filename");	// 6.2.4.1 (2015/05/22)

		if( startTime == null || stopTime == null ) {
			startStop = 0;
		}
		else {
			startStop = startTime.compareTo( stopTime );
		}
	}

	/**
	 * フィルターの終了処理メソッドです。
	 *
	 */
	public void destroy() {
		// ここでは処理を行いません。
	}

	/**
	 * フィルターの内部状態をチェックするメソッドです。
	 * 内部のフラグをもとに、停止/許可を求めます。
	 *
	 * @og.rev 3.1.8.0 (2003/05/16) 開始時刻と終了時刻を同一にしていると、画面からの制御が効かないバグを修正。
	 * @og.rev 5.5.3.2 (2012/06/08) 通過させるユーザーに、admin を追加します。
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するように修正します。
	 *
	 * @param request ServletRequestオブジェクト
	 *
	 * @return	(true:停止  false:実行許可)
	 */
	private boolean isStop( final ServletRequest request ) {
		final String userID = ((HttpServletRequest)request).getRemoteUser() ;
		// 5.5.3.2 (2012/06/08) 通過させるユーザーに、admin を追加
		if( "SYSTEM".equalsIgnoreCase( userID ) || "admin".equalsIgnoreCase( userID ) ) {
			return false;
		}

		// 4.0.0 (2005/01/31)
		final String time = HybsDateUtil.getDate( "HHmmss" );		// 5.5.7.2 (2012/10/09) HybsDateUtil を利用

		boolean rtnFlag = stopFilter;
		if( startStop < 0 ) {
			if( startTime.compareTo( time ) < 0 &&
				time.compareTo( stopTime )  < 0 ) {
					rtnFlag = true;
			}
		}
		else if( startStop > 0 ) {
			if( startTime.compareTo( time ) < 0 ||
				time.compareTo( stopTime )  < 0 ) {
					rtnFlag = true;
				}
		}
		return rtnFlag;
	}

	/**
	 * フィルターの強制停止/解除を設定するメソッドです。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) synchronized の廃止
	 *
	 * @param flag (true:停止  false:実行許可)
	 */
	public static void setStopFilter( final boolean flag ) {
		stopFilter = flag;
	}

	/**
	 * フィルターの内部状態(強制停止/解除)を取得するメソッドです。
	 * これは、現在、アクセス制限がどうなっているかという状態ではなく、
	 * 強制停止されているかどうかの確認メソッドです。
	 *
	 * @og.rev 4.0.0.0 (2007/11/29) getStopFilter() ⇒ isStopFilter() に変更
	 *
	 * @return	(true:停止  false:実行許可)
	 */
	public static boolean isStopFilter() {
		return stopFilter;
	}

	/**
	 * 内部状態を文字列で返します。
	 *
	 * @return	このクラスの文字列表示
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		final StringBuilder sb = new StringBuilder( BUFFER_MIDDLE )
			.append( "AccessStopFilter" )
			.append( '[' ).append( startTime ).append( "],")		// 6.0.2.5 (2014/10/31) char を append する。
			.append( '[' ).append( stopTime  ).append( "],")
			.append( '[' ).append( filename  ).append( "],");
		return sb.toString();
	}
}
