/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import java.util.LinkedHashMap;
import java.util.Map;

import org.opengion.fukurou.db.DBUtil;
import org.opengion.fukurou.util.ApplicationInfo;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.resource.ResourceFactory;
import org.opengion.hayabusa.resource.ResourceManager;
import static org.opengion.fukurou.util.HybsConst.BUFFER_LARGE;		// 6.1.0.0 (2014/12/26) refactoring

/**
 * データのコード情報を取り扱うクラスです。
 *
 * コードのキーとラベルの情報から、HTMLのメニューやリストを作成するための オプション
 * タグを作成したり、与えられたキーをもとに、チェック済みのオプションタグを作成したり
 * します。
 * QUERYのそれぞれのカラムは、以下の意味を持ちます。
 *  第１カラム(必須) : ラジオボタンのキー(値)
 *  第２カラム       : ラベル(指定されない場合は、ラベルリソースの短縮ラベルを使用します)
 *  第３カラム       : クラス そのオプションに色づけなどを行う為の指定します。
 *                     NULL(または、ゼロ文字列)の場合は、適用されません。
 *  第４カラム       : この値は'false'又は'0'である場合にそのラジオボタンを選択不可にします。
 *                     NULL(または、ゼロ文字列)の場合は、選択可能になります。
 *
 * @og.group 選択データ制御
 *
 * @version  4.0
 * @author   Hiroki Nakamura
 * @since    JDK5.0,
 */
public class Selection_DBRADIO extends Selection_NULL {
	private final long	  	DB_CACHE_TIME			= (long)HybsSystem.sysInt( "DB_CACHE_TIME" ) ;

	private final boolean	isClass ;			// クエリでクラス属性が指定されているかどうか
	private final boolean	isDisable ;			// クエリで選択不可属性が指定されているかどうか
	private final long		createTime ;		// キャッシュの破棄タイミングを計るための作成時間

	private final Map<String,Integer>	map  ;

	private final String[]	label 	;		// ラベル
	private final String[]	cls 	;		// クラス
	private final String[]	disabled ;		// ラジオ単位のdisable

	private static final int VAL		= 0;
	private static final int LBL		= 1;
	private static final int CLS		= 2;
	private static final int DISABLED	= 3;

	private static final ApplicationInfo appInfo;
	static {
		/** コネクションにアプリケーション情報を追記するかどうか指定 */
		final boolean USE_DB_APPLICATION_INFO  = HybsSystem.sysBool( "USE_DB_APPLICATION_INFO" ) ;

		// 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
		if( USE_DB_APPLICATION_INFO ) {
			final String SYSTEM_ID = HybsSystem.sys( "SYSTEM_ID" );
			appInfo = new ApplicationInfo();
			// ユーザーID,IPアドレス,ホスト名
			appInfo.setClientInfo( SYSTEM_ID,HybsSystem.HOST_ADRS,HybsSystem.HOST_NAME );
			// 画面ID,操作,プログラムID
			appInfo.setModuleInfo( "Selection_DBRADIO",null,null );
		}
		else {
			appInfo = null;
		}
	}

	/**
	 * コンストラクター
	 *
	 * ＤＢ検索用のＳＱＬ文を与えて、初期化します。
	 * SQL文は、KEY,SNAMEで、SNAMEが設定されてない場合は、LNAMEを使用します。
	 * LNAME は、通常の値を返す場合に、SNAME は、一覧表示の値を返す場合に使用します。
	 * 特別に、KEY のみの場合は、lang に基づく ResourceManager からﾗﾍﾞﾙを取得します。
	 * ただし、その場合は、オーナー(SYSTEM_ID)は選べません。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) String[] value 廃止
	 *
	 * @param	newQuery	DB検索(SQL)文字列
	 * @param dbid  データベース接続先ID
	 * @param lang  リソースを使用する場合の言語
	 */
	public Selection_DBRADIO( final String newQuery,final String dbid,final String lang ) {

		final String[][] cols = DBUtil.dbExecute( newQuery,null,appInfo,dbid );
		final int count = cols.length;

		label	= new String[count];
		cls		= new String[count];
		disabled= new String[count];
		map		= new LinkedHashMap<String,Integer>(count);

		final int len   = ( count > 0 ) ? cols[0].length : 0 ;
		isClass   = len > CLS ;			// >= 3 と同意;
		isDisable = len > DISABLED ;	// >= 4 と同意;

		boolean useLabelData = false ;
		ResourceManager resource = null;
		if( len == 1 ) {
			useLabelData = true;
			resource = ResourceFactory.newInstance( lang );
		}

		for( int i=0; i<count; i++ ) {
			final String value = cols[i][VAL];
			if( useLabelData ) {
				label[i] = resource.getShortLabel( value );
			}
			else {
				label[i] = cols[i][LBL];
			}

			if( isClass ) {
				cls[i] = cols[i][CLS];
			}
			if( isDisable ) {
				disabled[i] = cols[i][DISABLED];
			}

			map.put( value, Integer.valueOf( i ) );
		}

		createTime = System.currentTimeMillis() ;
	}

	/**
	 * 初期値が選択済みの 選択肢(オプション)を返します。
	 * このオプションは、引数の値を初期値とするオプションタグを返します。
	 *
	 * @og.rev 6.2.2.4 (2015/04/24) メソッド変更。旧 #getRadio( String , String , boolean )
	 *
	 * @param   name         ラジオの name
	 * @param   selectValue  選択されている値
	 * @param   useLabel     ラベル表示の有無 [true:有/false:無]
	 *
	 * @return  オプションタグ
	 * @og.rtnNotNull
	 */
	@Override
	public String getOption( final String name,final String selectValue,final boolean useLabel ) {
		final String inputTag = "<input type=\"radio\" name=\"" + name + "\" value=\"" ;
		final StringBuilder buf = new StringBuilder( BUFFER_LARGE );
		for( final Map.Entry<String, Integer> e : map.entrySet() ) {
			final String val = e.getKey();	// Ver 5.0.0.0 value ⇒ val
			final int valIdx = e.getValue().intValue();
			// 6.0.2.5 (2014/10/31) char を append する。
			if( useLabel ) {
				buf.append( "<label" );
				if( isClass && cls[valIdx] != null && cls[valIdx].length() > 0 ) {
					buf.append( " class=\"" ).append( cls[valIdx] ).append( '"' );
				}
				buf.append( '>' );
			}
			buf.append( inputTag ).append( val ).append( '"' );
			if( val.equals( selectValue ) ) {
				buf.append( " checked=\"checked\"" );
			}

			if( isDisable && disabled[valIdx] != null && disabled[valIdx].length() > 0 ) {
				if( "false".equalsIgnoreCase( disabled[valIdx] ) || "0".equals( disabled[valIdx] ) ) {
					buf.append( " disabled=\"disabled\"" );
				}
			}

			buf.append( "/>" );
			if( useLabel ) { buf.append( label[valIdx] ).append( "</label>" ); }
		}
		return buf.toString();
	}

	/**
	 * 選択肢(value)に対するラベルを返します。
	 * 選択肢(value)が、存在しなかった場合は、選択肢そのものを返します。
	 * このメソッドでは、短縮ラベルを返すかどうかを指定するフラグを指定します。
	 * getValueLabel( XX,false ) は、getValueLabel( XX ) と同じです。
	 *
	 * @og.rev 6.2.3.0 (2015/05/01) マーク表示を廃止。
	 *
	 * @param	selectValue	選択肢の値
	 * @param	flag	短縮ラベルを使用する [true:使用する/false:しない]
	 *
	 * @return  選択肢のラベル
	 * @see     #getValueLabel( String )
	 */
	@Override
	public String getValueLabel( final String selectValue,final boolean flag ) {
		// マッチするアドレスを探す。
		final Integer sel = map.get( selectValue );

		if( sel == null ) {
			return selectValue;
		}
		else {
			return label[sel.intValue()] ;
		}
	}

	/**
	 * オブジェクトのキャッシュが時間切れかどうかを返します。
	 * キャッシュが時間切れ(無効)であれば、true を、有効であれば、
	 * false を返します。
	 *
	 * @return  キャッシュが時間切れなら true
	 */
	@Override
	public boolean isTimeOver() {
		return ( System.currentTimeMillis() - createTime ) > DB_CACHE_TIME ;
	}
}
