/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.fukurou.util.TagBuffer;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)

import static org.opengion.fukurou.util.StringUtil.nval ;		// 6.1.0.0 (2014/12/26)

/**
 * ガントチャートの繰り返しタグのフォーマットに使用します。
 *
 * iGanttBar タグは、TDタグの繰返し部に記述することで、個々のガントのタスクを作成します。
 * これを使用する場合は、iGant タグを全体に一つ指定する必要があります。
 *
 * @og.formSample
 * ●形式：&lt;og:iGanttBar  ... /&gt;
 * ●body：なし
 * ●前提：headタグで、adjustEvent="Gantt" を指定してください。
 *
 * ●Tag定義：
 *   &lt;og:iGanttBar
 *       type       ○【TAG】ガント要素のタイプ（0:マイルストーン 1:バー ）(必須)
 *       color        【TAG】バーの色（マイルストーンの場合は不要）
 *       colorNo      【TAG】バーの色番号で指定します
 *       src          【TAG】マイルストーンのアイコン画像(URIより設定)
 *       start        【TAG】開始日(YYYYMMDD)、または開始時間(YYYYMMDDHH24MM)
 *       end          【TAG】終了日(YYYYMMDD)、または終了時間(YYYYMMDDHH24MM)
 *       text         【TAG】ガント要素の注釈メッセージ。
 *       href         【TAG】入力した場合はガントにリンクを付加します。
 *       target       【TAG】hrefを指定した場合に付加されるリンクのターゲット
 *       barLink      【TAG】hrefで付加されるリンクの範囲にバー本体を含めるかどうか[true/false]を指定します(初期値:false)
 *       startInput   【保留】ガント要素の開始日に紐付く入力欄（開始日または終了日に紐付く入力欄が設定しなければガント要素は動かせない）
 *       endInput     【保留】ガント要素の終了日に紐付く入力欄（開始日または終了日に紐付く入力欄が設定しなければガント要素は動かせない）
 *       chbox        【保留】行選択のチェックボックス 設定値：[ROW.ID]
 *       cost         【TAG】開始日、終了日を利用して積上げられる工数を指定します。
 *       capacity     【TAG】積上げの最大値（キャパシティ）を指定します。
 *       debug        【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *   (通常ガント)
 *  &lt;og:view
 *      viewFormType = "HTMLCustomTable"
 *      command      = "{&#064;command}"
 *      writable     = "false"
 *      useScrollBar = "false"
 *  &gt;
 *    &lt;og:thead rowspan="2"&gt;
 *      &lt;tr&gt;
 *          &lt;td&gt;[NOORDER]&lt;/td&gt;
 *          &lt;td rowspan="2" class="gantt zoom{&#064;VZOOM}" style="line-height:normal;text-align:left;background-image: url('../image/ganttBG{&#064;VZOOM}.gif');" &gt;
 *              &lt;div style="position:relative;left:0px;z-index:101; " height="100%" width="100%" &gt;
 *                  &lt;og:ganttHeader
 *                      startDate       = "{&#064;startDate}"
 *                      endDate         = "{&#064;endDate}"
 *                      zoom            = "{&#064;VZOOM}"
 *                      daySpan         = "{&#064;daySpan}"
 *                  /&gt;
 *              &lt;/div&gt;
 *          &lt;/td&gt;
 *      &lt;/tr&gt;
 *    &lt;/og:thead&gt;
 *    &lt;og:tbody rowspan="2"&gt;
 *      &lt;tr&gt;
 *          &lt;td&gt;[NOORDER]&lt;/td&gt;
 *          &lt;td rowspan="2" class="gantt zoom{&#064;VZOOM}" style="text-align:left;background-image: url('../image/ganttBG{&#064;VZOOM}.gif');" &gt;
 *              &lt;og:iGanttBar type="1" colorNo="[COLOR]" start="[DYORDER]" end="[DYNOKI]" text="[PN]" /&gt;
 *              &lt;og:iGanttBar type="0" src="../image/dia_blue.gif" start="[DYORDER]"      text="開始" /&gt;
 *              &lt;og:iGanttBar type="0" src="../image/dia_red.gif"  end="[DYNOKI]"         text="終了" /&gt;
 *          &lt;/td&gt;
 *      &lt;/tr&gt;
 *    &lt;/og:tbody&gt;
 *  &lt;/og:view&gt;
 *    
 *   &lt;og:iGantt
 *       margeRows      = "true"
 *       fixedCols      = "1"
 *   /&gt;
 *
 *   (積み上げガント)
 *  &lt;og:view
 *      viewFormType = "HTMLStackedGanttTable"
 *      command      = "{&#064;command}"
 *      writable     = "false"
 *      useScrollBar = "false"
 *      useParam     = "true"
 *      numberType   = "none"
 *  &gt;
 *  &lt;og:stackParam
 *      stackColumns = "NOORDER"
 *  /&gt;
 * 
 *    &lt;og:thead rowspan="2"&gt;
 *      &lt;tr&gt;
 *          &lt;td&gt;[NOORDER]&lt;/td&gt;
 *          &lt;td&gt;[PN]&lt;/td&gt;
 *          &lt;td rowspan="2" class="gantt zoom{&#064;SZOOM}" style="text-align:left;background-image: url('../image/ganttBG{&#064;SZOOM}.gif');" &gt;
 *              &lt;div style="position:relative;left:0px;z-index:101; " height="100%" width="100%" &gt;
 *                  &lt;og:ganttHeader
 *                      startDate   = "{&#064;startDate}"
 *                      endDate     = "{&#064;endDate}"
 *                      zoom        = "{&#064;SZOOM}"
 *                      calDB       = "GE13"
 *                      arg1        = "A"
 *                  /&gt;
 *              &lt;/div&gt;
 *          &lt;/td&gt;
 *      &lt;/tr&gt;
 *    &lt;/og:thead&gt;
 *    &lt;og:tbody rowspan="2"&gt;
 *      &lt;tr&gt;
 *          &lt;td&gt;[NOORDER]&lt;/td&gt;
 *          &lt;td&gt;[PN]&lt;/td&gt;
 *          &lt;td rowspan="2" class="gantt zoom{&#064;VZOOM}" style="text-align:left;background-image: url('../image/ganttBG{&#064;VZOOM}.gif');" &gt;
 *              &lt;iGanttBar type="1" colorNo="[COLOR]" start="[DYORDER]" end="[DYNOKI]"  text="[COSTCLM]"
 *                    cost="[COSTCLM]" capacity="[CAPACITY]" /&gt;
 *              &lt;og:iGanttBar type="0" src="../image/dia_blue.gif" start="[DYORDER]"      text="開始" /&gt;
 *          &lt;/td&gt;
 *      &lt;/tr&gt;
 *    &lt;/og:tbody&gt;
 *  &lt;/og:view&gt;
 *    
 *   &lt;og:iGantt
 *       margeRows      = "true"
 *       fixedCols      = "1"
 *       verticalShift  = "false"
 *       viewGantt      = "{&#064;viewGantt}"
 *       stackHoliday   = "{&#064;stackHoliday}"
 *       useBgColor     = "true"
 *       viewMode       = "2"
 *   /&gt;
 *
 * @og.rev 5.6.3.2 (2013/04/12) 新規作成
 * @og.group 画面部品
 *
 * @version  5.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK6.0,
 */
public class ViewIGanttBarTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.0.2.5 (2014/10/31)" ;

	private static final long serialVersionUID = 602520141031L ;

	private transient TagBuffer tag = new TagBuffer( "iGanttBar" ).addBody( "<!-- -->" );		// 6.1.1.0 (2015/01/17) TagBufferの連結記述

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 6.0.2.5 (2014/10/31) HTML5対応。javaScriptで、BODYがないと入れ子になってしまう。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		jspPrint( tag.makeTag() );

		return EVAL_PAGE ;		// ページの残りを評価する。
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		tag = new TagBuffer( "iGanttBar" ).addBody( "<!-- -->" );		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
	}

	/**
	 * 【TAG】ガント要素のタイプ[0:ﾏｲﾙｽﾄｰﾝ/1:ﾊﾞｰ]を指定します。
	 *
	 * @og.tag
	 * 0:マイルストーン を指定すると、その日（時間）に、マーカーがされます。
	 * ガントのような期間ではなく、そのポイントを表すのに使用します。
	 * 例えば、受注日、納期 などの設定値です。
	 * start か end どちらかの指定で構いません。
	 * start を使用した場合は、指定日（時間）の前方に、endを使用した場合は、
	 * 指定日（時間）の後方にマーカーが置かれます。
	 *
	 * @param   type ガント要素のタイプ [0:ﾏｲﾙｽﾄｰﾝ/1:ﾊﾞｰ]
	 */
	public void setType( final String type ) {
		tag.add( "type",nval( getRequestParameter( type ),null ) );
	}

	/**
	 * 【TAG】ガント要素（バー）の色コードを設定します。
	 *
	 * @og.tag
	 * color は、ガント要素の色コードを設定します。
	 * type="0:マイルストーン" の場合は不要です。
	 * このコードは、予め決められた色画像を呼び出しますので、任意の色が使えるという事ではありません。
	 * 色指定は、横棒の場合は、image/W_色コード.png の画像を使います。
	 * 色コード と 色番号が同時に指定された場合は、色コード が優先されます。
	 *
	 * type=2:差分枠 は、value ＞ baseVal の場合は、超過を、value ＜ baseValの場合は、不足を表します。
	 * <table border="1" frame="box" rules="all">
	 *   <caption>色番号 対応表</caption>
     *   <tr><th>色番号</th><th>設定値    </th><th>棒色   </th><th>　</th><th>横棒サンプル</th></tr>
     *   <tr><td> 0    </td><td>blue      </td><td>青     </td><td> </td><td style="background-image:url(../image/W_blue.png);">   　</td></tr>
     *   <tr><td> 1    </td><td>black     </td><td>黒     </td><td> </td><td style="background-image:url(../image/W_black.png);">  　</td></tr>
     *   <tr><td> 2    </td><td>brown     </td><td>茶     </td><td> </td><td style="background-image:url(../image/W_brown.png);">  　</td></tr>
     *   <tr><td> 3    </td><td>green     </td><td>緑     </td><td> </td><td style="background-image:url(../image/W_green.png);">  　</td></tr>
     *   <tr><td> 4    </td><td>lightblue </td><td>薄青   </td><td> </td><td style="background-image:url(../image/W_lightblue.png);"> 　</td></tr>
     *   <tr><td> 5    </td><td>lightgreen</td><td>薄緑   </td><td> </td><td style="background-image:url(../image/W_lightgreen.png);"> 　</td></tr>
     *   <tr><td> 6    </td><td>orange    </td><td>オレンジ</td><td> </td><td style="background-image:url(../image/W_orange.png);"> 　</td></tr>
     *   <tr><td> 7    </td><td>pink      </td><td>ピンク </td><td> </td><td style="background-image:url(../image/W_pink.png);">  　</td></tr>
     *   <tr><td> 8    </td><td>purple    </td><td>紫     </td><td> </td><td style="background-image:url(../image/W_purple.png);">  　</td></tr>
     *   <tr><td> 9    </td><td>red       </td><td>赤     </td><td> </td><td style="background-image:url(../image/W_red.png);">       　</td></tr>
     *   <tr><td> 10   </td><td>yellow    </td><td>黄     </td><td> </td><td style="background-image:url(../image/W_yellow.png);"> 　</td></tr>
     *   <tr><td> 11   </td><td>navy      </td><td>濃紺   </td><td> </td><td style="background-image:url(../image/W_navy.png);">  　</td></tr>
     *   <tr><td> 12   </td><td>teal      </td><td>青緑   </td><td> </td><td style="background-image:url(../image/W_teal.png);">  　</td></tr>
     *   <tr><td> 13   </td><td>gray      </td><td>灰     </td><td> </td><td style="background-image:url(../image/W_gray.png);">    　</td></tr>
     *   <tr><td> 14   </td><td>indigo    </td><td>藍     </td><td> </td><td style="background-image:url(../image/W_indigo.png);"> 　</td></tr>
     *   <tr><td> 15   </td><td>slategray </td><td>鉄色   </td><td> </td><td style="background-image:url(../image/W_slategray.png);"> 　</td></tr>
     *   <tr><td> 16   </td><td>chocorlate</td><td>チョコレート</td><td> </td><td style="background-image:url(../image/W_chocorlate.png);"> 　</td></tr>
     *   <tr><td> 17   </td><td>darkgreen </td><td>深緑   </td><td> </td><td style="background-image:url(../image/W_darkgreen.png);"> 　</td></tr>
     *   <tr><td> 18   </td><td>violet    </td><td>すみれ </td><td> </td><td style="background-image:url(../image/W_violet.png);"> 　</td></tr>
     *   <tr><td> 19   </td><td>darkslategray</td><td>深鉄色</td><td> </td><td style="background-image:url(../image/W_darkslategray.png);"> 　</td></tr>
     *   <tr><td> 20   </td><td>lime      </td><td>緑     </td><td> </td><td style="background-image:url(../image/W_lime.png);">    　</td></tr>
     *   <tr><td> 21   </td><td>aqua      </td><td>水色   </td><td> </td><td style="background-image:url(../image/W_aqua.png);">  　</td></tr>
     *   <tr><td> 22   </td><td>maroon    </td><td>ひわだいろ</td><td> </td><td style="background-image:url(../image/W_maroon.png);"> 　</td></tr>
     *   <tr><td> 23   </td><td>olive     </td><td>オリーブ</td><td> </td><td style="background-image:url(../image/W_olive.png);">  　</td></tr>
     *   <tr><td> 24   </td><td>silver    </td><td>銀     </td><td> </td><td style="background-image:url(../image/W_silver.png);"> 　</td></tr>
     *   <tr><td> 25   </td><td>fuchsia   </td><td>赤紫   </td><td> </td><td style="background-image:url(../image/W_fuchsia.png);"> 　</td></tr>
     *   <tr><td colspan="6"> </td></tr>
     *   <tr><td> BK   </td><td>black2    </td><td>黒     </td><td> </td><td style="background-image:url(../image/W_black2.png);"> 　</td></tr>
     *   <tr><td> WT   </td><td>white2    </td><td>白     </td><td> </td><td style="background-image:url(../image/W_white2.png);"> 　</td></tr>
     *   <tr><td> BL   </td><td>blue2     </td><td>青     </td><td> </td><td style="background-image:url(../image/W_blue2.png);">   　</td></tr>
     *   <tr><td> RD   </td><td>red2      </td><td>赤     </td><td> </td><td style="background-image:url(../image/W_red2.png);">   　</td></tr>
     *   <tr><td> YL   </td><td>yellow2   </td><td>黄     </td><td> </td><td style="background-image:url(../image/W_yellow2.png);"> 　</td></tr>
	 * </table>
	 *
	 * @param   color バー色
	 */
	public void setColor( final String color ) {
		tag.add( "color",nval( getRequestParameter( color ),null ) );
	}

	/**
	 * 【TAG】グラフ要素の色番号を設定します。
	 *
	 * @og.tag
	 * 色番号は、色コードを直接指定するのではなく、番号で指定する場合に使用します。
	 * 棒グラフ要素（type="1"(横棒),"6"(縦棒)）の色を設定します。
	 * "1～25 , BK,WT,BL,RD,YL" の色番号で指定します。
	 * 色番号は、与えた数字を内部色数で割り算した余を使用しますので、連番の場合でも
	 * 繰返し使われます。特殊な色として、BK,WT,BL,RD,YLは、番号ではなく記号で指定できます。
	 * 色コード と 色番号が同時に指定された場合は、色コード が優先されます。
	 *
	 * @param   clrNo 色番号 [1～25/BK/WT/BL/RD/YL]
	 * @see		#setColor( String )
	 */
	public void setColorNo( final String clrNo ) {
		tag.add( "colorNo",nval( getRequestParameter( clrNo ),null ) );
	}

	/**
	 * 【TAG】マイルストーンのアイコン画像(URIより設定)を設定します。
	 *
	 * @og.tag
	 * type="0:マイルストーン" の場合、表示するアイコンイメージの所在を指定します。
	 * 例："../image/xxx.png" 
	 *
	 * @param   src アイコン画像
	 */
	public void setSrc( final String src ) {
		tag.add( "src",nval( getRequestParameter( src ),null ) );
	}

	/**
	 * 【TAG】開始日(YYYYMMDD)、または開始時間(YYYYMMDDHH24MM)を設定します。
	 *
	 * @og.tag
	 * type="1:バー" の場合は、バー表示の場合の書き始め（開始）を指定します。
	 * この値は、バーでは、「含む」になります。
	 * type="0:マイルストーン" の場合は、マイルストンを左端に表示します。
	 *
	 * @param   start 開始日or開始時間 (YYYYMMDD or YYYYMMDDHH24MM)
	 * @see		#setType( String )
	 * @see		#setEnd( String )
	 */
	public void setStart( final String start ) {
		tag.add( "start",nval( getRequestParameter( start ),null ) );
	}

	/**
	 * 【TAG】終了日(YYYYMMDD)、または終了時間(YYYYMMDDHH24MM)を設定します。
	 *
	 * @og.tag
	 * type="1:バー" の場合は、バー表示の場合の書き終わり（終了）を指定します。
	 * この値は、日付か時刻かで「含む」/「含まない」が異なります。
	 * 日付の場合は、この値を「含む」です。例えば、4/12 を指定した場合、4/12は含みます。
	 * 時刻の場合は、「含まない」です。4/12 070000 を指定した場合は、午前7時は含まないので、
	 * ６時５９分５９秒までのバーが引かれます。
	 * type="0:マイルストーン" の場合は、マイルストンを右端に表示します。
	 *
	 * @param   end 終了日or終了時間 (YYYYMMDD or YYYYMMDDHH24MM)
	 * @see		#setType( String )
	 * @see		#setStart( String )
	 */
	public void setEnd( final String end ) {
		tag.add( "end",nval( getRequestParameter( end ),null ) );
	}

	/**
	 * 【TAG】バー上に表示するテキストを指定します。
	 *
	 * @og.tag
	 * バーの上に、値などの文字を重ねて表示できます。
	 *
	 * @param   text 表示テキスト
	 */
	public void setText( final String text ) {
		tag.add( "text",nval( getRequestParameter( text ),null ) );
	}

	/**
	 * 【TAG】リンク作成時の値を設定します。
	 *
	 * @og.tag
	 * バーの上の文字に対して、リンクを作成できます。
	 *
	 * @param   href リンク
	 * @see		#setTarget( String )
	 */
	public void setHref( final String href ) {
		tag.add( "href",nval( getRequestParameter( href ),null ) );
	}

	/**
	 * 【TAG】リンク作成時のターゲット名を指定します。
	 *
	 * @og.tag
	 * バーの上の文字に対して、リンクを作成するときの、ターゲット名を指定できます。
	 *
	 * @param   target ターゲット
	 * @see		#setHref( String )
	 */
	public void setTarget( final String target ) {
		tag.add( "target",nval( getRequestParameter( target ),null ) );
	}

	/**
	 * 【TAG】hrefで付加されるリンクの範囲にバー本体を含めるかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * true を指定すると、hrefで付加されるリンクの範囲にバー本体を含めます。
	 *
	 * @param   barLink バー可否 [true:する/false:しない]
	 * @see		#setHref( String )
	 */
	public void setBarLink( final String barLink ) {
		tag.add( "barLink",nval( getRequestParameter( barLink ),null ) );
	}

	/**
	 * 【保留】ガント要素の開始日に紐付く入力欄を指定します。
	 *
	 * @og.tag
	 * この入力欄がないと、ガント要素は動かせません。
	 *
	 * @param   startInput 開始日入力欄
	 * @see		#setStart( String )
	 */
	public void setStartInput( final String startInput ) {
		tag.add( "startInput",nval( getRequestParameter( startInput ),null ) );
	}

	/**
	 * 【保留】ガント要素の終了日に紐付く入力欄を指定します。
	 *
	 * @og.tag
	 * この入力欄がないと、ガント要素は動かせません。
	 *
	 * @param   endInput 終了日入力欄
	 * @see		#setEnd( String )
	 */
	public void setEndInput( final String endInput ) {
		tag.add( "endInput",nval( getRequestParameter( endInput ),null ) );
	}

	/**
	 * 【TAG】行選択のチェックボックスを指定します。
	 *
	 * @og.tag
	 * 例えば、[ROW.ID] を指定する事で、選択行を指定する事ができます。
	 *
	 * @param   chbox チェックボックス指定
	 */
	public void setChbox( final String chbox ) {
		tag.add( "chbox",nval( getRequestParameter( chbox ),null ) );
	}

	/**
	 * 【TAG】積み上げガントの工数を指定します。
	 *
	 * @og.tag
	 * 積み上げガントの工数を指定します。
	 * 開始日、終了日を利用して積上げ計算されます。
	 *
	 * @param   cost 積み上げガントの工数
	 */
	public void setCost( final String cost ) {
		tag.add( "cost",nval( getRequestParameter( cost ),null ) );
	}

	/**
	 * 【TAG】積上げの最大値（キャパシティ）を指定します。
	 *
	 * @og.tag
	 * 最大値を超えた場合は、マーカーされます。
	 *
	 * @param   capacity 積み上げガントの工数
	 */
	public void setCapacity( final String capacity ) {
		tag.add( "capacity",nval( getRequestParameter( capacity ),null ) );
	}

	/**
	 * タグの名称を、返します。
	 * 自分自身のクラス名より、自動的に取り出せないため、このメソッドをオーバーライドします。
	 *
	 * @return  タグの名称
	 * @og.rtnNotNull
	 */
	@Override
	protected String getTagName() {
		return "iGanttBar" ;
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "tag"			,tag.makeTag()	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
