/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.query;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.DBConstValue;

/**
 * データベース登録時に、固定値カラムの設定を行う情報を管理するクラスです。
 * このクラスは、GrowingMIS用の管理クラスです。
 * 具体的には、
 *     更新フラグ  FGUPD
 *     作成日時    DYSET
 *     作成ＰＧ    PGSET
 *     作成社員    USRSET
 *     更新日時    DYUPD
 *     更新ＰＧ    PGUPD
 *     更新社員    USRUPD
 * などの情報を、INSERT、UPDATE の状況に応じて提供します。
 * INSERT には、すべての情報(XXUPD)にも値をセットします。
 * これは、UPDATE された日付や時刻で、一覧検索できるようにする為で、
 * null にしておくと、IS NULL で問い合わせる必要があります。
 * INSERT データだけ取出す場合は、WHERE DYSET = DYUPD とします。
 *
 * コンストラクターでは、テーブル名、ユーザーID,画面ID を受け取ります。
 * 登録時刻は、このオブジェクトが生成された時間を通常は使用します。
 * テーブル名を渡すのは、ワークなど、テーブル単位に処理するカラムが
 * 異なるので、それへの対応です。
 * 処理が簡素化できるように、ルールを規定すべきです。
 *
 * @og.rev 4.1.2.0 (2008/03/12) 新規追加
 * @og.group データ登録
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DBConstValue_ZY implements DBConstValue {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.3.7.0 (2009/06/01)" ;

	private static final String[] MSTR_INSKEYS =
		new String[] { "FGUPD","DYSET","PGSET","USRSET","DYUPD","PGUPD","USRUPD" };

	private static final String[] MSTR_UPDKEYS =
		new String[] { "FGUPD","DYUPD","PGUPD","USRUPD" };

	private static final String[] ZERO = new String[0] ;

	private String[] insKeys = ZERO;
	private String[] insVals = ZERO;

	private String[] updKeys = ZERO;
	private String[] updVals = ZERO;

	/**
	 * 初期化情報を設定します。
	 *
	 * @param	table	テーブル名
	 * @param	user 	ユーザーID
	 * @param	gamen	画面ID
	 */
	public void init( final String table, final String user, final String gamen ) {
		if( isUseClm( table ) ) {
			final String userID  = "'" + user + "'";
			final String gamenID = "'" + gamen + "'";
			final String timestamp = "'" + HybsSystem.getDate( "yyyyMMddHHmmss" ) + "'" ;

			insKeys = MSTR_INSKEYS;
			insVals = new String[] {
							 "'1'"			// FGUPD
							,timestamp		// DYSET
							,gamenID		// PGSET
							,userID			// USRSET
							,timestamp		// DYUPD
							,gamenID		// PGUPD
							,userID			// USRUPD
			};

			updKeys = MSTR_UPDKEYS;
			updVals = new String[] {
							 "'1'"			// FGUPD
							,timestamp		// DYUPD
							,gamenID		// PGUPD
							,userID			// USRUPD
			};
		}
	}

	/**
	 * インサートする場合のキー配列を返します。
	 *
	 * 指定の値がなければ、長さゼロ配列を返します。
	 *
	 * @og.rev 4.3.1.1 (2008/08/23) 変更可能なオブジェクトの参照を直接返さないようにします。
	 *
	 * @return	インサートする場合のキー配列(非null)
	 * @og.rtnNotNull
	 */
	public String[] getInsertKeys() {
		return insKeys.clone();
	}

	/**
	 * インサート する場合の値配列を返します。
	 *
	 * 指定の値がなければ、長さゼロ配列を返します。
	 *
	 * @og.rev 4.3.1.1 (2008/08/23) 変更可能なオブジェクトの参照を直接返さないようにします。
	 *
	 * @return 	インサートする場合の値配列(非null)
	 * @og.rtnNotNull
	 */
	public String[] getInsertVals() {
		return insVals.clone();
	}

	/**
	 * アップデートする場合のキー配列を返します。
	 *
	 * 指定の値がなければ、長さゼロ配列を返します。
	 *
	 * @og.rev 4.3.1.1 (2008/08/23) 変更可能なオブジェクトの参照を直接返さないようにします。
	 *
	 * @return	アップデートする場合のキー配列(非null)
	 * @og.rtnNotNull
	 */
	public String[] getUpdateKeys() {
		return updKeys.clone();
	}

	/**
	 * アップデートする場合の値配列を返します。
	 *
	 * 指定の値がなければ、長さゼロ配列を返します。
	 *
	 * @og.rev 4.3.1.1 (2008/08/23) 変更可能なオブジェクトの参照を直接返さないようにします。
	 *
	 * @return	アップデートする場合のキー配列(非null)
	 * @og.rtnNotNull
	 */
	public String[] getUpdateVals() {
		return updVals.clone();
	}

	/**
	 * デリート(論理削除) する場合のキー配列を返します。
	 *
	 * 指定の値がなければ、長さゼロ配列を返します。
	 *
	 * @og.rev 4.3.7.0 (2009/06/01) 新規作成
	 *
	 * @return	デリートする場合のキー配列(非null)
	 * @og.rtnNotNull
	 */
	public String[] getLogicalDeleteKeys() {
		return ZERO;
	}

	/**
	 * デリート(論理削除) する場合の値配列を返します。
	 *
	 * 指定の値がなければ、長さゼロ配列を返します。
	 *
	 * @og.rev 4.3.7.0 (2009/06/01) 新規作成
	 *
	 * @return	デリートする場合のキー配列(非null)
	 * @og.rtnNotNull
	 */
	public String[] getLogicalDeleteVals() {
		return ZERO;
	}

	/**
	 * カラムを返すかどうかを決めるロジック
	 *
	 * テーブル名より、ロジック的に導き出します。
	 * Ｇ－ＭＩＳ の場合は、Z で始まり、かつ、ZYW でなく、かつ、ZY9 で
	 * ない、テーブルの場合は、すべてのテーブルに、
	 * "FGUPD","DYSET","PGSET","USRSET","DYUPD","PGUPD","USRUPD"
	 * の属性を持っていると仮定します。
	 *
	 * @param	table	テーブル名
	 *
	 * @return	カラムを返すかどうか(true:カラムを生成/false:カラムは生成しない)
	 */
	private boolean isUseClm( final String table ) {
		return	  table != null
			&&	  table.length() > 0
			&&	  table.charAt(0) == 'Z'				// 6.1.0.0 (2014/12/26) refactoring
			&&	! table.startsWith( "ZYW" )
			&&	! table.startsWith( "ZY9" ) ;
	}
}
