/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.ArrayList;
import java.util.Locale ;		// 6.0.2.4 (2014/10/17)

import javax.servlet.http.HttpServletRequest;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.DBTableModelUtil;
import org.opengion.hayabusa.servlet.MultipartRequest;
import org.opengion.hayabusa.servlet.UploadedFile;

import static org.opengion.fukurou.util.StringUtil.nval;

/**
 * クライアントのファイルをサーバーにアップロードするタグです。
 *
 * アップロード後の属性は、DBTableModel に格納することにより、通常のデータと
 * 同様の取り出し方が可能です。
 * また、通常のファイルアップロード時の、form で使用する、enctype="multipart/form-data"
 * を指定した場合の、他のリクエスト情報も、{&#064;XXXX} 変数で取り出すことが可能です。
 *
 * この upload タグでは、アップロード後に、指定のファイル名に変更する機能があります。
 * file 登録ダイアログで指定した name に、"_NEW" という名称を付けたリクエスト値を
 * ファイルのアップロードと同時に送信することで、この名前にファイルを付け替えます。
 * また、アップロード後のファイル名は、name 指定の名称で、取り出せます。
 * クライアントから登録したオリジナルのファイル名は、name に、"_ORG" という名称
 * で取り出すことが可能です。
 *
 * 通常では、これらのパラメータを、RequestAttribute 変数にセットしますので、{&#064;XXXX}で
 * 取り出すことが可能になります。さらに、"KEY","VALUE","ISFILE" のカラムを持った、
 * DBTableModel にセットします。
 *
 * 新機能として、columns を指定する事で、columns のカラムを持つ DBTableModel にセットします。
 * その場合は、カラム名_01 ～ カラム名_99 のように、アンダーバーで列データとなるキーを定義してください。
 * アンダーバーがない場合は、カラムだけ作成されます。カラム名と同じリクエストがあれば、
 * すべてのレコードに同じ値がセットされます。
 *
 * 新ファイル名に拡張子が設定されていない場合は、オリジナルファイル名の拡張子をセットします。
 *
 * HTML5 の type="file" 時の multiple 属性（アップロードファイルの複数選択機能）に対応します。(5.7.1.1 (2013/12/13))
 * その場合は、新しいファイル名への変更はできません。オリジナルのファイル名でアップロードされます
 *
 * 5.7.1.2 (2013/12/20) zip 対応
 * filename 属性に、".zip" の拡張子のファイル名を指定した場合は、アップロードされた一連のファイルを
 * ZIP圧縮します。これは、アップロード後の処理になります。
 * ZIP圧縮のオリジナルファイルは、そのまま残ります。
 * なお、ZIPファイルは、useBackup属性を true に設定しても、無関係に、上書きされます。
 *
 * 5.7.4.3 (2014/03/28) filename 属性のリクエスト変数対応
 * filename 属性のみ、{&#064;XXXX} のリクエスト変数が使えるようにします。
 * 他のパラメータでは使えません。
 * これは、multipart/form-data のリクエストでは、パートの分解処理をしないと、リクエスト変数が
 * 拾えない為、リクエスト変数は、この、upload タグ以降でのみ利用可能でした。
 * zip対応と関連付けて、filename 属性のみ、利用できるように、MultipartRequest 側の処理に組み込みます。
 *
 * 5.7.6.3 (2014/05/23) アップロードファイルのCSVセット
 * 個々に指定したアップロードファイル名は、XXX_NEW や XXX_ORG で取得できますが、
 * HTML5 の multiple 属性使用時や、アップロードされたファイルを一連で処理したい場合に
 * ファイル名を、CSV形式で取り出せるようにします。
 * キーは、共通で、UPLOAD_FILES とします。
 *
 * 6.0.2.4 (2014/10/17)
 * ① name + _NEW と同じ考え方で、_PFX(接頭辞) , _SFX(接尾辞) 機能を追加します。
 * ② 上記機能は、_NEW との併用可能。_NEW がない場合は、オリジナル(アップロードされた元のファイル名)に付与
 * ③ useBackup="rename" で、すでに同名のファイルが存在した場合に、"_001" のような文字列を追加したファイルにリネームします。
 *    Windowsの " - コピー (2)" に近いですが、桁数を抑えるのと、useBackup="true" と異なり、過去の同一ファイル名は
 *    そのまま、有効になります。同一ファイルが同一フォルダに存在する場合のみ連番が付与されます。
 * ④ name + _NEW の指定に、フォルダを含めることが可能。ただし、接続文字列は、"/" とする。
 *    同時にアップロードを行う場合に、個別にフォルダを振り分けたり、_PFX(接頭辞) に、フォルダを指定
 *    することも可能です。
 * ⑤ 最大転送サイズ(Byte)を 10M → 30M に変更
 *
 * @og.formSample
 * ●形式：&lt;og:upload fileURL="…" maxPostSize="…" /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:upload
 *       fileURL            【TAG】ファイルをアップロードするディレクトリを指定します (初期値:FILE_URL[=filetemp/])
 *       filename           【TAG】(通常使いません)ファイルを作成するときのファイル名をセットします(初期値:null)
 *       maxPostSize        【TAG】最大転送サイズ(Byte)を指定します(初期値:31457280=30M) 0,またはマイナスで無制限です。
 *       scope              【TAG】キャッシュする場合のスコープ[request/page/session/applicaton]を指定します(初期値:session)
 *       tableId            【TAG】(通常使いません)sessionから所得する DBTableModelオブジェクトの ID
 *       columns            【TAG】DBTableModel作成時に、指定のカラムの"_01"～"_99"の添え字をレコードとして作成します。
 *       useBackup          【TAG】すでに同名のファイルが存在した場合に、バックアップ処理するかどうか[true/false/rename]を指定します(初期値:null=false)
 *                                 useBackup="rename" で、"_001" のような文字列を追加します。
 *       language           【TAG】タグ内部で使用する言語コード[ja/en/zh/…]を指定します
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例 :
 * 【query.jsp】
 *       &lt;form method="POST" action="result.jsp" enctype="multipart/form-data" target="RESULT"&gt;
 *       &lt;table summary="layout" &gt;
 *       &lt;tr&gt;&lt;og:input type="text" name="submitter" value="{&#064;USER.JNAME}" size="20" lbl="MSG0014" /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;
 *           &lt;og:input type="file" name="file_01"      size="30" lbl="MSG0015" /&gt;
 *           &lt;og:input             name="file_01_NEW"  size="10" lbl="FILENAME" /&gt;
 *       &lt;/tr&gt;&lt;tr&gt;
 *           &lt;og:input type="file" name="file_02"      size="30" lbl="MSG0015" /&gt;
 *           &lt;og:input             name="file_02_NEW"  size="10" lbl="FILENAME" /&gt;
 *       &lt;/tr&gt;&lt;tr&gt;
 *           &lt;og:input type="file" name="file_03"      size="30" lbl="MSG0015" /&gt;
 *           &lt;og:input             name="file_03_NEW"  size="10" lbl="FILENAME" /&gt;
 *       &lt;/tr&gt;&lt;tr&gt;
 *           &lt;og:column name="writable"     value="false"  /&gt;
 *       &lt;/tr&gt;
 *       &lt;/table&gt;
 *
 * 【result.jsp】
 *       &lt;og:upload
 *           fileURL     = "{&#064;USER.ID}"
 *       /&gt;
 *       &lt;br /&gt;
 *       &lt;og:message lbl="MSG0003" comment="ファイルの登録が完了しました。" /&gt;
 *
 *       &lt;og:view
 *           command      = "NEW"
 *           viewFormType = "HTMLTable"
 *           writable     = "{&#064;writable}"
 *       /&gt;
 *
 *       &lt;table&gt;
 *       &lt;tr&gt;&lt;og:input name="submitter"   value="{&#064;submitter}"   /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="writable"    value="{&#064;writable}"    /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="directory"   value="{&#064;directory}"   /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="file_01"     value="{&#064;file_01}"     /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="file_01_NEW" value="{&#064;file_01_NEW}" /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="file_01_ORG" value="{&#064;file_01_ORG}" /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="file_02"     value="{&#064;file_02}"     /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="file_02_NEW" value="{&#064;file_02_NEW}" /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="file_02_ORG" value="{&#064;file_02_ORG}" /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="file_03"     value="{&#064;file_03}"     /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="file_03_NEW" value="{&#064;file_03_NEW}" /&gt;&lt;/tr&gt;
 *       &lt;tr&gt;&lt;og:input name="file_03_ORG" value="{&#064;file_03_ORG}" /&gt;&lt;/tr&gt;
 *       &lt;/table&gt;
 *
 * 【result.jsp】
 *       &lt;og:upload
 *           fileURL     = "{&#064;USER.ID}"
 *           columns     = "submitter,file"
 *       /&gt;
 *       &lt;br /&gt;
 *       &lt;og:message lbl="MSG0003" comment="ファイルの登録が完了しました。" /&gt;
 *
 *       &lt;og:view
 *           command      = "NEW"
 *           viewFormType = "HTMLTable"
 *           writable     = "{&#064;writable}"
 *       /&gt;
 *
 * @og.group ファイル入力
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class FileUploadTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.2.3.0 (2015/05/01)" ;

	private static final long serialVersionUID = 623020150501L ;

	/** 5.7.6.3 (2014/05/23) アップロードファイルのCSVセットのキー */
	public static final String UPLOAD_FILES = "UPLOAD_FILES" ;

	// 3.5.2.0 (2003/10/20) カラム名に、ISFILEを追加。
	private static final String[] names = new String[] { "KEY","VALUE","ISFILE" };
	private static final String   ENCODE = "UTF-8";			// 3.5.2.0 (2003/10/20) 廃止
	private String	fileURL 	= HybsSystem.sys( "FILE_URL" );
	private String  filename    ;							// 3.5.4.2 (2003/12/15)

	private int 	maxPostSize = 30*1024*1024;				// 6.0.2.4 (2014/10/17) 最大ファイル容量  30MB
	private String	tableId 	= HybsSystem.TBL_MDL_KEY ;

	// 5.6.5.2 (2013/06/21) DBTableModel作成時に、指定のカラムの"_01"～"_99"の添え字をレコードとして作成します。
	private String		columns		;
	// 5.6.5.3 (2013/06/28) すでに同名のファイルが存在した場合に、バックアップ処理(renameTo)するかどうか[true/false]を指定します(初期値:false)
	// 6.0.2.4 (2014/10/17) useBackup は、true/false/rename が指定可能
	private String	useBackup		;

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語(国際化)対応 エンコードの取得方法変更
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.1.3.0 (2003/04/10) UTF-8 決め打ちのエンコード情報を取得する。
	 * @og.rev 3.5.2.0 (2003/10/20) scope 属性を追加。
	 * @og.rev 3.5.4.2 (2003/12/15) ファイル名を指定できるようにします。
	 * @og.rev 3.6.0.8 (2004/11/19) DBTableModel をセーブする時に、トランザクションチェックを行います。
	 * @og.rev 3.7.1.1 (2005/05/23) フォルダがない場合は、複数階層分のフォルダを自動で作成します。
	 * @og.rev 3.8.1.3A (2006/01/30) 新ファイル名にオリジナルファイル名の拡張子をセットします
	 * @og.rev 5.3.7.0 (2011/07/01) エラーメッセージ内容変更
	 * @og.rev 5.6.5.2 (2013/06/21) columns 属性の追加
	 * @og.rev 5.6.5.3 (2013/06/28) useBackup 属性の追加
	 * @og.rev 5.8.8.0 (2015/06/05) エラー文の日本語化
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		startQueryTransaction( tableId );		// 3.6.0.8 (2004/11/19)
		final HttpServletRequest request = (HttpServletRequest)getRequest();

		try {
			final String directory = HybsSystem.url2dir( fileURL );
			final File dir = new File(directory);
			if( ! dir.exists() && ! dir.mkdirs() ) {
				final String errMsg = getResource().getLabel( "ERR0043" ) + "[" + directory + "]"; // 5.8.8.0 (2015/06/05)
				throw new HybsSystemException( errMsg );
			}

			// 3.8.1.3A (2006/01/30) 新ファイル名にオリジナルファイル名の拡張子をセットします
			// 5.6.5.3 (2013/06/28) useBackup 属性の追加
			final MultipartRequest multi = new MultipartRequest( request,directory,maxPostSize,ENCODE,filename,useBackup );
			// 5.6.5.2 (2013/06/21) columns 属性の追加
			DBTableModel table = null;
			if( columns == null ) {						// 5.6.5.2 (2013/06/21) columns 属性の追加
				table = makeDBTable( multi );
			}
			else {
				table = makeDBTableFromClms( multi );
			}

			// 3.5.2.0 (2003/10/20) scope 属性を追加。
			// 3.6.0.8 (2004/11/19) トランザクションチェックを行います。
			if( ! commitTableObject( tableId, table ) ) {
				jspPrint( "FileUploadTag " + getResource().getLabel( "ERR0041" ) ); // 5.8.8.0 (2015/06/05)
				return SKIP_PAGE;
			}

		} catch(IOException ex) {
			final String errMsg = getResource().getLabel( "ERR0044" ) // 5.8.8.0 (2015/06/05)
				+ ex.getMessage() + CR	// 5.3.7.0 (2011/07/01) errMsg 修正
				+ "(" + toString() + CR
				+ "FileURL=" + fileURL + ")";
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}

		return EVAL_PAGE ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.0.1.1 (2003/03/06) columns を廃止
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.5.4.2 (2003/12/15) ファイル名を指定できるようにします。
	 * @og.rev 5.6.5.2 (2013/06/21) columns 属性の追加
	 * @og.rev 5.6.5.2 (2013/06/21) useBackup 属性の追加
	 * @og.rev 6.0.2.4 (2014/10/17) useBackup 修正、最大ファイル容量 変更
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		fileURL 		= HybsSystem.sys( "FILE_URL" );
		maxPostSize		= 30*1024*1024;				// 6.0.2.4 (2014/10/17) 最大ファイル容量  30MB
		tableId 		= HybsSystem.TBL_MDL_KEY ;
		filename		= null;						// 3.5.4.2 (2003/12/15)
		columns			= null;						// 5.6.5.2 (2013/06/21)
		useBackup		= null;						// 6.0.2.4 (2014/10/17) 
	}

	/**
	 * ファイルアップロードの実行結果を DBTableModel に記述します。
	 *
	 * ここでは、"KEY","VALUE","ISFILE" のカラムに対して、値を設定していきます。
	 * 同時に、RequestAttribute 変数に、これらの値をセットすることで、
	 * {&#064;XXXX} で値が取り出せる様にしています。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語(国際化)対応 エンコードの取得方法変更
	 * @og.rev 3.0.1.1 (2003/03/06) request 情報から{&#064;XXXX} で値が取り出せる様に修正。
	 * @og.rev 3.5.2.0 (2003/10/20) カラム名(KEY,VALUE)に ISFILE を追加
	 * @og.rev 3.5.6.5 (2004/08/09) MultipartRequest 変更に伴なう修正(Enum変更、元ファイル名取得)
	 * @og.rev 3.5.6.6 (2004/08/23) 上記変更時のバグ修正。
	 * @og.rev 3.5.6.6 (2004/08/23) 元ファイルのキーを、XXXX_ORG にします。
	 * @og.rev 4.0.0.0 (2007/10/12) テーブルモデルの登録方法を変更
	 * @og.rev 5.3.2.0 (2011/02/01) チェック行のパラメーターはint配列側に変換して復元する。パラメーター名を保存する。
	 * @og.rev 5.4.4.2 (2012/02/03) CommonTagSupportと同様のチェックボックス判定を行う
	 * @og.rev 5.7.1.1 (2013/12/13) HTML5 ファイルアップロードの複数選択（multiple）対応
	 * @og.rev 5.7.1.2 (2013/12/20) 5.7.1.2 (2013/12/20) zip対応で、UploadedFile のメソッド変更
	 * @og.rev 5.7.3.0 (2014/02/07) zip対応の修正で、取得ファイル名が異なっていた。
	 * @og.rev 5.7.6.3 (2014/05/23) アップロードファイルのCSVセット
	 * @og.rev 6.2.3.0 (2015/05/01) アップロードファイルを、UPLOAD_FILESキーに、Listオブジェクトに格納します。
	 *
	 * @param	 multi	   MultipartRequestオブジェクト
	 *
	 * @return	 テーブルモデル
	 */
	private DBTableModel makeDBTable( final MultipartRequest multi ) {

		final DBTableModel table = DBTableModelUtil.newDBTable();

		table.init( names.length );

		for( int i=0; i<names.length; i++ ) {
			final DBColumn dbColumn = getDBColumn( names[i] );
			table.setDBColumn( i,dbColumn );
		}

		String[] values ;		// 4.0.0.0 (2007/10/12)
		final List<String> prmNames = new ArrayList<String>();

		// 5.7.1.1 (2013/12/13) HTML5 ファイルアップロードの複数選択（multiple）対応
		// 6.2.3.0 (2015/05/01) アップロードファイルを、UPLOAD_FILESキーに、Listオブジェクトに格納します。
		final List<String> list = new ArrayList<String>();
		final UploadedFile[] files = multi.getUploadedFile();
		for( int i=0; i<files.length; i++ ) {
			final String name = files[i].getName();					// multiple対応では、キーがかぶることがある。
			String val  = files[i].getUploadFile().getName();	// 5.7.3.0 (2014/02/07) 取得ファイル名が異なっていた。

			// 6.2.3.0 (2015/05/01) アップロードファイルを、UPLOAD_FILESキーに、Listオブジェクトに格納します。
			list.add( val );

			// "KEY","VALUE","ISFILE" の順にデータを作成します。
			values = new String[] { name, val, "1" };
			table.addColumnValues( values );
			setRequestAttribute( name,val );
			prmNames.add( name );				// 5.7.1.1 (2013/12/13) List に設定する。

			final String orgName = name + "_ORG" ;
			val  = files[i].getOriginalFileName();	// 注意：取得は、送信名

			// "KEY","VALUE","ISFILE" の順にデータを作成します。
			values = new String[] { orgName, val, "2" };
			table.addColumnValues( values );
			setRequestAttribute( orgName,val );
		}

		// 6.2.3.0 (2015/05/01) アップロードファイルを、UPLOAD_FILESキーに、Listオブジェクトに格納します。
		setRequestAttribute( UPLOAD_FILES,list );

		// "KEY","VALUE","ISFILE" の順にデータを作成します。
		values = new String[] { "directory", fileURL, "0" };
		table.addColumnValues( values );
		setRequestAttribute( "directory",fileURL );

		final String[] params = multi.getParameterNames();
		for( int i=0; i<params.length; i++ ) {
			final String name = params[i];
			// 5.3.2.0 (2011/02/01) チェック行のパラメーターはint配列側に変換
			if ( HybsSystem.ROW_SEL_KEY.equals( name ) ) {
				setRequestAttribute( name,multi.getIntParameters(name) );
			}
			else {
				// 5.6.5.2 (2013/06/21) チェックボックス配列の値取得を考慮した、MultipartRequest のパラメータ値取得
				final String val = getParamVal( name,multi );

				values = new String[] { name, val, "0" };
				table.addColumnValues( values );
				setRequestAttribute( name,val );
				prmNames.add( name );				// 5.7.1.1 (2013/12/13) List に設定する。
			}
		}

		// 5.3.2.0 (2011/02/01) パラメーター名を保存する。
		// 5.7.1.1 (2013/12/13) List に設定する。
	    setParameterNames( prmNames.toArray( new String[prmNames.size()] ) );

		return table ;
	}

	/**
	 * ファイルアップロードの実行結果を 横持の DBTableModel に記述します。
	 * 
	 * この処理は、columns 属性を設定した場合のみとします。
	 * 
	 * DBTableModel作成時に、指定のカラムの"_01"～"_99"の添え字をレコードとして作成します。
	 * 現状は、"KEY","VALUE","ISFILE" のカラムに、データを縦持ちで作成しています。
 	 * これを、横持で作成しますが、カラムの末尾に、"_01"～"_99" までの添え字を
	 * 持つ場合は、これをレコードと認識させます。
	 * 添え字がない場合は、カラムだけ作成されます。カラム名と同じリクエストがあれば、
	 * すべてのレコードに同じ値がセットされます。
	 *
	 * @og.rev 5.6.5.2 (2013/06/21) 新規作成
	 * @og.rev 5.6.6.1 (2013/07/12) 添え字がない場合の処理の見直し
	 * @og.rev 5.7.1.2 (2013/12/20) zip対応で、UploadedFile のメソッド変更
	 * @og.rev 5.7.3.0 (2014/02/07) zip対応の修正で、取得ファイル名が異なっていた。
	 * @og.rev 5.7.6.3 (2014/05/23) アップロードファイルのCSVセット
	 * @og.rev 6.0.2.4 (2014/10/17) _PFX , _SFX 対応
	 * @og.rev 6.2.3.0 (2015/05/01) アップロードファイルを、UPLOAD_FILESキーに、Listオブジェクトに格納します。
	 *
	 * @param	 multi	   MultipartRequestオブジェクト
	 *
	 * @return	 テーブルモデル
	 */
	private DBTableModel makeDBTableFromClms( final MultipartRequest multi ) {

		final DBTableModel table = DBTableModelUtil.newDBTable();

		final String[] clmNames = columns.split( "," );

		table.init( clmNames.length );

		// 値配列（１行分）
		String[] rowVal = new String[clmNames.length];

		// 5.7.1.1 (2013/12/13) HTML5 ファイルアップロードの複数選択（multiple）対応
		final UploadedFile[] files = multi.getUploadedFile();

		final List<String> prmNames = new ArrayList<String>();

		for( int i=0; i<clmNames.length; i++ ) {
			final String clm = clmNames[i] ;
			final DBColumn dbColumn = getDBColumn( clm );
			table.setDBColumn( i,dbColumn );

			// 先に、カラム名と一致するパラメータを初期値としてセットしておきます。
			String val = getParamVal( clm,multi );

			// 5.6.6.1 (2013/07/12) ファイル名も同様に、あれば初期値セットしておきます。
			if( val == null ) {
				// 5.7.1.1 (2013/12/13) HTML5 ファイルアップロードの複数選択（multiple）対応
				for( int j=0; j<files.length; j++ ) {
					final String nm = files[j].getName();
					if( clm.equalsIgnoreCase( nm ) ) {
						val = files[i].getUploadFile().getName();	// 5.7.3.0 (2014/02/07) 取得ファイル名が異なっていた。
						break;										// 5.7.6.3 (2014/05/23) たぶん有ったほうが良い。
					}
				}
			}
			// 5.7.1.1 (2013/12/13) getFilesystemName() の中に、newFile が null の場合は、original を返す処理がある。
			rowVal[i] = ( val == null ) ? "" : val ;
		}

		// 6.2.3.0 (2015/05/01) アップロードファイルを、UPLOAD_FILESキーに、Listオブジェクトに格納します。
		final List<String> list = new ArrayList<String>();

		for( int i=0; i<files.length; i++ ) {
			final String name = files[i].getName();
			// 5.6.6.1 (2013/07/12) 添え字がない場合の処理の見直し。先にレコードを作成
			String[] values = new String[clmNames.length];
			System.arraycopy( rowVal,0,values,0,values.length );		// 行にセットするに当たり、rowVal を values にコピーしておく。

			// ファイル名を Attribute で使えるようにセットしておく。
			final String fval = files[i].getUploadFile().getName();	// 5.7.3.0 (2014/02/07) 取得ファイル名が異なっていた。
			setRequestAttribute( name,fval );
			prmNames.add( name );								// 5.7.1.1 (2013/12/13) List に設定する。

			// 6.2.3.0 (2015/05/01) アップロードファイルを、UPLOAD_FILESキーに、Listオブジェクトに格納します。
			list.add( fval );

			final String orgName = name + "_ORG" ;
			final String oval  = files[i].getOriginalFileName();	// 注意：取得は、送信名
			setRequestAttribute( orgName,oval );

			// ファイルのキーを元に、添え字を検索します。
			final int adrs = name.lastIndexOf( '_' );		// 添え字は、'_' で区切られます。
			// 5.6.6.1 (2013/07/12) 添え字がない場合の処理の見直し。後続処理を行う。
			if( adrs > 0 ) {
				final String fnm = name.substring( 0,adrs );	// ファイル名(分割後)
				final String sub = name.substring( adrs );	// 添え字(アンダーバー含む)

				// カラム名で検索しながら、レコード単位になるようにセットしていきます。
				for( int j=0; j<clmNames.length; j++ ) {
					final String clm = clmNames[j] ;
					String nm  = null;
					String val = null;

					if( fnm.equalsIgnoreCase( clm ) ) {		// ファイル名カラム(_NEWファイル名も、この値にすでに変わっている）
						val = fval;
					}
					else if( ( fnm + "_ORG" ).equalsIgnoreCase( clm ) ) {		// 元ファイル名カラム
						val  = oval;
					}
					else if( ( fnm + "_NEW" ).equalsIgnoreCase( clm ) ) {		// 新ファイル名カラム
						nm   = name + "_NEW" ;
						val  = multi.getParameter( nm );
					}
					// 6.0.2.4 (2014/10/17) _PFX , _SFX 対応
					else if( ( fnm + "_PFX" ).equalsIgnoreCase( clm ) ) {		// プレフィックスカラム
						nm   = name + "_PFX" ;
						val  = multi.getParameter( nm );
					}
					else if( ( fnm + "_SFX" ).equalsIgnoreCase( clm ) ) {		//サフィックスカラム
						nm   = name + "_SFX" ;
						val  = multi.getParameter( nm );
					}
					else {
						nm  = clmNames[j] + sub;					// sub は、アンダーバー含む添え字
						// 5.6.5.2 (2013/06/21) チェックボックス配列の値取得を考慮した、MultipartRequest のパラメータ値取得
						val = getParamVal( nm,multi );
					}
					if( val != null ) { values[j] = val ; }		// val が null でないときのみセットします。
				}
			}
			table.addColumnValues( values );
		}
		// 6.2.3.0 (2015/05/01) アップロードファイルを、UPLOAD_FILESキーに、Listオブジェクトに格納します。
		setRequestAttribute( UPLOAD_FILES,list );

		// Attribute で使えるようにセットしておく。
		setRequestAttribute( "directory",fileURL );

		// Attribute で使えるようにセットしておく。
		final String[] params = multi.getParameterNames();
		for( int i=0; i<params.length; i++ ) {
			final String name = params[i];
			// 5.3.2.0 (2011/02/01) チェック行のパラメーターはint配列側に変換
			if ( HybsSystem.ROW_SEL_KEY.equals( name ) ) {
				setRequestAttribute( name,multi.getIntParameters(name) );
			}
			else {
				// 5.6.5.2 (2013/06/21) チェックボックス配列の値取得を考慮した、MultipartRequest のパラメータ値取得
				final String val = getParamVal( name,multi );
				setRequestAttribute( name,val );
				prmNames.add( name );				// 5.7.1.1 (2013/12/13) List に設定する。
			}
		}

		// 5.3.2.0 (2011/02/01) パラメーター名を保存する。
		// 5.7.1.1 (2013/12/13) List に設定する。
	    setParameterNames( prmNames.toArray( new String[prmNames.size()] ) );

		return table ;
	}

	/**
	 * チェックボックス配列の値取得を考慮した、MultipartRequest のパラメータ値取得
	 * 
	 * ここでは、、MultipartRequest のパラメータ値を取得します。
	 * 値の取得で、チェックボックス配列の場合は、取得した値が、"0" の場合のみ
	 * 配列でパラメータを取得し直し、"1" がないかどうか再度検索します。
 	 * チェックボックスでは、チェック時の "1" と、チェックされなかった場合の、
	 * hidden の "0" の両方の値が配列としてリクエストされるケースがある為です。
	 *
	 * @og.rev 5.6.5.2 (2013/06/21) 新規作成
	 *
	 * @param	 key	   MultipartRequestのパラメータ取得のためのキー
	 * @param	 multi	   MultipartRequestオブジェクト
	 *
	 * @return	 チェックボックス配列を考慮したパラメータ値
	 */
	private String getParamVal( final String key , final MultipartRequest multi ) {
		String val = multi.getParameter( key );

		if( "0".equals(val) ){ // 5.4.4.2 チェックボックス配列対応。
			final String[] vals = multi.getParameters( key );
			if( vals != null && vals.length > 1 ) {
				for( int j=0; j<vals.length; j++ ) {
					if( "1".equals( vals[j] ) ) {
						val = "1";
						break;
					}
				}
			}
		}
		return val ;
	}

	/**
	 * 【TAG】ファイルをアップロードするディレクトリを指定します
	 *		(初期値:FILE_URL[={@og.value org.opengion.hayabusa.common.SystemData#FILE_URL}])。
	 *
	 * @og.tag
	 * この属性で指定されるディレクトリに、アップロードされたファイルをセーブします。
	 * 指定方法は、通常の fileURL 属性と同様に、先頭が、'/' (UNIX) または、２文字目が、
	 * ":" (Windows)の場合は、指定のURLそのままのディレクトリに、そうでない場合は、
	 * fileURL = "{&#064;USER.ID}" と指定すると、FILE_URL 属性で指定のフォルダの下に、
	 * さらに、各個人ID別のフォルダを作成して、そこにセーブします。
	 * (初期値:システム定数のFILE_URL[={@og.value org.opengion.hayabusa.common.SystemData#FILE_URL}])。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) StringUtil.urlAppend メソッドの利用
	 * @og.rev 4.0.0.0 (2007/11/20) 指定されたディレクトリ名の最後が"\"or"/"で終わっていない場合に、"/"を付加する。
	 * @og.rev 5.7.1.1 (2013/12/13) リクエスト変数が使えないエラーを表示する。
	 * @og.rev 5.7.4.3 (2014/03/28) リクエスト変数が使えない事のチェックを行う。
	 *
	 * @param	url ファイルURL
	 * @see		org.opengion.hayabusa.common.SystemData#FILE_URL
	 */
	public void setFileURL( final String url ) {
		String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			final char ch = furl.charAt( furl.length()-1 );
			if( ch != '/' && ch != '\\' ) { furl = furl + "/"; }
			fileURL = StringUtil.urlAppend( fileURL,furl );
		}
		else {
			chckReqParam( url,"fileURL" );		// 5.7.4.3 (2014/03/28) リクエスト変数が使えない事のチェック
		}
	}

	/**
	 * 【TAG】最大転送サイズ(Byte)を指定します(初期値:31457280)。
	 *
	 * @og.tag
	 * 最大転送サイズを指定します。初期値は、30*1024*1024 = 30MB です。
	 * 指定は、Byte 単位で指定します。
	 * 0,またはマイナスを指定することで、制限チェックを外す（＝無制限）事ができます。
	 *
	 * @og.rev 3.0.1.1 (2003/03/06) maxPostSize の設定バグ修正。
	 * @og.rev 5.6.5.3 (2013/06/28) コメント追加（0,またはマイナスで無制限）
	 * @og.rev 5.7.4.3 (2014/03/28) リクエスト変数が使えない事のチェックを行う。
	 * @og.rev 6.0.2.4 (2014/10/17) 最大ファイル容量 10M → 30M 変更(JavaDocの修正)
	 *
	 * @param	maxPS 最大転送サイズ
	 */
	public void setMaxPostSize( final String maxPS ) {
		maxPostSize = nval( getRequestParameter( maxPS ),maxPostSize );

		chckReqParam( maxPS,"maxPostSize" );		// 5.7.4.3 (2014/03/28) リクエスト変数が使えない事のチェック
	}

	/**
	 * 【TAG】(通常は使いません)結果のDBTableModelを、sessionに登録するときのキーを指定します
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value org.opengion.hayabusa.common.HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @og.tag
	 * 検索結果より、DBTableModelオブジェクトを作成します。これを、下流のviewタグ等に
	 * 渡す場合に、通常は、session を利用します。その場合の登録キーです。
	 * query タグを同時に実行して、結果を求める場合、同一メモリに配置される為、
	 * この tableId 属性を利用して、メモリ空間を分けます。
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value org.opengion.hayabusa.common.HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @og.rev 5.7.4.3 (2014/03/28) リクエスト変数が使えない事のチェックを行う。
	 *
	 * @param	id テーブルID (sessionに登録する時のID)
	 */
	public void setTableId( final String id ) {
		tableId = nval( getRequestParameter( id ),tableId );

		chckReqParam( id,"tableId" );		// 5.7.4.3 (2014/03/28) リクエスト変数が使えない事のチェック
	}

	/**
	 * 【TAG】(通常使いません)ファイルを作成するときのファイル名をセットします(初期値:null)。
	 *
	 * @og.tag
	 * ファイルを作成するときのファイル名をセットします。
	 * これは、複数同時にアップロードファイル名を変更する時に使用できません。
	 * 通常、アップロードされたファイル名を指定する場合、アップロードするinput タグの
	 * name 属性に指定する名称 ＋ "_NEW" というリクエスト値を同時に送信すれば、
	 * 内部的に関連付けて、ファイル名を更新します。
	 * その場合、クライアントより指定したファイル名は、name属性＋"_ORG" という
	 * リクエスト値として取得することが可能になります。
	 * name属性 には、最終的に設定されたファイル名がセットされています。
	 * いずれの値も、{&#064;name属性＋"_ORG"} や、{&#064;name属性＋"_NEW"}として、
	 * アップロードのオリジナルと変更後のファイル名を取得することが出来ます。
	 *
	 * 5.7.1.2 (2013/12/20) zip 対応
	 * filename 属性に、".zip" の拡張子のファイル名を指定した場合は、アップロードされた一連のファイルを
	 * ZIP圧縮します。これは、アップロード後の処理になります。
	 * ZIP圧縮のオリジナルファイルは、そのまま残ります。
	 * なお、ZIPファイルは、useBackup属性を true に設定しても、無関係に、上書きされます。
	 *
	 * 5.7.4.3 (2014/03/28) filename 属性のリクエスト変数対応
	 * filename 属性のみ、{&#064;XXXX} のリクエスト変数が使えるようにします。
	 * 他のパラメータでは使えません。
	 * これは、multipart/form-data のリクエストでは、パートの分解処理をしないと、リクエスト変数が
	 * 拾えない為、リクエスト変数は、この、upload タグ以降でのみ利用可能でした。
	 * zip対応と関連付けて、filename 属性のみ、利用できるように、MultipartRequest 側の処理に組み込みます。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) ファイル名を指定できるようにします。
	 * @og.rev 5.7.1.1 (2013/12/13) リクエスト変数が使えないエラーを表示する。
	 * @og.rev 5.7.4.3 (2014/03/28) リクエスト変数を使えるようにします。
	 *
	 * @param   fname ファイル名
	 */
	public void setFilename( final String fname ) {
		filename = nval( getReservedParameter( fname ),null );		// 予約語のみ処理をします。
	}

	/**
	 * 【TAG】DBTableModel作成時に、指定のカラムの"_01"～"_99"の添え字をレコードとして作成します。
	 *
	 * @og.tag
	 * 現状は、"KEY","VALUE","ISFILE" のカラムに、データを縦持ちで作成しています。
	 * これを、横持で作成しますが、カラムの末尾に、"_01"～"_99" までの添え字を
	 * 持つ場合は、これをレコードと認識させます。
	 * アンダーバーがない場合は、カラムだけ作成されます。カラム名と同じリクエストがあれば、
	 * すべてのレコードに同じ値がセットされます。
	 * この処理は、columns 属性を設定した場合のみとします。
	 *
	 * @og.rev 5.6.5.2 (2013/06/21) 新規作成
	 * @og.rev 5.7.4.3 (2014/03/28) リクエスト変数が使えない事のチェックを行う。
	 *
	 * @param   clms DBTableModel作成時のカラム列(CSV形式)
	 */
	public void setColumns( final String clms ) {
		columns = nval( getRequestParameter( clms ),columns );

		chckReqParam( clms,"columns" );		// 5.7.4.3 (2014/03/28) リクエスト変数が使えない事のチェック
	}

	/**
	 * 【TAG】すでに同名のファイルが存在した場合に、バックアップ処理するかどうか[true/false/rename]を指定します(初期値:null=false)。
	 *
	 * @og.tag
	 * ファイルアップロード時に、アップロード先に、同名のファイルが存在した場合は、既存機能は、そのまま
	 * 置き換えていましたが、簡易バージョンアップ機能として、useBackup="true" を指定すると、既存のファイルを
	 * リネームして、バックアップファイルを作成します。
	 * バックアップファイルは、アップロードフォルダを基準として、_backup/ファイル名.拡張子_処理時刻のlong値.拡張子 になります。
	 * オリジナルのファイル名（拡張子付）を残したまま、"_処理時刻のlong値" を追加し、さらに、オリジナルの拡張子を追加します。
	 * バックアップファイルの形式は指定できません。
	 * 
	 * 初期値は、互換性を持たせるため、null(=false) です。
	 * 
	 * 6.0.2.4 (2014/10/17)
	 * useBackup="rename" で、すでに同名のファイルが存在した場合に、"_001" のような文字列を追加したファイルにリネームします。
	 * Windowsの " - コピー (2)" に近いですが、桁数を抑えるのと、useBackup="true" と異なり、過去の同一ファイル名は
	 * そのまま、有効になります。同一ファイルが同一フォルダに存在する場合のみ連番が付与されます。
	 *
	 * @og.rev 5.6.5.3 (2013/06/28) 新規作成
	 * @og.rev 5.7.4.3 (2014/03/28) リクエスト変数が使えない事のチェックを行う。
	 * @og.rev 6.0.2.4 (2014/10/17) true/false 以外に、rename も有効とする。
	 *
	 * @param   flag バックアップ処理可否 [true:する/false:しない]
	 */
	public void setUseBackup( final String flag ) {
		useBackup = nval( getRequestParameter( flag ),useBackup );

		// 6.0.2.4 (2014/10/17) パラメータチェックをしておきます。
		if( useBackup != null && "true/false/rename".indexOf( useBackup.toLowerCase(Locale.JAPAN) ) < 0 ) {
			final String errMsg = "useBackup 属性は、[true/false/rename] から、指定してください。"
								+ " useBackup=[" + useBackup + "]" + CR ;
			throw new HybsSystemException( errMsg );
		}

		chckReqParam( flag,"useBackup" );		// 5.7.4.3 (2014/03/28) リクエスト変数が使えない事のチェック
	}

	/**
	 * リクエスト変数が使えない事のチェックを行います。
	 *
	 * upload では、enctype="multipart/form-data" のため、{&#064;XXXX}形式のパラメータが使えません。
	 * 5.7.4.3 (2014/03/28) から、filename のみ利用可能としたことで、同様に利用できると
	 * 勘違いするケースに対応する為、すべてのパラメータについてチェックを行います。
	 * ここでは、getRequestParameter( String ) の実行後、すぐに、isNull() 判定を行う事で、
	 * リクエスト変数の存在チェックを行う事にしています。
	 *
	 * @og.rev 5.7.4.3 (2014/03/28) リクエスト変数が使えない事のチェックを行う。
	 *
	 * @param   org 引数のオリジナル値
	 * @param   key エラーの発生した変数名
	 * @throws	HybsSystemException	パラメータが使用されていた場合
	 */
	private void chckReqParam( final String org,final String key ) {
		if( isNull() && org != null && org.contains( "{@" ) ) {
			final String errMsg = "upload では、enctype=\"multipart/form-data\" のため、{@XXXX}形式のパラメータが使えません。"
								+ CR
								+ "    " + key + "=[" + org + "]" ;
			throw new HybsSystemException( errMsg );		// リクエスト変数が使えないエラー
		}
	}

	/**
	 * タグの名称を、返します。
	 * 自分自身のクラス名より、自動的に取り出せないため、このメソッドをオーバーライドします。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規追加
	 *
	 * @return  タグの名称
	 * @og.rtnNotNull
	 */
	@Override
	protected String getTagName() {
		return "upload" ;
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "names"		,names		)
				.println( "ENCODE"		,ENCODE		)
				.println( "fileURL" 	,fileURL	)
				.println( "filename"    ,filename	)
				.println( "maxPostSize" ,maxPostSize)
				.println( "tableId" 	,tableId	)
				.println( "columns" 	,columns	)				// 5.6.5.2 (2013/06/21)
				.println( "useBackup" 	,useBackup	)				// 5.6.5.3 (2013/06/28)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
