/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.db.AbstractDBType;

/**
 * 情報機器事業部向け、品番情報の文字列を扱う為の、カラム属性を定義します。
 *
 * 品番情報は、３-５-３品番情報として扱われます。また、それぞれのフィールドにおいて、
 * 使用可能文字(例えば、Rev文字列の制限)などを考慮しています。
 *
 *   電気品番(1桁目が'D' 5桁目が'8'OR'9')は、一番最後の桁をﾘﾋﾞｼﾞｮﾝUP
 *   (ABCEFGHJKLMNPRSTUVWY)
 *       □□□-□□□□□-□□■
 *   機械品番(上記以外の品番)は、10桁目をﾘﾋﾞｼﾞｮﾝUPする。
 *       □□□-□□□□□-□■
 *   (0123456789ABCEFGHJKLMNPRSTUVWY)
 *
 * タイプチェックとして、以下の条件を判定します。
 * ・文字列長は、Byte換算での文字数との比較
 * ・半角文字列チェック「 c &lt; 0x20 || c &gt; 0x7e 以外」エラー
 * ・文字パラメータの 正規表現チェック
 * ・クロスサイトスクリプティングチェック
 *
 * @og.group データ属性
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DBType_PN extends AbstractDBType {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0.0 (2005/08/31)" ;

	private static final String DENKI_REV = "ABCEFGHJKLMNPRSTUVWY";
	private static final String KIKAI_REV = "0123456789ABCEFGHJKLMNPRSTUVWY";

	/**
	 * 製造品番のRev を UPした文字列を返します。
	 * 引数が null 、ゼロ文字列("")、の場合は,物理的初期設定値
	 * (String getDefault())の値を返します。
	 * 10桁未満、購入品(Z品番)、RevUPできない場合は,そのままの品番を返します。
	 *
	 * 但し、電気品番(1桁目が'D' 5桁目が'8'OR'9')は、一番最後の桁をﾘﾋﾞｼﾞｮﾝUP
	 *       □□□-□□□□□-□□■
	 *       機械品番(上記以外の品番)は、10桁目をﾘﾋﾞｼﾞｮﾝUPする。
	 *       □□□-□□□□□-□■
	 *
	 * @og.rev 2.1.3.0 (2002/12/12) RevUP 対象でないコードをRevUPする場合の不具合対応
	 *
	 * @param	value	String引数の文字列
	 *
	 * @return  String引数の文字列を＋１した文字列
	 */
	@Override
	public String valueAdd( final String value ) {
		if( value == null || value.isEmpty() ) { return getDefault(); }
		if( StringUtil.startsChar( value , 'Z' ) ) { return value; }	// 6.2.0.0 (2015/02/27) １文字 String.startsWith

		char[] ch = value.toCharArray();

		// 電気品番の場合
		if( ch[0] == 'D' && ( ch[4] == '8' || ch[4] == '9' ) ) {
			if( value.length() < 11 ) { return value; }				// 2002.07.12
			final int pos = DENKI_REV.indexOf( (int)ch[10] );
			if( pos >= 0 ) { ch[10] = DENKI_REV.charAt( pos+1 ); }
			else {           ch[10] ++ ;                         }	// 2.1.3.0 追加
		}
		else {
			final int pos = KIKAI_REV.indexOf( (int)ch[9] );
			if( pos >= 0 ) { ch[9] = KIKAI_REV.charAt( pos+1 ); }
			else {           ch[9] ++ ;                         }	// 2.1.3.0 追加
		}

		return new String( ch );
	}

	/**
	 * エディターで編集されたデータを登録する場合に、データそのものを
	 * 変換して、実登録データを作成します。
	 * データの表示用文字列を返します。
	 * XXX-XXXXX-XXX 形式で入力された情報を、XXXXXXXXXXX 形式で表示します。
	 * カット＆ペースト対策です。
	 *
	 * @og.rev 3.8.6.1 (2006/10/24) 新規追加
	 *
	 * @param	value	(一般に編集データとして登録されたデータ)
	 *
	 * @return  修正後の文字列(一般にデータベースに登録するデータ)
	 */
	@Override
	public String valueSet( final String value ) {
		String newVal = StringUtil.rTrim( value );

		if( newVal != null && newVal.indexOf( '-' ) >= 0 ) {
			newVal = StringUtil.replace( newVal,"-","" );
		}

		return newVal;
	}
}
