/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.daemon;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.fukurou.util.Shell;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.HybsTimerTask;
import java.io.File;
import java.util.Date;

/**
 * 【Shell実行】
 * 指定したパラメータでShellを実行します。
 * このクラスは、HybsTimerTask を継承した タイマータスククラスです。
 * startDaemon() がタイマータスクによって、呼び出されます。
 *
 * 接続のためのパラメータは以下です
 * fukurouのShellをキックするパラメータと同じです。
 * program	 : 動作プオグラム
 * workDir	 : 実行ディレクトリ
 * useBatch	 : BATCHプロセスを実行するのかどうか(初期値:false)
 * stdout 	 : 標準出を出力するかどうか(初期値:false)
 * stderr	 : エラー出力を出力するかどうか(初期値:false)
 * wait		 : プロセスの終了を待つ(true)/待たない(false) (初期値:true)
 *
 * @og.rev 5.6.9.1 (2013/10/11) 新規作成
 * @og.group デーモン
 *
 * @version  4.0
 * @author   Takahashi Masakazu
 * @since    JDK5.0,
 */
public class Daemon_RunShell extends HybsTimerTask {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.6.9.1 (2013/10/11)" ;

	private static final int LOOP_COUNTER = 24; // カウンタを24回に設定

	// 3.6.1.0 (2005/01/05) タイムアウト時間を設定
	private final int timeout	= HybsSystem.sysInt( "SHELL_TIMEOUT" );

	private int		loopCnt		;
	private String	program		;
	private boolean	useBatch	;		// BATCHプロセスを実行するのかどうか(初期値:false)
	private boolean	stdout		;		// 標準出を出力するかどうか(初期値:false)
	private boolean	stderr		;		// エラー出力を出力するかどうか(初期値:false)
	private boolean	wait		= true;	// プロセスの終了を待つ(true)/待たない(false) (初期値:true)
	private File	workDir		;
	private boolean	debug		;

	private Shell shell;

	/**
	 * このタイマータスクによって初期化されるアクションです。
	 * パラメータを使用した初期化を行います。
	 *
	 */
	@Override
	public void initDaemon() {
		debug = StringUtil.nval( getValue( "DEBUG" ),debug ) ;
		program		= StringUtil.nval( getValue( "program" ), program );
		useBatch	= StringUtil.nval( getValue( "useBatch" ), useBatch );
		stdout		= StringUtil.nval( getValue( "stdout" ), stdout );
		stderr		= StringUtil.nval( getValue( "stderr" ), stderr );
		wait  		= StringUtil.nval( getValue( "wait" ), wait   );
		if( getValue( "workDir" ) != null ){ workDir = new File(getValue( "workDir" ));  }
		shell = new Shell();
		shell.setCommand( program,useBatch );
		shell.setWait( wait );
		shell.setTimeout( timeout );		// 3.6.1.0 (2005/01/05)
		shell.setWorkDir( workDir );
		if(debug){System.out.println(program+"/"+useBatch+"/"+wait+"/"+timeout+"/"+workDir.toString());}
	}

	/**
	 * タイマータスクのデーモン処理の開始ポイントです。
	 *
	 */
	@Override
	protected void startDaemon() {
		if( loopCnt % LOOP_COUNTER == 0 ) {
			loopCnt = 1;
			System.out.println( toString() + " " + new Date()  + " " );
		}
		else {
			loopCnt++ ;
		}
		// 実行
		final int rtnCode = shell.exec();		// 0 は正常終了を示す
		if( rtnCode < 0 ){System.out.println( "Shell Run Error:" + program );}
	}
}
