/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.model;

import java.io.File;									// 6.2.0.0 (2015/02/27)
import java.io.IOException;
import java.io.OutputStream;
import java.io.FileOutputStream;
import java.io.BufferedOutputStream;
import java.util.Locale;
import java.util.Map;									// 6.0.2.3 (2014/10/10) 画像関連
import java.util.HashMap;								// 6.0.2.3 (2014/10/10) 画像関連

import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.Font;
import org.apache.poi.ss.usermodel.IndexedColors;
import org.apache.poi.ss.usermodel.RichTextString;
import org.apache.poi.ss.usermodel.Hyperlink;
import org.apache.poi.ss.usermodel.CreationHelper;
import org.apache.poi.ss.usermodel.Drawing;				// 6.0.2.3 (2014/10/10) 画像関連
import org.apache.poi.ss.usermodel.ClientAnchor;		// 6.0.2.3 (2014/10/10) 画像関連
import org.apache.poi.ss.usermodel.Picture;				// 6.0.2.3 (2014/10/10) 画像関連
import static org.opengion.fukurou.util.HybsConst.CR;				// 6.1.0.0 (2014/12/26) refactoring
import static org.opengion.fukurou.util.HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring

import org.apache.poi.ss.util.WorkbookUtil;

import org.apache.poi.hssf.usermodel.HSSFWorkbook;		// .xls
import org.apache.poi.xssf.usermodel.XSSFWorkbook;		// .xlsx

import org.apache.poi.POIXMLDocumentPart;				// 6.2.4.2 (2015/05/29) テキスト変換処理
import org.apache.poi.xssf.usermodel.XSSFDrawing;		// 6.2.4.2 (2015/05/29) テキスト変換処理
import org.apache.poi.xssf.usermodel.XSSFShape;			// 6.2.4.2 (2015/05/29) テキスト変換処理
import org.apache.poi.xssf.usermodel.XSSFSimpleShape;	// 6.2.4.2 (2015/05/29) テキスト変換処理
import org.apache.poi.xssf.usermodel.XSSFTextParagraph;	// 6.2.4.2 (2015/05/29) テキスト変換処理
import org.apache.poi.xssf.usermodel.XSSFTextRun;		// 6.2.4.2 (2015/05/29) テキスト変換処理

import org.opengion.fukurou.util.Closer;
import org.opengion.fukurou.util.ImageUtil;				// 6.0.2.3 (2014/10/10) 画像関連

/**
 * POI による、EXCELバイナリファイルに対する、データモデルクラスです。
 *
 * 共通的な EXCEL処理 を集約しています。
 * staticメソッドによる簡易的なアクセスの他に、順次処理も可能なように
 * 現在アクセス中の、Workbook、Sheet、Row、Cell オブジェクトを内部で管理しています。
 *
 * 入力形式は、openXML形式にも対応しています。
 * ファイルの内容に応じて、.xlsと.xlsxのどちらで読み取るかは、内部的に
 * 自動判定されます。
 *
 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
 * @og.group その他
 *
 * @version  6.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK7.0,
 */
public class ExcelModel {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.3.1.0 (2015/06/28)" ;

	private static final String DEF_SHEET_NAME = "Sheet" ;

	// 6.0.2.3 (2014/10/10) ImageUtil の Suffix と、Workbook.PICTURE_TYPE_*** の関連付けをしておきます。
	// Suffix 候補は、[bmp, gif, jpeg, jpg, png, wbmp] だが、対応する PICTURE_TYPE は一致しない。
	private static final Map<String,Integer> PICTURE_TYPE ;

	private final String inFilename ;		// エラー発生時のキーとなる、EXCELファイル名
	private final String sufix		;		// 6.1.0.0 (2014/12/26) オープンしたファイル形式を記憶(ピリオドを含む)

	private final Workbook	wkbook	;		// 現在処理中の Workbook
	private Sheet	 		sheet	;		// 現在処理中の Sheet
	private Row				rowObj	;		// 現在処理中の Row

	private int refSheetIdx	= -1;			// 雛形シートのインデックス

	private final CreationHelper createHelper	;	// poi.xssf対応

	private CellStyle style			;		// 共通のセルスタイル
	private CellStyle hLinkStyle	;		// Hyperlink用のセルスタイル(青文字＋下線)

	private int maxColCount			= 5 ;	// 標準セル幅の５倍を最大幅とする。
	private int dataStartRow		= -1;	// データ行の開始位置。未設定時は、-1
	private boolean isAutoCellSize	;		// カラム幅の自動調整を行うかどうか(true:行う/false:行わない)

	private String addTitleSheet	;		// Sheet一覧を先頭Sheetに作成する場合のSheet名

	static {
		PICTURE_TYPE = new HashMap<String,Integer>() ;
		PICTURE_TYPE.put( "png"  , Integer.valueOf( Workbook.PICTURE_TYPE_PNG   ) );
		PICTURE_TYPE.put( "jpeg" , Integer.valueOf( Workbook.PICTURE_TYPE_JPEG  ) );
		PICTURE_TYPE.put( "jpg"  , Integer.valueOf( Workbook.PICTURE_TYPE_JPEG  ) );
	}

	/**
	 * EXCELファイルのWookbookのデータ処理モデルを作成します。
	 * 
	 * ここでは、既存のファイルを読み込んで、データ処理モデルを作成しますので、
	 * ファイルがオープンできなければエラーになります。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 * @og.rev 6.2.0.0 (2015/02/27) ファイル引数を、String → File に変更
	 *
	 * @param   file  EXCELファイル
	 * @see		#ExcelModel( File , boolean )
	 */
	public ExcelModel( final File file ) {
		this( file,true );
	}

	/**
	 * EXCELファイルのWookbookのデータ処理モデルを作成します。
	 * 
	 * isOpen条件によって、ファイルオープン(true)か、新規作成(false)が分かれます。
	 * ファイルオープンの場合は、EXCELの読み込み以外に、追記するとか、雛形参照する
	 * 場合にも、使用します。
	 * ファイルオープンの場合は、当然、ファイルがオープンできなければエラーになります。
	 *
	 * isOpen=新規作成(false) の場合は、ファイル名の拡張子で、XSSFWorkbook か HSSFWorkbook を
	 * 判定します。.xlsx の場合⇒XSSFWorkbook オブジェクトを使用します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 * @og.rev 6.0.2.3 (2014/10/10) POIUtil#createWorkbook( String ) を使用するように変更
	 * @og.rev 6.1.0.0 (2014/12/26) 入力ファイルの拡張子判定の対応
	 * @og.rev 6.2.0.0 (2015/02/27) ファイル引数を、String → File に変更
	 * @og.rev 6.2.2.0 (2015/03/27) マクロ付Excel(.xlsm)対応
	 *
	 * @param   file   EXCELファイル
	 * @param   isOpen true:ファイルオープン/false:新規作成
	 * @see		#ExcelModel( File )
	 */
	public ExcelModel( final File file , final boolean isOpen ) {
		inFilename	= file.getName();

		final int idx = inFilename.lastIndexOf( '.' );	// 拡張子の位置
		if( idx >= 0 ) {
			sufix = inFilename.substring( idx ).toLowerCase( Locale.JAPAN );		// ピリオドを含む
		}
		else {
			final String errMsg = "ファイルの拡張子が見当たりません。(.xls か .xlsx/.xlsm を指定下さい)" + CR
							+ " filename=[" + file + "]"  + CR ;
			throw new IllegalArgumentException( errMsg );
		}

		if( isOpen ) {
			wkbook = POIUtil.createWorkbook( file );
		}
		else {
			// 新規の場合、ファイル名に.xlsxで終了した場合⇒.xlsx形式ファイル作成、その他⇒.xls形式ファイル作成
			if( ".xlsx".equals( sufix ) || ".xlsm".equals( sufix ) ) {		// 6.2.2.0 (2015/03/27)
				wkbook = new XSSFWorkbook();
	//			wkbook = new SXSSFWorkbook();	// 機能制限有：シートや行の削除や、AutoCellSize の指定ができないなど。
			}
			else if( ".xls".equals( sufix ) ) {
				wkbook = new HSSFWorkbook();
			}
			else {
				final String errMsg = "ファイルの拡張子が不正です。(.xls か .xlsx/.xlsm のみ可能)" + CR
								+ " filename=[" + file + "]"  + CR ;
				throw new IllegalArgumentException( errMsg );
			}
		}

		createHelper = wkbook.getCreationHelper();		// poi.xssf対応
	}

	/**
	 * 内部 Workbook に、フォント名、フォントサイズを設定します。
	 * fontName(フォント名)は、"ＭＳ Ｐゴシック" など名称になります。
	 * fontPoint は、フォントの大きさを指定します。
	 * 内部的には、setFontHeightInPoints(short)メソッドで設定します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	fontName	フォント名 ("ＭＳ Ｐゴシック" など。nullの場合セットしません)
	 * @param	fontPoint	フォントの大きさ (0やマイナスの場合はセットしません)
	 */
	public void setFont( final String fontName , final short fontPoint ) {
		if( style == null ) { style = wkbook.createCellStyle(); }

		final Font font = wkbook.createFont();
		if( fontName != null ) {
			font.setFontName( fontName );	// "ＭＳ Ｐゴシック" など
		}
		if( fontPoint > 0 ) {
			font.setFontHeightInPoints( fontPoint );
		}

		style.setFont( font );
	}

	/**
	 * データ設定する セルに、罫線を追加します。
	 *
	 * ここで設定するのは、罫線の種類と、罫線の色ですが、内部的に固定にしています。
	 *   Border=CellStyle.BORDER_THIN
	 *   BorderColor=IndexedColors.BLACK
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 */
	public void setCellStyle() {
		if( style == null ) { style = wkbook.createCellStyle(); }

		style.setBorderBottom(	CellStyle.BORDER_THIN );
		style.setBorderLeft(	CellStyle.BORDER_THIN );
		style.setBorderRight(	CellStyle.BORDER_THIN );
		style.setBorderTop(		CellStyle.BORDER_THIN );

		style.setBottomBorderColor(	IndexedColors.BLACK.getIndex() );
		style.setLeftBorderColor(	IndexedColors.BLACK.getIndex() );
		style.setRightBorderColor(	IndexedColors.BLACK.getIndex() );
		style.setTopBorderColor(	IndexedColors.BLACK.getIndex() );

		style.setVerticalAlignment( CellStyle.VERTICAL_TOP );	// isAutoCellSize=true 文字は上寄せする。
	//	style.setWrapText( true );								// isAutoCellSize=true 折り返して表示する。
	}

	/**
	 * 全てのSheetに対して、autoSizeColumn設定を行うかどうか指定します(初期値:false)。
	 *
	 * autoSize設定で、カラム幅が大きすぎる場合、現状では、
	 * 初期カラム幅の５倍を限度にしています。
	 *
	 * なお、autoSizeColumn設定は負荷の大きな処理なので、saveFile(String)の
	 * 中で実行されます。(セーブしなければ実行されません。)
	 * よって、指定は、いつ行っても構いません。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param flag autoSizeColumn設定を行うかどうか [true:自動カラム幅設定を行う/false:行わない]
	 * @see	#useAutoCellSize( boolean,int )
	 */
	public void useAutoCellSize( final boolean flag ) {
		isAutoCellSize = flag;
	}

	/**
	 * 全てのSheetに対して、autoSizeColumn設定を行うかどうか指定します(初期値:false)。
	 *
	 * autoSize設定で、カラム幅が大きすぎる場合、現状では、
	 * 初期カラム幅のcount倍を限度に設定します。
	 * ただし、count がマイナスの場合は、無制限になります。
	 *
	 * なお、autoSizeColumn設定は負荷の大きな処理なので、saveFile(String)の
	 * 中で実行されます。(セーブしなければ実行されません。)
	 * よって、指定は、いつ行っても構いません。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param flag autoSizeColumn設定を行うかどうか [true:自動カラム幅設定を行う/false:行わない]
	 * @param count 最大幅を標準セル幅の何倍にするかを指定。マイナスの場合は、無制限
	 * @see	#useAutoCellSize( boolean )
	 */
	public void useAutoCellSize( final boolean flag, final int count ) {
		isAutoCellSize = flag;
		maxColCount    = count ;
	}

	/**
	 * データ行の書き込み開始位置の行番号を設定します。
	 *
	 * これは、autoSize設定で、自動調整するカラムを、ヘッダーではなく、
	 * データ部で計算する場合に使用します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param st  データ行の開始位置。未設定時は、-1
	 * @see	#useAutoCellSize( boolean )
	 */
	public void setDataStartRow( final int st ) {
		dataStartRow = st;
	}

	/**
	 * Sheet一覧を先頭Sheetに作成する場合のSheet名を指定します。
	 *
	 * これは、Workbook に含まれる Sheet 一覧を作成する場合に、利用可能です。
	 *
	 * この処理は、#saveFile( File ) 処理時に、実行されます。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param sheetName  Sheet一覧のSheet名
	 * @see	#makeAddTitleSheet()
	 */
	public void setAddTitleSheet( final String sheetName ) {
		addTitleSheet = sheetName;
	}

	/**
	 * 内部 Workbookの Sheet数を返します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @return	シート数
	 */
	public int getNumberOfSheets() {
		return wkbook.getNumberOfSheets();
	}

	/**
	 * 内部 Workbookより、雛形Sheetをセットします。
	 * 
	 * これは、雛形シートを使用する場合に、使います。このメソッドが呼ばれると、
	 * 雛形シートを使用すると判定されます。
	 * 雛形シート名が、内部 Workbook に存在しない場合は、エラーになります。
	 * ただし、null をセットした場合は、最初のシートを雛形シートとして使用すると
	 * 判定します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	refSheetName	参照シート名(nullの場合、参照シート使用する場合は、先頭のシート)
	 */
	public void setRefSheetName( final String refSheetName ) {
		// 参照シート名の指定がない場合は、最初のシート
		refSheetIdx = ( refSheetName == null ) ? 0 : wkbook.getSheetIndex( refSheetName );

		if( refSheetIdx < 0 ) {		// 参照シート名が存在しなかった。
			final String errMsg = "指定の参照シート名は存在しませんでした。" + CR
							+ " inFilename=[" + inFilename + "] , refSheetName=[" + refSheetName + "]"  + CR ;
			throw new IllegalArgumentException( errMsg );
		}
	}

	/**
	 * 内部 Workbookより、新しいSheetを作ります。
	 *
	 * 先に雛形シートを指定している場合は、その雛形シートから作成します。
	 * 指定していない場合は、新しいシートを作成します。
	 * 雛形シートを参照する場合は、雛形シートそのものを返します。
	 * また、雛形シートの枚数を超える場合は、前の雛形シートをコピーします。
	 * 雛形シートが存在しない場合は、新しいシートを作成します。
	 * 
	 * シート名は、重複チェックを行い、同じ名前のシートの場合は、(1),(2)が付けられます。
	 * sheetName が null の場合は、"Sheet" が割り振られます。
	 *
	 * この処理を行うと、内部の Sheet にも、ここで作成された Sheet が設定されます。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 * @og.rev 6.2.2.3 (2015/04/10) 雛形シートにそのままデータを書き込んでいく。
	 *
	 * @param	sheetName	シート名 (重複する場合は、(2)、(3)のような文字列を追加 、nullの場合は、"Sheet")
	 */
	public void createSheet( final String sheetName ) {
		// 参照シートを使う場合(整合性の問題で、両方ともチェックしておきます)

		// 6.2.2.3 (2015/04/10) 雛形シートにそのままデータを書き込んでいく。
		final int shtNo ;
		if( refSheetIdx < 0 ) {									// 雛形シートを使用しない。
			sheet = wkbook.createSheet();
			shtNo = wkbook.getNumberOfSheets() - 1;
		}
		else if( refSheetIdx >= wkbook.getNumberOfSheets() ) {	// シート数が雛形より超えている。
			sheet = wkbook.cloneSheet( refSheetIdx-1 );			// 最後の雛形シートをコピーします。
			shtNo = wkbook.getNumberOfSheets() - 1;
			refSheetIdx++ ;
		}
		else {												
			sheet = wkbook.getSheetAt( refSheetIdx );			// 雛形シートをそのまま使用
			shtNo = refSheetIdx;
			refSheetIdx++ ;
		}

		setSheetName( shtNo , sheetName );

	}

	/**
	 * 内部 Workbook の指定のシート番号の Sheet の名前を設定します。
	 *
	 * 指定のシート名が、既存のシートになければ、そのまま設定します。
	 * すでに、同じ名前のシートが存在する場合は、そのシート名の後に
	 * (1)、(2)、(3)のような文字列を追加します。
	 * sheetName が null の場合は、"Sheet" が割り振られます。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 * @og.rev 6.2.5.1 (2015/06/12) シート名重複が自分自身の場合は、(1)等の追加は行わない。
	 *
	 * @param	shNo		シート番号
	 * @param	sheetName	シート名 (重複する場合は、(1)、(2)のような文字列を追加 、nullの場合は、"Sheet")
	 */
	public void setSheetName( final int shNo, final String sheetName ) {
		String tempName = ( sheetName == null ) ? DEF_SHEET_NAME : WorkbookUtil.createSafeSheetName( sheetName ) ;
		int cnt = 1;

		// 6.2.5.1 (2015/06/12) シート名重複が自分自身の場合は、(1)等の追加は行わない。
		// ※ EXCELのシート名は、大文字、小文字だけでなく、全角半角の区別もしない。
		final String shtName = wkbook.getSheetName( shNo );
		if( tempName != null && !tempName.equals( shtName ) ) {			// 全く同一の場合は、何もしない。
			if( shNo == wkbook.getSheetIndex( tempName ) ) {			// シート名判定が、自身の場合
				wkbook.setSheetName( shNo,tempName );
			}
			else {
				while( wkbook.getSheetIndex( tempName ) >= 0 ) {		// シート名が存在している場合
					tempName = WorkbookUtil.createSafeSheetName( sheetName + "(" + cnt + ")" );
					if( tempName.length() >= 31 ) {						// 重複時の追加文字分を減らす。
						tempName = tempName.substring( 0,26 ) + "(" + cnt + ")" ;	// cnt３桁まで可能
					}
					cnt++;
				}
				wkbook.setSheetName( shNo,tempName );
			}
		}
	}

	/**
	 * 内部 Workbook の 指定のSheet番号のシート名前を返します。
	 *
	 * シートが存在しない場合は、null を返します。
	 *
	 * この処理を行うと、内部の Sheet にも、ここで見つけた Sheet が設定されます。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	 shNo		シート番号
	 *
	 * @return	sheetName	シート名
	 */
	public String getSheetName( final int shNo ) {
		final int shLen = wkbook.getNumberOfSheets();

		String shName = null;
		if( shNo < shLen ) {
			sheet = wkbook.getSheetAt( shNo );		// 現在の sheet に設定する。
			shName = sheet.getSheetName();
		}

		return shName ;
	}

	/**
	 * 内部 Workbook の 指定のSheet名のシート番号を返します。
	 *
	 * シートが存在しない場合は、-1 を返します。
	 * この処理を行うと、内部の Sheet にも、ここで見つけた Sheet が設定されます。
	 * シートが存在しない場合、内部の Sheet オブジェクトも null がセットされますのでご注意ください。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	 shName		シート名
	 *
	 * @return	シート番号(名前のシートがなければ、-1)
	 */
	public int getSheetNo( final String shName ) {
		sheet = wkbook.getSheet( shName );				// シート名がマッチしなければ、null

		return wkbook.getSheetIndex( shName ) ;			// シート名がマッチしなければ、-1
	}

	/**
	 * Excelの指定Sheetオブジェクトを削除します。
	 *
	 * 削除するシートは、シート番号でFrom-To形式で指定します。
	 * Fromも Toも、削除するシート番号を含みます。
	 * 例えば、0,3 と指定すると、0,1,2,3 の ４シート分を削除します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	 fromNo		削除する開始シート番号(含む)
	 * @param	 toNo		削除する終了シート番号(含む)
	 */
	public void removeSheet( final int fromNo,final int toNo ) {
		for( int shtNo=toNo; shtNo>=fromNo; shtNo-- ) {			// 逆順に処理します。
			wkbook.removeSheetAt( shtNo );
		}
	}

	/**
	 * 内部 Workbookの 現在Sheet の最初の行番号を返します。
	 *
	 * 行は、0 から始まります。
	 * この処理は、内部Sheetが作成されているか、null でない場合のみ実行できます。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @return	最初の行番号
	 */
	public int getFirstRowNum() {
		return sheet.getFirstRowNum();
	}

	/**
	 * 内部 Workbookの 現在Sheet の最後の行番号を返します。
	 *
	 * 最終行は、含みます。よって、行数は、getLastRowNum()＋１になります。
	 * この処理は、内部Sheetが作成されているか、null でない場合のみ実行できます。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @return	最後の行番号
	 */
	public int getLastRowNum() {
		return sheet.getLastRowNum();
	}

	/**
	 * Excelの指定行のRowオブジェクトを作成します。
	 *
	 * 指定行の Row オブジェクトが存在しない場合は、新規作成します。
	 * この処理を実行すると、指定行の Rowオブジェクトが内部 Row に設定されます。
	 * この処理は、内部Sheetが作成されているか、null でない場合のみ実行できます。
	 *
	 * この処理を行うと、内部の Rowオブジェクトが設定されます。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	 rowNo		行の番号
	 */
	public void createRow( final int rowNo ) {
		rowObj = sheet.getRow( rowNo );
		if( rowObj == null ) { rowObj = sheet.createRow( rowNo ); }
	}

	/**
	 * Excelの指定行以降の余計なRowオブジェクトを削除します。
	 *
	 * 指定行の Row オブジェクトから、getLastRowNum() までの行を、削除します。
	 * この処理は、内部Sheetが作成されているか、null でない場合のみ実行できます。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	 startRowNum		指定以降の余計な行を削除
	 */
	public void removeRow( final int startRowNum ) {
		final int stR = startRowNum;
		final int edR = sheet.getLastRowNum();

		for( int rowNo=edR; rowNo>=stR && rowNo>=0; rowNo-- ) {			// 逆順に処理します。
			final Row rowObj = sheet.getRow( rowNo );
			if( rowObj != null ) { sheet.removeRow( rowObj ); }
		}
	}

	/**
	 * Excelの処理中のRowオブジェクトの指定カラム以降の余計なCellオブジェクトを削除します。
	 *
	 * 指定行の Row オブジェクトから、getLastCellNum() までのカラムを、削除します。
	 * この処理は、内部Rowが作成されているか、null でない場合のみ実行できます。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	 startCellNum		指定以降の余計なカラムを削除
	 */
	public void removeCell( final int startCellNum ) {
		final int stC = startCellNum;
		final int edC = rowObj.getLastCellNum();

		for( int colNo=edC; colNo>=stC; colNo-- ) {			// 逆順に処理します。
			final Cell colObj = rowObj.getCell( colNo );
			if( colObj != null ) { rowObj.removeCell( colObj ); }
		}
	}

	/**
	 * row にあるセルのオブジェクト値を設定します。
	 *
	 * 行が存在しない場合、行を追加します。
	 * この処理を行うと、内部の Rowオブジェクトがなければ新規作成されます。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param   vals  新しい配列値。
	 * @param   rowNo   値が変更される行(無視されます)
	 */
	public void setValues( final String[] vals,final int rowNo ) {
		if( rowObj == null ) { createRow( rowNo ); }

		if( vals != null ) {
			for( int colNo=0; colNo<vals.length; colNo++ ) {
				setCellValue( vals[colNo],colNo );
			}
		}
	}

	/**
	 * row にあるセルのオブジェクト値を設定します。
	 *
	 * 行が存在しない場合、行を追加します。
	 * 引数に、カラムがNUMBER型かどうかを指定することが出来ます。
	 * この処理を行うと、内部の Rowオブジェクトがなければ新規作成されます。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param   vals  新しい配列値。
	 * @param   rowNo   値が変更される行(無視されます)
	 * @param	isNums	セルが、NUMBER型の場合は、true/それ以外は、false
	 */
	public void setValues( final String[] vals,final int rowNo,final boolean[] isNums ) {
		if( rowObj == null ) { createRow( rowNo ); }

		if( vals != null ) {
			for( int colNo=0; colNo<vals.length; colNo++ ) {
				setCellValue( vals[colNo],colNo,isNums[colNo] );
			}
		}
	}

	/**
	 * Excelの指定セルにデータを設定します。
	 *
	 * ここで設定する行は、現在の内部 Row です。
	 * Row を切り替えたい場合は、#createRow( int ) を呼び出してください。
	 * このメソッドでは、データを文字列型として設定します。
	 * この処理は、内部Rowが作成されているか、null でない場合のみ実行できます。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	dataVal    String文字列
	 * @param	colNo		セルの番号(0,1,2････)
	 * @see		#setCellValue( String,int,boolean )
	 */
	public void setCellValue( final String dataVal , final int colNo ) {
		setCellValue( dataVal,colNo,false );
	}

	/**
	 * Excelの指定セルにデータを設定します。
	 *
	 * ここで設定する行は、現在の内部 Row です。
	 * Row を切り替えたい場合は、#createRow( int ) を呼び出してください。
	 * このメソッドでは、引数のデータ型をNUMBER型の場合は、doubleに変換して、
	 * それ以外は文字列としてとして設定します。
	 * この処理は、内部Rowが作成されているか、null でない場合のみ実行できます。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	dataVal		String文字列
	 * @param	colNo		セルの番号(0,1,2････)
	 * @param	isNumber	セルが、NUMBER型の場合は、true/それ以外は、false
	 * @see		#createRow( int )
	 * @see		#setCellValue( String,int )
	 */
	public void setCellValue( final String dataVal , final int colNo , final boolean isNumber ) {
		Cell colObj = rowObj.getCell( colNo );
		if( colObj == null ) { colObj = rowObj.createCell( colNo ); }

		if( style != null ) { colObj.setCellStyle(style); }

		// CELL_TYPE_NUMERIC 以外は、String扱いします。
		if( isNumber ) {
			final Double dbl = parseDouble( dataVal );
			if( dbl != null ) {
				colObj.setCellValue( dbl.doubleValue() );
				return ;		// Double 変換できた場合は、即抜けます。
			}
		}

		final RichTextString richText = createHelper.createRichTextString( dataVal );
		colObj.setCellValue( richText );
	}

	/**
	 * Excelの指定セルにHyperlinkを設定します。
	 *
	 * ここで設定する行は、現在の内部 Row です。
	 * Row を切り替えたい場合は、#createRow( int ) を呼び出してください。
	 * このメソッドで設定するHyperlinkは、Sheetに対する LINK_DOCUMENT です。
	 * 先に、セルに対する値をセットしておいてください。
	 * Hyperlinkは、文字に対して、下線 と 青字 のスタイル設定を行います。
	 *
	 * Link文字列(シート名) が、null や ゼロ文字列の場合は、処理を行いません。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	linkVal		Link文字列(シート名)
	 * @param	colNo		セルの番号(0,1,2････)
	 * @see		#setCellValue( String,int )
	 */
	public void setCellLink( final String linkVal , final int colNo ) {
		if( linkVal == null || linkVal.isEmpty() ) { return; }

		Cell colObj = rowObj.getCell( colNo );
		if( colObj == null ) { colObj = rowObj.createCell( colNo ); }

		if( hLinkStyle == null ) {
			hLinkStyle = wkbook.createCellStyle();
			if( style != null ) { hLinkStyle.cloneStyleFrom(style); }

			final Font font = wkbook.createFont();
			font.setColor( IndexedColors.BLUE.getIndex() );		// リンクは青文字
			font.setUnderline( Font.U_SINGLE );					// 下線付

			hLinkStyle.setFont( font );
		}
		colObj.setCellStyle(hLinkStyle);

		final Hyperlink hLink = createHelper.createHyperlink( Hyperlink.LINK_DOCUMENT );
		hLink.setAddress( "'" + linkVal + "'!A1" );
		colObj.setHyperlink( hLink );
	}

	/**
	 * 現在のRow にあるセルの属性値を配列で返します。
	 *
	 * Rowオブジェクトが存在しない場合は、null を返します。
	 * また、Rowオブジェクトの中の セルオブジェクトが存在しない場合は、
	 * null がセットされます。
	 *
	 * この処理は、内部Sheetが作成されているか、null でない場合のみ実行できます。
	 * この処理を実行すると、指定行の Rowオブジェクトが内部 Row に設定されます。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	rowNo		行の番号
	 * @return	指定されたセルの属性値。Rowがnullの場合は、nullを返します。
	 */
	public String[] getValues( final int rowNo ) {
		rowObj = sheet.getRow( rowNo );
		if( rowObj == null ) { return null; }

		final int len = rowObj.getLastCellNum();			// 含まないので、length と同じ意味になる。
		String[] vals = new String[len];

		for( int colNo=0; colNo<len; colNo++ ) {
			final Cell colObj = rowObj.getCell( colNo );
			vals[colNo] = POIUtil.getValue( colObj );
		}

		return vals ;
	}

	/**
	 * 現在のrow にあるセルの属性値を返します。
	 *
	 * セルオブジェクトが存在しない場合は、null を返します。
	 *
	 * この処理は、内部Sheetが作成されているか、null でない場合のみ実行できます。
	 * この処理を実行すると、指定行の Rowオブジェクトが内部 Row に設定されます。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param   rowNo     値が参照される行
	 * @param   colNo     値が参照される列
	 *
	 * @return  指定されたセルの値 T
	 */
	public String getValue( final int rowNo, final int colNo ) {
		rowObj = sheet.getRow( rowNo );
		if( rowObj == null ) { return null; }

		final Cell colObj = rowObj.getCell( colNo );
		return POIUtil.getValue( colObj );
	}

	/**
	 * 指定のシートの行・列の箇所に、イメージファイルを挿入します。
	 *
	 * ここでは、セル範囲ではなく、指定の行列の箇所に、アンカーを設定して、画像ファイルを
	 * 挿入します。一応、リサイズして、元の大きさ近くに戻しますが、縦横比が変わってしまいます。
	 * 正確に挿入する場合は、セル範囲の指定と、マージンを指定しなければなりませんが、
	 * 微調整が必要です。
	 *
	 * この処理で使用される Sheetオブジェクトは一時的に作成されます。(キャッシュされません)
	 * 一連処理のどのタイミングで実行しても、内部の状態には影響はありません。
	 *
	 * @og.rev 6.0.2.3 (2014/10/10) 新規作成
	 *
	 * @param   imgFile   挿入するイメージファイル名
	 * @param   shtNo     シート番号
	 * @param   rowNo     挿入する行
	 * @param   colNo     挿入する列
	 */
	public void addImageFile( final String imgFile, final int shtNo, final int rowNo, final int colNo ) {
		addImageFile( imgFile,shtNo,rowNo,colNo,rowNo,colNo,0,0,0,0 );
	}

	/**
	 * 指定のシートの行・列の箇所に、イメージファイルを挿入します。
	 *
	 * ここでは、セル範囲ではなく、指定の行列の箇所に、アンカーを設定して、画像ファイルを
	 * 挿入します。一応、リサイズして、元の大きさ近くに戻しますが、縦横比が変わってしまいます。
	 * 正確に挿入する場合は、セル範囲の指定と、マージンを指定しなければなりませんが、
	 * 微調整が必要です。
	 *
	 * この処理で使用される Sheetオブジェクトは一時的に作成されます。(キャッシュされません)
	 * 一連処理のどのタイミングで実行しても、内部の状態には影響はありません。
	 *
	 * @og.rev 6.0.2.3 (2014/10/10) 新規作成
	 *
	 * @param   imgFile   挿入するイメージファイル名
	 * @param   shtNo     シート番号
	 * @param   row1      挿入する行(開始)
	 * @param   col1      挿入する列(開始)
	 * @param   row2      挿入する行(終了-含まず)
	 * @param   col2      挿入する列(終了-含まず)
	 * @param   dx1       開始セルのX軸座標(マージン)
	 * @param   dy1       開始セルのY軸座標(マージン)
	 * @param   dx2       終了セルのX軸座標(マージン)
	 * @param   dy2       終了セルのY軸座標(マージン)
	 */
	public void addImageFile( final String imgFile , final int shtNo ,
								final int row1 , final int col1 , final int row2 , final int col2 ,
								final int dx1  , final int dy1  , final int dx2  , final int dy2   ) {
		final String suffix   = ImageUtil.getSuffix( imgFile );
		final Integer picType = PICTURE_TYPE.get( suffix );

		// 実験した結果、bmp,gif,tif については、PICTURE_TYPE_PNG で、挿入できた。
		final int pictureType = picType != null ? picType.intValue() : Workbook.PICTURE_TYPE_PNG ;

		final byte[] imgs = ImageUtil.byteImage( imgFile );

		final int pictureIdx = wkbook.addPicture( imgs, pictureType );

		final Sheet sheet = wkbook.getSheetAt( shtNo );
		final Drawing patriarch = sheet.createDrawingPatriarch();		// 昔は一度しか実行できなかったようです。

		final ClientAnchor anchor = patriarch.createAnchor( dx1,dy1,dx2,dy2,col1,row1,col2,row2 );
		// ClientAnchor anchor = createHelper.createClientAnchor();	でも作成可能。

		// MOVE_AND_RESIZE, MOVE_DONT_RESIZE, DONT_MOVE_AND_RESIZE から、決め打ち。
		anchor.setAnchorType( ClientAnchor.MOVE_DONT_RESIZE );

		final Picture pic = patriarch.createPicture( anchor, pictureIdx );
		// セルの範囲指定がゼロの場合、画像サイズもゼロになる為、リサイズしておく。
		if( row1 == row2 || col1 == col2 ) { pic.resize(); }	// resize すると、anchor のマージンが無視されるようです。
	}

	/**
	 * 内部 Workbook オブジェクトをファイルに書き出します。
	 *
	 * Excelの形式は、ここで指定する出力ファイルの拡張子ではなく、コンストラクタで
	 * 指定したファイルの拡張子で決まります。
	 * 異なる形式の拡張子を持つファイルを指定した場合、強制的に、オープンした 
	 * Workbook の形式の拡張子を追加します。
	 *
	 * 拡張子は、Excel 2007以降の形式(.xlsx)か、Excel 2003以前の形式(.xls) が指定できます。
	 * 拡張子が未設定の場合は、オープンした Workbook の形式に合わせた拡張子を付与します。
	 *
	 * isAutoCellSize=true の場合は、ここで全Sheetに対してCell幅の自動調整が行われます。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 * @og.rev 6.1.0.0 (2014/12/26) 入力ファイルの拡張子判定の対応
	 * @og.rev 6.2.0.0 (2015/02/27) ファイル引数を、String → File に変更
	 *
	 * @param	file	セーブするファイル
	 */
	public void saveFile( final File file ) {
		final File   saveFile ;
		String fname = file.getName();
		if( fname.toLowerCase(Locale.JAPAN).endsWith( sufix ) ) {
			saveFile = file;
		}
		else {
			final int idx = fname.lastIndexOf( '.' );
			if( idx >= 0 ) { fname = fname.substring( 0,idx ); }
			saveFile = new File( file.getParent() , fname + sufix );
		}

		if( isAutoCellSize ) { POIUtil.autoCellSize( wkbook, maxColCount, dataStartRow ); }

		// こちらの都合で、TitleSheet は、autoCellSize ではなく、Sheet#autoSizeColumn(int) を使用して、自動計算させる。
		if( addTitleSheet != null ) { makeAddTitleSheet(); }

		OutputStream fileOut = null ;
		try {
			fileOut = new BufferedOutputStream( new FileOutputStream( saveFile ) );		// 6.1.0.0 (2014/12/26)
			wkbook.write( fileOut );
		}
		catch( IOException ex ) {
			final String errMsg = "ファイルへ書込み中にエラーが発生しました。" + CR
							+ "  File=" + saveFile + CR
							+ ex.getMessage() ;
			throw new RuntimeException( errMsg,ex );
		}
		finally {
			Closer.ioClose( fileOut );
		}
	}

	/**
	 * 内部 Workbook オブジェクトのSheet一覧のSheetを、先頭に追加します。
	 *
	 * これは、Workbook に含まれる Sheet 一覧を作成する場合に、利用可能です。
	 *
	 * この処理は、内部のWorkbook、Sheetオブジェクトに依存して実行されます。
	 * また、単独ではなく、#saveFile( File ) 実行時に、addTitleSheet が
	 * 設定されている場合のみ、実行されます。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @see		#saveFile( File )
	 * @see		#setAddTitleSheet( String )
	 */
	private void makeAddTitleSheet() {
		sheet = wkbook.createSheet();
		final String shtNm = sheet.getSheetName();				// Sheet名の取得
		wkbook.setSheetOrder( shtNm,0 );					// そのSheetを先頭に移動
		setSheetName( 0,addTitleSheet );					// そのSheet名を変更 → これが、TitleSheet

		int rowNo = 0;
		createRow( rowNo++ );								// 先頭行(インスタンス共通のRowオブジェクト)作成
		setCellValue( "No"	 , 0 );
		setCellValue( "Sheet", 1 );

		final int shCnt = wkbook.getNumberOfSheets();
		for( int shNo=1; shNo<shCnt; shNo++,rowNo++ ) {
			final String nm = wkbook.getSheetName( shNo );

			createRow( rowNo );									// 行の追加作成
			setCellValue( String.valueOf( rowNo ),0,true );		// 行番号として、数字型で登録
			setCellValue( nm , 1 );								// シートの値を書き込む
			setCellLink(  nm , 1 );								// シートへのリンクを作成する。
		}

		sheet.autoSizeColumn( 0 );
		sheet.autoSizeColumn( 1 );
	}

	/**
	 * Workbook の全Sheetを対象に、空行を取り除き、全体をシュリンクします。
	 *
	 * この処理は、#saveFile( File ) の直前に行うのがよいでしょう。
	 *
	 * ここでは、Row を逆順にスキャンし、Cellが 存在しない間は、行を削除します。
	 * 途中の空行の削除ではなく、最終行からの連続した空行の削除です。
	 * 
	 * isCellDel=true を指定すると、Cellの末尾削除を行います。
	 * 有効行の最後のCellから空セルを削除していきます。
	 * 表形式などの場合は、Cellのあるなしで、レイアウトが崩れる場合がありますので
	 * 処理が不要な場合は、isCellDel=false を指定してください。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	isCellDel	Cellの末尾削除を行うかどうか(true:行う/false:行わない)
	 */
	public void activeWorkbook( final boolean isCellDel ) {
		POIUtil.activeWorkbook( wkbook, isCellDel );
	}

	/**
	 * Workbook の全Sheetを対象に、テキスト変換処理を行います(XSLX限定)。
	 *
	 * この処理は、#saveFile( File ) の直前に行うのがよいでしょう。
	 * #activeWorkbook( boolean ) との順番は構いません。
	 *
	 * ・シート名の一覧をピックアップします。
	 * ・セル値を、セル単位にピックアップします。
	 * ・オブジェクト文字列を、改行単位にピックアップし、結果を合成します。
	 *
	 * ここでは、内部的に、TextConverterインターフェースを作成して処理します。
	 *
	 * @og.rev 6.2.4.2 (2015/05/29) テキスト変換処理
	 * @og.rev 6.3.1.0 (2015/06/28) TextConverterに、引数(cmnt)を追加
	 *
	 * @param	convMap	変換対象を管理するMapオブジェクト
	 * @see		#textConverter( TextConverter )
	 */
	public void textConverter( final Map<String,String> convMap ) {
	//	textConverter(
	//		( val,cmnt ) -> convMap.get( val )
	//	);

		textConverter(
			new TextConverter<String,String>() {
				public String change( final String val , final String cmnt ) {
					return convMap.get( val );
				}
			}
		);
	}

	/**
	 * Workbook の全Sheetを対象に、テキスト変換処理を行います(XSLX限定)。
	 *
	 * この処理は、#saveFile( File ) の直前に行うのがよいでしょう。
	 * #activeWorkbook( boolean ) との順番は構いません。
	 *
	 * ・シート名の一覧をピックアップします。
	 * ・セル値を、セル単位内の改行単位にピックアップし、結果を合成ます。
	 * ・オブジェクト文字列を、改行単位にピックアップし、結果を合成します。
	 *
	 * ここでは、シート名、セルテキスト、SimpleShapeオブジェクトのテキストを
	 * input に、TextConverterインターフェース の change メソッドを呼び出します。
	 * 戻り値が、null でないなら、元のデータと置き換えます。
	 * 戻り値が、null の場合は、そのまま読み飛ばします。(なにもしません)
	 * EXCELへの書き戻しが発生しますので、万一、ファイル破損で、開けなくなる場合を
	 * 想定して、バックアップファイルは、各自で準備してください。
	 *
	 * @og.rev 6.2.4.2 (2015/05/29) テキスト変換処理
	 * @og.rev 6.2.5.0 (2015/06/05) xsl形式のオブジェクト取得…はできなかった。
	 * @og.rev 6.3.1.0 (2015/06/28) TextConverterに、引数(cmnt)を追加
	 *
	 * @param	conv	TextConverterインターフェース
	 * @see		#textConverter( Map )
	 */
	public void textConverter( final TextConverter<String,String> conv ) {
	//	if( ".xlsx".equals( sufix ) || ".xlsm".equals( sufix ) ) {
			final int shCnt = wkbook.getNumberOfSheets();
			for( int shNo=0; shNo<shCnt; shNo++ ) {
				final Sheet sht = wkbook.getSheetAt( shNo );
				// シート名の変換
//				final String shtNm = conv.change( sht.getSheetName() );
				final String shtNm = conv.change( sht.getSheetName() , "Sheet" );
				if( shtNm != null ) {
					setSheetName( shNo,shtNm );			// 同一シート対策済みのメソッドを呼び出す。
				}

				// セル値の変換
				final int stR = Math.max( sht.getFirstRowNum(),0 );		// stR が、マイナスのケースがある。
				final int edR = sht.getLastRowNum();

				for( int rowNo=stR; rowNo<=edR; rowNo++ ) {
					final Row rowObj = sht.getRow( rowNo );
					if( rowObj != null ) {
						final int stC = Math.max( rowObj.getFirstCellNum(),0 );		// stC が、マイナスのケースがある。
						final int edC = rowObj.getLastCellNum();
						for( int colNo=stC; colNo<=edC; colNo++ ) {
							final Cell colObj = rowObj.getCell( colNo );
							if( colObj != null && colObj.getCellType() != Cell.CELL_TYPE_BLANK ) {
								final String cmnt= "Sht" + shNo + ":" + POIUtil.getCelKigo( rowNo,colNo );
								final String val = crConv( conv, POIUtil.getValue( colObj ),cmnt );		// 改行対応
								if( val != null ) { 
									colObj.setCellValue( val );
								}
							}
						}
					}
				}

				// オブジェクト文字列の変換
				if( sht instanceof POIXMLDocumentPart ) {
					for( final POIXMLDocumentPart pxdp : ((POIXMLDocumentPart)sht).getRelations() ) {
						if( pxdp instanceof XSSFDrawing ) {
							for( final XSSFShape shape : ((XSSFDrawing)pxdp).getShapes() ) {
								final org.apache.poi.xssf.usermodel.XSSFAnchor anc = shape.getAnchor();
								final String ancSt = "XY(" + anc.getDx1() + "-" + anc.getDy1() + ")" ;
								int cnt = 0;
								if( shape instanceof XSSFSimpleShape ) {
									for( final XSSFTextParagraph para : ((XSSFSimpleShape)shape).getTextParagraphs() ) {
										for( final XSSFTextRun text : para.getTextRuns() ) {
											final String cmnt= "Sht" + shNo + ":" + ancSt + ":Tb(" + cnt++ + ")" ;
											final String val = crConv( conv,text.getText() , cmnt );
											if( val != null ) {
												text.setText( val );
											}
										}
									}
								}
							}
						}
					}
				}
			// 6.2.5.0 (2015/06/05) xsl形式のオブジェクト取得…はできなかった。
			//	else if( sht instanceof HSSFSheet ) {
			//		HSSFPatriarch patri = ((HSSFSheet)sht).getDrawingPatriarch();
			//		for( final HSSFShape shape : patri.getChildren() ) {
			//			if( shape instanceof HSSFTextbox ) {
			//				HSSFRichTextString rts = ((HSSFSimpleShape)shape).getString();
			//				if( rts != null ) {
			//					final String val = crConv( conv,rts.getString() );
			//					if( val != null ) {
			//						HSSFRichTextString rts2 = new HSSFRichTextString( val );
			//						((HSSFSimpleShape)shape).setString( rts2 );
			//					}
			//				}
			//			}
			//		}
			//	}
			}
	//	}
	}

	/**
	 * Workbook の全Sheetを対象に、テキスト変換処理を行います(XSLX限定)。
	 *
	 * この処理は、#saveFile( File ) の直前に行うのがよいでしょう。
	 * #activeWorkbook( boolean ) との順番は構いません。
	 *
	 * ・シート名の一覧をピックアップします。
	 * ・セル値を、セル単位内の改行単位にピックアップし、結果を合成ます。
	 * ・オブジェクト文字列を、改行単位にピックアップし、結果を合成します。
	 *
	 * ここでは、シート名、セルテキスト、SimpleShapeオブジェクトのテキストを
	 * input に、TextConverterインターフェース の change メソッドを呼び出します。
	 * 戻り値が、null でないなら、元のデータと置き換えます。
	 * 戻り値が、null の場合は、そのまま読み飛ばします。(なにもしません)
	 * EXCELへの書き戻しが発生しますので、万一、ファイル破損で、開けなくなる場合を
	 * 想定して、バックアップファイルは、各自で準備してください。
	 *
	 * @og.rev 6.2.4.2 (2015/05/29) テキスト変換処理
	 * @og.rev 6.3.1.0 (2015/06/28) TextConverterに、引数(cmnt)を追加
	 *
	 * @param	conv	TextConverterインターフェース
	 * @param	val		改行処理を行う元の値
	 * @param	cmnt	コメント
	 * @return	改行処理の結果の値(対象が無ければ、null)
	 * @see		#textConverter( Map )
	 */
//	private String crConv( final TextConverter conv , final String val ) {
	private String crConv( final TextConverter<String,String> conv , final String val , final String cmnt ) {
		String rtn = null;
		if( val != null ) {
			if( val.contains( "\n" ) ) {						// 改行がある場合(EXCEL のセル内改行コードは、LF(0A)=\n のみ。
				final String[] val2 = val.split( "\\n" );		// 改行で分割する。
				boolean flag = false;
				for( int i=0; i<val2.length; i++ ) {
//					final String val3 = conv.change( val2[i] );
					final String val3 = conv.change( val2[i],cmnt );	// 6.3.1.0 (2015/06/28)
					if( val3 != null ) { val2[i] = val3; flag = true; }
				}
				if( flag ) {
					final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
					buf.append( val2[0] );
					for( int i=1; i<val2.length; i++ ) {
						buf.append( '\n' ).append( val2[i] );		// LF(\n)で、セパレートしているので、LF のみ追加する。
					}
					rtn = buf.toString();
				}
			}
			else {												// 改行がない場合
//				rtn = conv.change( val );
				rtn = conv.change( val,cmnt );					// 6.3.1.0 (2015/06/28)
			}
		}
		return rtn;
	}

	/**
	 * シート一覧を、内部の Workbook から取得します。
	 *
	 * 取得元が、Workbook なので、xls , xlsx どちらの形式でも取り出せます。
	 *
	 * EXCEL上のシート名を、配列で返します。
	 *
	 * @og.rev 6.2.6.0 (2015/06/19) 新規作成
	 *
	 * @return	シート名の配列
	 * @see		POIUtil#getSheetNames( Workbook )
	 */
	public String[] getSheetNames() {
		return POIUtil.getSheetNames( wkbook );
	}

	/**
	 * 名前定義一覧を内部の Workbook から取得します。
	 *
	 * EXCEL上に定義された名前を、配列で返します。
	 * ここでは、名前とFormulaをタブで連結した文字列を配列で返します。
	 * Name オブジェクトを削除すると、EXCELが開かなくなったりするので、
	 * 取りあえず一覧を作成して、手動で削除してください。
	 * なお、名前定義には、非表示というのがありますので、ご注意ください。
	 *
	 * @og.rev 6.2.6.0 (2015/06/19) 新規作成
	 *
	 * @return	名前定義(名前+TAB+Formula)の配列
	 * @see		POIUtil#getNames( Workbook )
	 * @og.rtnNotNull
	 */
	public String[] getNames() {
		return POIUtil.getNames( wkbook );
	}

	/**
	 * 書式のスタイル一覧を内部の Workbook から取得します。
	 *
	 * EXCEL上に定義された書式のスタイルを、配列で返します。
	 * 書式のスタイルの名称は、CellStyle にメソッドが定義されていません。
	 * 実クラスである HSSFCellStyle にキャストして使用する
	 * 必要があります。(XSSFCellStyle にも名称を取得するメソッドがありません。)
	 *
	 * ※ EXCEL2010 ホームタブ→セルのスタイル は、一つづつしか削除できません。
	 *    マクロは、開発タブ→Visual Basic で、挿入→標準モジュール を開き
	 *    テキストを張り付けてください。
	 *    実行は、開発タブ→マクロ で、マクロ名を選択して、実行します。
	 *    最後は、削除してください。
	 *
	 * @og.rev 6.2.6.0 (2015/06/19) 新規作成
	 *
	 * @return	書式のスタイル一覧
	 * @see		POIUtil#getStyleNames( Workbook )
	 * @og.rtnNotNull
	 */
	public String[] getStyleNames() {
		return POIUtil.getStyleNames( wkbook );
	}

	/**
	 * 文字列を Double オブジェクトに変換します。
	 *
	 * これは、引数の カンマ(,) を削除した文字列から、Double オブジェクトを生成します。
	 * 処理中に、文字列が解析可能な double を含まない場合(NumberFormatException)
	 * また、引数が、null,ゼロ文字列,'_', エラー の時には、null を返します。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	value	Doubleに変換する元の文字列
	 *
	 * @return	変換後のDoubleオブジェクト(エラー発生時や変換不可の場合は、null)
	 */
	private Double parseDouble( final String value ) {
		Double rtn = null ;

		try {
			if( value == null || value.isEmpty() || value.equals( "_" ) ) {
				rtn = null;
			}
			else if( value.indexOf( ',' ) < 0 ) {
				rtn = Double.valueOf( value );		// 6.0.2.4 (2014/10/17) メソッドが非効率だった。
			}
			else {
				char[] chs = value.toCharArray() ;
				int j=0;
				for( int i=0;i<chs.length; i++ ) {
					if( chs[i] == ',' ) { continue; }
					chs[j] = chs[i];
					j++;
				}
				// 6.0.2.5 (2014/10/31) refactoring
				rtn = Double.valueOf( String.valueOf( chs,0,j ) );
			}
		}
		catch( NumberFormatException ex ) {		// 文字列が解析可能な数値を含まない場合
			final String errMsg = "Double変換できませんでした。" + CR
								+ ex.getMessage() + CR
								+ "  value=" + value;
			System.err.println( errMsg );
			rtn = null;
		}

		return rtn ;
	}

	/**
	 * アプリケーションのサンプルです。
	 *
	 * Usage: java org.opengion.fukurou.model.ExcelModel 入力ファイル名 [出力ファイル名] ・・・
	 *  通常は標準出力に行単位に、セルをタブ区切り出力します。
	 *  出力ファイル名 を指定すると、EXCEL ファイルとしてセーブし直します。
	 *  その場合は、以下のパラメータも使用できます。
	 *   -CS      CellStyleを 設定します。
	 *   -AS      useAutoCellSizeを 設定します。
	 *   -FN=***  FontNameを 設定します。
	 *   -FP=**   FontPointを 設定します。
	 *   -IMG     画像ファイルを挿入します。(-IMG 画像ファイル名 シート番号 行 列)をスペース区切りで続けます。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規作成
	 *
	 * @param	args	コマンド引数配列
	 */
	public static void main( final String[] args ) {
		if( args.length == 0 ) {
			final String usage = "Usage: java org.opengion.fukurou.model.ExcelModel 入力ファイル名 [出力ファイル名] ・・・\n" +
						"\t-CS      CellStyleを 設定します。        \n" +
						"\t-TC      TextConverterを実行します。     \n" +
						"\t-AS      useAutoCellSizeを 設定します。  \n" +
						"\t-FN=***  FontNameを 設定します。         \n" +
						"\t-FP=**   FontPointを 設定します。        \n" +
						"\t-IMG     画像ファイルを挿入します。      \n" +
						"\t     (-IMG ファイル名 シート番号 行 列)  \n" ;
			System.err.println( usage );
			return ;
		}

		final ExcelModel excel = new ExcelModel( new File( args[0] ) , true );

		excel.activeWorkbook( true );			// 余計な行を削除します。

		if( args.length > 1 ) {
			final File outFile = new File( args[1] );			// 6.2.0.0 (2015/02/27)
			boolean isCS = false;
			boolean isAS = false;
			boolean isTC = false;				// 6.2.4.2 (2015/05/29) テキスト変換処理
			String  fn   = null;
			short   fp   = -1;
			for( int i=2; i<args.length; i++ ) {
				final String prm = args[i];

				if( prm.equalsIgnoreCase( "-CS" ) ) { isCS = true; }
				if( prm.equalsIgnoreCase( "-AS" ) ) { isAS = true; }
				if( prm.equalsIgnoreCase( "-TC" ) ) { isTC = true; }
				if( prm.startsWith( "-FN" ) ) { fn   = prm.substring( 3 ); }
				if( prm.startsWith( "-FP" ) ) { fp   = Short.parseShort( prm.substring( 3 ) ); }
				if( prm.equalsIgnoreCase( "-IMG" ) ) {
					final String img = args[++i];
					final int  shtNo = Integer.parseInt( args[++i] );
					final int  rowNo = Integer.parseInt( args[++i] );
					final int  colNo = Integer.parseInt( args[++i] );

					excel.addImageFile( img,shtNo,rowNo,colNo );
				}
			}

			if( isCS ) { excel.setCellStyle(); }
			excel.useAutoCellSize( isAS );
			excel.setFont( fn,fp );

			// 6.2.4.2 (2015/05/29) テキスト変換処理
			if( isTC ) {
				excel.textConverter(
					new TextConverter<String,String>() {
//						public String change( final String val ) {
						public String change( final String val , final String cmnt ) {
							System.out.println( val );					// すべてのテキストを読み取る。
							return null;								// 変換せず。
						}
					}
				);
			}

			excel.saveFile( outFile );
		}
		else {
			final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );

			final int shLen = excel.getNumberOfSheets();
			for( int shNo=0; shNo<shLen; shNo++ ) {
				final String sheetName = excel.getSheetName( shNo );

				final int stRow = excel.getFirstRowNum();
				final int edRow = excel.getLastRowNum();
				for( int rowNo=stRow; rowNo<=edRow; rowNo++ ) {
					buf.setLength(0);		// Clearの事
					buf.append( sheetName ).append( '\t' ).append( rowNo );
					final String[] vals = excel.getValues( rowNo );
					if( vals != null ) {
						for( int colNo=0; colNo<vals.length; colNo++ ) {
							final String val = vals[colNo] == null ? "" : vals[colNo];
							buf.append( '\t' ).append( val );
						}
					}
					System.out.println( buf );
				}
				System.out.println();
			}
		}
	}
}
