/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.TagBuffer;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)

import static org.opengion.fukurou.util.StringUtil.nval ;		// 6.1.0.0 (2014/12/26)

/**
 * 明細棒グラフ（横棒、縦棒）に使用します。
 *
 * iBar タグは、TDタグの繰返し部に記述することで、横棒、縦棒グラフを作成します。
 * 使用するには、headタグで、xxxx ="true" を設定する必要があります。
 * これにより、common/option/adjustBar.js が読み込まれます。
 * 
 * type は、グラフ要素の表示形式を指定します。0～9 が予約語で、0,1,2,6 が現時点で実装済みです。
 *    0:ラベル表示 1:横棒 2:差分枠 6:縦棒
 *
 * <table border="1" frame="box" rules="all">
 *   <caption>棒グラフ 描画方向</caption>
 *   <tr><th colspan="2">横方向描画                  </th><th colspan="2">縦方向描画   </th></tr>
 *   <tr><td>0</td><td>ラベル表示（比率、進捗率など）</td><td>5</td><td>(保留  未実装) </td></tr>
 *   <tr><td>1</td><td>横棒                          </td><td>6</td><td>縦棒           </td></tr>
 *   <tr><td>2</td><td>差分枠                        </td><td>7</td><td>(保留  未実装) </td></tr>
 *   <tr><td>3</td><td>(保留  未実装)                </td><td>8</td><td>(保留  未実装) </td></tr>
 *   <tr><td>4</td><td>(保留  未実装)                </td><td>9</td><td>(保留  未実装) </td></tr>
 * </table>
 *
 * value は、位置、長さを設定します。設定方法は、type 毎に異なります。
 * baseVal は、グラフ要素の描画開始位置を設定します。
 * type=2:差分枠 は、value ＞ baseVal の場合は、超過を、value ＜ baseValの場合は、不足を表します。
 * <table border="1" frame="box" rules="all">
 *   <caption>type 別設定方法</caption>
 *   <tr><th>type</th><th>設定値           </th><th>説明           </th></tr>
 *   <tr><td>0   </td><td>[V1]             </td><td>xx% または 0.xx　の比率を設定する。（一つのみ） , 描画位置＝baseValの各カラムの合計値 * カラムV1の値</td></tr>
 *   <tr><td>1   </td><td>[V1],[V2],…,[Vn]</td><td>棒の長さ＝[V1],[V2],…,[Vn]各カラムの合計値</td></tr>
 *   <tr><td>2   </td><td>[V1],[V2],…,[Vn]</td><td>描画位置＝baseValの各カラムの合計値 , 差分枠の長さ＝valueの各カラムの合計値 - baseValの各カラム合計値</td></tr>
 *   <tr><td>6   </td><td>[V1],[V2],…,[Vn]</td><td>棒の長さ＝[V1],[V2],…,[Vn]各カラムの合計値</td></tr>
 * </table>
 * 
 * color は、棒グラフ要素（type="1"(横棒),"6"(縦棒)）の色コードを設定します。
 *       このコードは、予め決められた色画像を呼び出しますので、任意の色が使えるという事ではありません。
 * <table border="1" frame="box" rules="all">
 *   <caption>色番号 対応表</caption>
 *   <tr><th>色番号</th><th>設定値 </th><th>棒色  </th><th>　</th><th>横棒サンプル</th><th>縦棒サンプル</th></tr>
 *   <tr><td> 0</td><td>blue  </td><td>青  </td><td> </td><td style="background-image:url(../image/W_blue.png);">  　</td><td style="background-image:url(../image/H_blue.png);">  　</td></tr>
 *   <tr><td> 1</td><td>black </td><td>黒  </td><td> </td><td style="background-image:url(../image/W_black.png);">  　</td><td style="background-image:url(../image/H_black.png);">  　</td></tr>
 *   <tr><td> 2</td><td>brown </td><td>茶  </td><td> </td><td style="background-image:url(../image/W_brown.png);">  　</td><td style="background-image:url(../image/H_brown.png);">  　</td></tr>
 *   <tr><td> 3</td><td>green </td><td>緑  </td><td> </td><td style="background-image:url(../image/W_green.png);">  　</td><td style="background-image:url(../image/H_green.png);">  　</td></tr>
 *   <tr><td> 4</td><td>lightblue </td><td>薄青  </td><td> </td><td style="background-image:url(../image/W_lightblue.png);"> 　</td><td style="background-image:url(../image/H_lightblue.png);"> 　</td></tr>
 *   <tr><td> 5</td><td>lightgreen </td><td>薄緑  </td><td> </td><td style="background-image:url(../image/W_lightgreen.png);"> 　</td><td style="background-image:url(../image/H_lightgreen.png);"> 　</td></tr>
 *   <tr><td> 6</td><td>orange </td><td>オレンジ </td><td> </td><td style="background-image:url(../image/W_orange.png);"> 　</td><td style="background-image:url(../image/H_orange.png);"> 　</td></tr>
 *   <tr><td> 7</td><td>pink  </td><td>ピンク </td><td> </td><td style="background-image:url(../image/W_pink.png);">  　</td><td style="background-image:url(../image/H_pink.png);">  　</td></tr>
 *   <tr><td> 8</td><td>purple </td><td>紫  </td><td> </td><td style="background-image:url(../image/W_purple.png);"> 　</td><td style="background-image:url(../image/H_purple.png);"> 　</td></tr>
 *   <tr><td> 9</td><td>red  </td><td>赤  </td><td> </td><td style="background-image:url(../image/W_red.png);">  　</td><td style="background-image:url(../image/H_red.png);">  　</td></tr>
 *   <tr><td> 10</td><td>yellow </td><td>黄  </td><td> </td><td style="background-image:url(../image/W_yellow.png);"> 　</td><td style="background-image:url(../image/H_yellow.png);"> 　</td></tr>
 *   <tr><td> 11</td><td>navy  </td><td>濃紺  </td><td> </td><td style="background-image:url(../image/W_navy.png);">  　</td><td style="background-image:url(../image/H_navy.png);">  　</td></tr>
 *   <tr><td> 12</td><td>teal  </td><td>青緑  </td><td> </td><td style="background-image:url(../image/W_teal.png);">  　</td><td style="background-image:url(../image/H_teal.png);">  　</td></tr>
 *   <tr><td> 13</td><td>gray  </td><td>灰  </td><td> </td><td style="background-image:url(../image/W_gray.png);">  　</td><td style="background-image:url(../image/H_gray.png);">  　</td></tr>
 *   <tr><td> 14</td><td>indigo </td><td>藍  </td><td> </td><td style="background-image:url(../image/W_indigo.png);"> 　</td><td style="background-image:url(../image/H_indigo.png);"> 　</td></tr>
 *   <tr><td> 15</td><td>slategray </td><td>鉄色  </td><td> </td><td style="background-image:url(../image/W_slategray.png);"> 　</td><td style="background-image:url(../image/H_slategray.png);"> 　</td></tr>
 *   <tr><td> 16</td><td>chocorlate </td><td>チョコレート </td><td> </td><td style="background-image:url(../image/W_chocorlate.png);"> 　</td><td style="background-image:url(../image/H_chocorlate.png);"> 　</td></tr>
 *   <tr><td> 17</td><td>darkgreen </td><td>深緑  </td><td> </td><td style="background-image:url(../image/W_darkgreen.png);"> 　</td><td style="background-image:url(../image/H_darkgreen.png);"> 　</td></tr>
 *   <tr><td> 18</td><td>violet </td><td>すみれ </td><td> </td><td style="background-image:url(../image/W_violet.png);"> 　</td><td style="background-image:url(../image/H_violet.png);"> 　</td></tr>
 *   <tr><td> 19</td><td>darkslategray</td><td>深鉄色 </td><td> </td><td style="background-image:url(../image/W_darkslategray.png);"> 　</td><td style="background-image:url(../image/H_darkslategray.png);"> 　</td></tr>
 *   <tr><td> 20</td><td>lime  </td><td>緑  </td><td> </td><td style="background-image:url(../image/W_lime.png);">  　</td><td style="background-image:url(../image/H_lime.png);">  　</td></tr>
 *   <tr><td> 21</td><td>aqua  </td><td>水色  </td><td> </td><td style="background-image:url(../image/W_aqua.png);">  　</td><td style="background-image:url(../image/H_aqua.png);">  　</td></tr>
 *   <tr><td> 22</td><td>maroon </td><td>ひわだいろ </td><td> </td><td style="background-image:url(../image/W_maroon.png);"> 　</td><td style="background-image:url(../image/H_maroon.png);"> 　</td></tr>
 *   <tr><td> 23</td><td>olive </td><td>オリーブ </td><td> </td><td style="background-image:url(../image/W_olive.png);">  　</td><td style="background-image:url(../image/H_olive.png);">  　</td></tr>
 *   <tr><td> 24</td><td>silver </td><td>銀  </td><td> </td><td style="background-image:url(../image/W_silver.png);"> 　</td><td style="background-image:url(../image/H_silver.png);"> 　</td></tr>
 *   <tr><td> 25</td><td>fuchsia </td><td>赤紫  </td><td> </td><td style="background-image:url(../image/W_fuchsia.png);"> 　</td><td style="background-image:url(../image/H_fuchsia.png);"> 　</td></tr>
 *   <tr><td> 　</td><td>  </td><td>  </td><td> </td><td>        　</td><td>        　</td></tr>
 *   <tr><td> BK</td><td>black2 </td><td>黒  </td><td> </td><td style="background-image:url(../image/W_black2.png);"> 　</td><td style="background-image:url(../image/H_black2.png);"> 　</td></tr>
 *   <tr><td> WT</td><td>white2 </td><td>白  </td><td> </td><td style="background-image:url(../image/W_white2.png);"> 　</td><td style="background-image:url(../image/H_white2.png);"> 　</td></tr>
 *   <tr><td> BL</td><td>blue2 </td><td>青  </td><td> </td><td style="background-image:url(../image/W_blue2.png);">  　</td><td style="background-image:url(../image/H_blue2.png);">  　</td></tr>
 *   <tr><td> RD</td><td>red2  </td><td>赤  </td><td> </td><td style="background-image:url(../image/W_red2.png);">  　</td><td style="background-image:url(../image/H_red2.png);">  　</td></tr>
 *   <tr><td> YL</td><td>yellow2 </td><td>黄  </td><td> </td><td style="background-image:url(../image/W_yellow2.png);"> 　</td><td style="background-image:url(../image/H_yellow2.png);"> 　</td></tr>
 * </table>
 *
 * colorNo 色番号は、色コードを直接指定するのではなく、番号で指定する場合に使用します。
 *         棒グラフ要素（type="1"(横棒),"6"(縦棒)）の色を設定します。
 *         "1～25 , BK,WT,BL,RD,YL" の色番号で指定します。
 *         色番号は、与えた数字を内部色数で割り算した余を使用しますので、連番の場合でも
 *         繰返し使われます。特殊な色として、BK,WT,BL,RD,YLは、番号ではなく記号で指定できます。
 * 
 * zeroBorder は、"true" で、値がゼロの時にも、最小のバーを表示する。
 * 
 * graphThick は、グラフのバーの幅を指定します。
 *
 * @og.formSample
 * ●形式：&lt;og:iBar  ... /&gt;
 * ●body：なし
 * ●前提：headタグで、adjustEvent="Bar" を指定してください。
 *
 * ●Tag定義：
 *   &lt;og:iBar
 *       type             ○【TAG】グラフ要素の表示形式を設定(0:ラベル表示 1:横棒 2:差分枠 6:縦棒)(必須)
 *       value            ○【TAG】グラフ要素の位置、長さを設定(typeにより設定方法が異なる)(必須)
 *       baseVal            【TAG】グラフ要素の描画開始位置を設定(type=0,2の場合必須)
 *       color              【TAG】棒グラフ要素（type="1"(横棒),"6"(縦棒)）の色コードを設定する。
 *       colorNo            【TAG】"1～25 , BK,WT,BL,RD,YL" の色番号を設定する。
 *       text               【TAG】バー上に表示するテキストを指定します。
 *       href               【TAG】リンク作成時の値を設定します。
 *       target             【TAG】リンク作成時のターゲット名を指定します。
 *       optionAttributes   【TAG】 {kay:val,key:val} 形式で、属性を追加します。
 *       zeroBorder         【TAG】最小のバーを表示するかどうか[true/false]を指定します(初期値:false)
 *       graphThick         【TAG】グラフのバーの幅を指定します。
 *       paddingLeft        【TAG】ガントバーの間の左区切りスペースを指定します(初期値=null)
 *       paddingRigth       【TAG】ガントバーの間の右区切りスペースを指定します(初期値=null)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *    &lt;og:view command="{&#064;command}" /&gt;
 *      &lt;og:tbody rowspan="1" &gt;
 *        &lt;tr&gt;
 *          &lt;og:td&gt;
 *            &lt;og:iBar
 *                  type    = "X"
 *                  value   = "[V1],[V2],…,[Vn]"
 *                  baseVal = "[BV1],[BV2],…,[BVn]"
 *                  color   = "C1,C2,…,Cn"
 *            /&gt;
 *        &lt;/td&gt; &lt;/tr&gt;
 *      &lt;/og:tbody&gt;
 *    &lt;/og:view&gt;
 *
 * @og.rev 5.6.3.2 (2013/04/12) 新規作成
 * @og.group 画面部品
 *
 * @version  5.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK1.7,
 */
public class ViewIBarTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.3.4.0 (2015/08/01)" ;
	private static final long serialVersionUID = 634020150801L ;

	/** type 引数に渡す事の出来る タイプリスト  */
	private static final String[] TYPE_LIST = { "0" , "1" , "2" , "6" };

	// 6.1.1.0 (2015/01/17) TagBufferの連結記述
	private transient TagBuffer tag = new TagBuffer( "iBar" ).addBody( "<!-- -->" );

	// ページ内の iBar タグのid設定用のユニーク番号。
	// java.util.concurrent.atomic.AtomicInteger を使うまでもない。
	private static int tagId = 100;

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 6.0.2.5 (2014/10/31) HTML5対応。javaScriptで、BODYがないと入れ子になってしまう。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		tag.add( "id",getTagId() );

		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		jspPrint( tag.makeTag() );

		return EVAL_PAGE ;		// ページの残りを評価する。
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		tag = new TagBuffer( "iBar" ).addBody( "<!-- -->" );		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
	}

	/**
	 * 【TAG】グラフ要素の表示形式を設定[0:ラベル表示 1:横棒 2:差分枠 6:縦棒]します。
	 *
	 * @og.tag
	 * type は、グラフ要素の表示形式を指定します。0～9 が予約語で、0,1,2,6 が現時点で実装済みです。
	 * 0:ラベル表示 1:横棒 2:差分枠 6:縦棒
	 *
	 * @og.rev 6.3.4.0 (2015/08/01) Arrays.toString から String.join に置き換え。
	 *
	 * @param   type グラフ要素の表示形式 [0:ラベル表示 1:横棒 2:差分枠 6:縦棒]
	 */
	public void setType( final String type ) {
		final String tmpType = nval( getRequestParameter( type ),null );

		if( tmpType == null || !check( tmpType, TYPE_LIST ) ) {
			final String errMsg = "指定の表示形式(type)は必須か、または、指定の値のみ登録可能です。"	+ CR
							+ "type=[" + tmpType + "] "													+ CR
//							+ "登録可能な値=" + Arrays.toString( TYPE_LIST ) ;		// 6.1.0.0 (2014/12/26)
							+ "登録可能な値=" + String.join( ", " , TYPE_LIST ) ;	// 6.3.4.0 (2015/08/01)
			throw new HybsSystemException( errMsg );
		}

		tag.add( "type",tmpType );
	}

	/**
	 * 【TAG】グラフ要素の位置、長さを設定します(typeにより設定方法が異なる)。
	 *
	 * @og.tag
	 * value は、グラフ要素の位置、長さを設定します。設定方法は、type 毎に異なります。
	 * baseVal は、グラフ要素の描画開始位置を設定します。
	 * type=2:差分枠 は、value ＞ baseVal の場合は、超過を、value ＜ baseValの場合は、不足を表します。
	 * <table border="1" frame="box" rules="all">
	 *   <caption>グラフのtype別設定方法</caption>
	 *   <tr><th>type</th><th>設定値           </th><th>説明           </th></tr>
	 *   <tr><td>0   </td><td>[V1]             </td><td>xx% または 0.xx　の比率を設定する。（一つのみ）, 描画位置＝baseValの各カラムの合計値 * カラムV1の値</td></tr>
	 *   <tr><td>1   </td><td>[V1],[V2],…,[Vn]</td><td>棒の長さ＝[V1],[V2],…,[Vn]各カラムの合計値</td></tr>
	 *   <tr><td>2   </td><td>[V1],[V2],…,[Vn]</td><td>描画位置＝baseValの各カラムの合計値 , 差分枠の長さ＝valueの各カラムの合計値 - baseValの各カラム合計値</td></tr>
	 *   <tr><td>6   </td><td>[V1],[V2],…,[Vn]</td><td>棒の長さ＝[V1],[V2],…,[Vn]各カラムの合計値</td></tr>
	 * </table>
	 *
	 * @param   val 位置、長さ
	 */
	public void setValue( final String val ) {
		final String tmpVal = nval( getRequestParameter( val ),null );

		if( tmpVal == null ) {
			final String errMsg = "value は、必須です。";
			throw new HybsSystemException( errMsg );
		}

		tag.add( "value",tmpVal );
	}

	/**
	 * 【TAG】グラフ要素の描画開始位置を設定します。
	 *
	 * @og.tag
	 * この値は、type=0,2の場合、必須になります。
	 * type=2:差分枠 は、value ＞ baseVal の場合は、超過を、value ＜ baseValの場合は、不足を表します。
	 * <table border="1" frame="box" rules="all">
	 *   <caption>グラフのtype別設定方法</caption>
	 *   <tr><th>type</th><th>設定値           </th><th>説明           </th></tr>
	 *   <tr><td>0   </td><td>[V1]             </td><td>xx% または 0.xx　の比率を設定する。（一つのみ） , 描画位置＝baseValの各カラムの合計値 * カラムV1の値</td></tr>
	 *   <tr><td>2   </td><td>[V1],[V2],…,[Vn]</td><td>描画位置＝baseValの各カラムの合計値 , 差分枠の長さ＝valueの各カラムの合計値 - baseValの各カラム合計値</td></tr>
	 * </table>
	 *
	 * @param   bval 描画開始位置
	 */
	public void setBaseVal( final String bval ) {
		tag.add( "baseVal",nval( getRequestParameter( bval ), null ) );
	}

	/**
	 * 【TAG】棒グラフ要素（type="1"(横棒),"6"(縦棒)）の色コードを設定します。
	 *
	 * @og.tag
	 * color は、棒グラフ要素（type="1"(横棒),"6"(縦棒)）の色コードを設定します。
	 * このコードは、予め決められた色画像を呼び出しますので、任意の色が使えるという事ではありません。
	 * 色指定は、横棒の場合は、image/W_色コード.png で、縦棒の場合は、image/H_色コード.png の画像を使います。
	 * 色コード と 色番号が同時に指定された場合は、色コード が優先されます。
	 *
	 * type=2:差分枠 は、value ＞ baseVal の場合は、超過を、value ＜ baseValの場合は、不足を表します。
	 * <table border="1" frame="box" rules="all">
	 *   <caption>色番号 対応表</caption>
	 *   <tr><th>色番号</th><th>設定値 </th><th>棒色  </th><th>　</th><th>横棒サンプル</th><th>縦棒サンプル</th></tr>
	 *   <tr><td> 0</td><td>blue  </td><td>青  </td><td> </td><td style="background-image:url(../image/W_blue.png);">  　</td><td style="background-image:url(../image/H_blue.png);">  　</td></tr>
	 *   <tr><td> 1</td><td>black </td><td>黒  </td><td> </td><td style="background-image:url(../image/W_black.png);">  　</td><td style="background-image:url(../image/H_black.png);">  　</td></tr>
	 *   <tr><td> 2</td><td>brown </td><td>茶  </td><td> </td><td style="background-image:url(../image/W_brown.png);">  　</td><td style="background-image:url(../image/H_brown.png);">  　</td></tr>
	 *   <tr><td> 3</td><td>green </td><td>緑  </td><td> </td><td style="background-image:url(../image/W_green.png);">  　</td><td style="background-image:url(../image/H_green.png);">  　</td></tr>
	 *   <tr><td> 4</td><td>lightblue </td><td>薄青  </td><td> </td><td style="background-image:url(../image/W_lightblue.png);"> 　</td><td style="background-image:url(../image/H_lightblue.png);"> 　</td></tr>
	 *   <tr><td> 5</td><td>lightgreen </td><td>薄緑  </td><td> </td><td style="background-image:url(../image/W_lightgreen.png);"> 　</td><td style="background-image:url(../image/H_lightgreen.png);"> 　</td></tr>
	 *   <tr><td> 6</td><td>orange </td><td>オレンジ </td><td> </td><td style="background-image:url(../image/W_orange.png);"> 　</td><td style="background-image:url(../image/H_orange.png);"> 　</td></tr>
	 *   <tr><td> 7</td><td>pink  </td><td>ピンク </td><td> </td><td style="background-image:url(../image/W_pink.png);">  　</td><td style="background-image:url(../image/H_pink.png);">  　</td></tr>
	 *   <tr><td> 8</td><td>purple </td><td>紫  </td><td> </td><td style="background-image:url(../image/W_purple.png);"> 　</td><td style="background-image:url(../image/H_purple.png);"> 　</td></tr>
	 *   <tr><td> 9</td><td>red  </td><td>赤  </td><td> </td><td style="background-image:url(../image/W_red.png);">  　</td><td style="background-image:url(../image/H_red.png);">  　</td></tr>
	 *   <tr><td> 10</td><td>yellow </td><td>黄  </td><td> </td><td style="background-image:url(../image/W_yellow.png);"> 　</td><td style="background-image:url(../image/H_yellow.png);"> 　</td></tr>
	 *   <tr><td> 11</td><td>navy  </td><td>濃紺  </td><td> </td><td style="background-image:url(../image/W_navy.png);">  　</td><td style="background-image:url(../image/H_navy.png);">  　</td></tr>
	 *   <tr><td> 12</td><td>teal  </td><td>青緑  </td><td> </td><td style="background-image:url(../image/W_teal.png);">  　</td><td style="background-image:url(../image/H_teal.png);">  　</td></tr>
	 *   <tr><td> 13</td><td>gray  </td><td>灰  </td><td> </td><td style="background-image:url(../image/W_gray.png);">  　</td><td style="background-image:url(../image/H_gray.png);">  　</td></tr>
	 *   <tr><td> 14</td><td>indigo </td><td>藍  </td><td> </td><td style="background-image:url(../image/W_indigo.png);"> 　</td><td style="background-image:url(../image/H_indigo.png);"> 　</td></tr>
	 *   <tr><td> 15</td><td>slategray </td><td>鉄色  </td><td> </td><td style="background-image:url(../image/W_slategray.png);"> 　</td><td style="background-image:url(../image/H_slategray.png);"> 　</td></tr>
	 *   <tr><td> 16</td><td>chocorlate </td><td>チョコレート </td><td> </td><td style="background-image:url(../image/W_chocorlate.png);"> 　</td><td style="background-image:url(../image/H_chocorlate.png);"> 　</td></tr>
	 *   <tr><td> 17</td><td>darkgreen </td><td>深緑  </td><td> </td><td style="background-image:url(../image/W_darkgreen.png);"> 　</td><td style="background-image:url(../image/H_darkgreen.png);"> 　</td></tr>
	 *   <tr><td> 18</td><td>violet </td><td>すみれ </td><td> </td><td style="background-image:url(../image/W_violet.png);"> 　</td><td style="background-image:url(../image/H_violet.png);"> 　</td></tr>
	 *   <tr><td> 19</td><td>darkslategray</td><td>深鉄色 </td><td> </td><td style="background-image:url(../image/W_darkslategray.png);"> 　</td><td style="background-image:url(../image/H_darkslategray.png);"> 　</td></tr>
	 *   <tr><td> 20</td><td>lime  </td><td>緑  </td><td> </td><td style="background-image:url(../image/W_lime.png);">  　</td><td style="background-image:url(../image/H_lime.png);">  　</td></tr>
	 *   <tr><td> 21</td><td>aqua  </td><td>水色  </td><td> </td><td style="background-image:url(../image/W_aqua.png);">  　</td><td style="background-image:url(../image/H_aqua.png);">  　</td></tr>
	 *   <tr><td> 22</td><td>maroon </td><td>ひわだいろ </td><td> </td><td style="background-image:url(../image/W_maroon.png);"> 　</td><td style="background-image:url(../image/H_maroon.png);"> 　</td></tr>
	 *   <tr><td> 23</td><td>olive </td><td>オリーブ </td><td> </td><td style="background-image:url(../image/W_olive.png);">  　</td><td style="background-image:url(../image/H_olive.png);">  　</td></tr>
	 *   <tr><td> 24</td><td>silver </td><td>銀  </td><td> </td><td style="background-image:url(../image/W_silver.png);"> 　</td><td style="background-image:url(../image/H_silver.png);"> 　</td></tr>
	 *   <tr><td> 25</td><td>fuchsia </td><td>赤紫  </td><td> </td><td style="background-image:url(../image/W_fuchsia.png);"> 　</td><td style="background-image:url(../image/H_fuchsia.png);"> 　</td></tr>
	 *   <tr><td> 　</td><td>  </td><td>  </td><td> </td><td>        　</td><td>        　</td></tr>
	 *   <tr><td> BK</td><td>black2 </td><td>黒  </td><td> </td><td style="background-image:url(../image/W_black2.png);"> 　</td><td style="background-image:url(../image/H_black2.png);"> 　</td></tr>
	 *   <tr><td> WT</td><td>white2 </td><td>白  </td><td> </td><td style="background-image:url(../image/W_white2.png);"> 　</td><td style="background-image:url(../image/H_white2.png);"> 　</td></tr>
	 *   <tr><td> BL</td><td>blue2 </td><td>青  </td><td> </td><td style="background-image:url(../image/W_blue2.png);">  　</td><td style="background-image:url(../image/H_blue2.png);">  　</td></tr>
	 *   <tr><td> RD</td><td>red2  </td><td>赤  </td><td> </td><td style="background-image:url(../image/W_red2.png);">  　</td><td style="background-image:url(../image/H_red2.png);">  　</td></tr>
	 *   <tr><td> YL</td><td>yellow2 </td><td>黄  </td><td> </td><td style="background-image:url(../image/W_yellow2.png);"> 　</td><td style="background-image:url(../image/H_yellow2.png);"> 　</td></tr>
	 * </table>
	 *
	 * @param   color 色コード
	 */
	public void setColor( final String color ) {
		tag.add( "color",nval( getRequestParameter( color ),null ) );
	}

	/**
	 * 【TAG】グラフ要素の色番号を設定します。
	 *
	 * @og.tag
	 * 色番号は、色コードを直接指定するのではなく、番号で指定する場合に使用します。
	 * 棒グラフ要素（type="1"(横棒),"6"(縦棒)）の色を設定します。
	 * "1～25 , BK,WT,BL,RD,YL" の色番号で指定します。
	 * 色番号は、与えた数字を内部色数で割り算した余を使用しますので、連番の場合でも
	 * 繰返し使われます。特殊な色として、BK,WT,BL,RD,YLは、番号ではなく記号で指定できます。
	 * 色コード と 色番号が同時に指定された場合は、色コード が優先されます。
	 *
	 * @param   clrNo 色番号 [1～25/BK/WT/BL/RD/YL]
	 * @see		#setColor( String )
	 */
	public void setColorNo( final String clrNo ) {
		tag.add( "colorNo",nval( getRequestParameter( clrNo ),null ) );
	}

	/**
	 * 【TAG】バー上に表示するテキストを指定します。
	 *
	 * @og.tag
	 * バーの上に、値などの文字を重ねて表示できます。
	 *
	 * @param   text テキスト
	 */
	public void setText( final String text ) {
		tag.add( "text",nval( getRequestParameter( text ),null ) );
	}

	/**
	 * 【TAG】リンク作成時の値を設定します。
	 *
	 * @og.tag
	 * バーの上の文字に対して、リンクを作成できます。
	 *
	 * @param   href リンク
	 * @see		#setTarget( String )
	 */
	public void setHref( final String href ) {
		tag.add( "href",nval( getRequestParameter( href ),null ) );
	}

	/**
	 * 【TAG】リンク作成時のターゲット名を指定します。
	 *
	 * @og.tag
	 * バーの上の文字に対して、リンクを作成するときの、ターゲット名を指定できます。
	 *
	 * @param   target ターゲット
	 * @see		#setHref( String )
	 */
	public void setTarget( final String target ) {
		tag.add( "target",nval( getRequestParameter( target ),null ) );
	}

	/**
	 * 【TAG】既定の属性以外の属性を追加します。
	 *
	 * @og.tag
	 * {kay:val,key:val} 形式で、属性を追加します。
	 *
	 * @param   optAtt 追加属性
	 */
	public void setOptionAttributes( final String optAtt ) {
		tag.add( "optionAttributes",nval( getRequestParameter( optAtt ),null ) );
	}

	/**
	 * 【TAG】最小のバーを表示するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * 設定値が、ゼロの場合に、バーを出したい場合は、true に、
	 * 何も出したくない場合は、false に設定します。
	 * 初期値は、false(バーを出さない) です。
	 *
	 * @param   zeroBorder 最小のバー表示可否 [true:する/false:しない]
	 */
	public void setZeroBorder( final String zeroBorder ) {
		tag.add( "zeroBorder",nval( getRequestParameter( zeroBorder ),null ) );
	}

	/**
	 * 【TAG】グラフのバーの幅を指定します。
	 *
	 * @og.tag
	 * 何も指定しない場合は、通常のバーの画像ファイルが適用されます。
	 *
	 * @param   graphThick グラフのバーの幅
	 */
	public void setGraphThick( final String graphThick ) {
		tag.add( "graphThick",nval( getRequestParameter( graphThick ),null ) );
	}

	/**
	 * 【TAG】バーの間の左区切りスペースをピクセルで指定します(初期値:null)。
	 *
	 * @og.tag
	 * バー表示で連続して同じ色で繋げる場合、くっついて表示されます。
	 * これを、区別できるように、左区切りスペースをピクセルで指定します。
	 *
	 * 初期値は、null(属性を出力しない) です。
	 *
	 * @og.rev 5.6.4.3 (2013/05/24) 新規追加
	 *
	 * @param   paddingLeft 左区切りスペース
	 * @see		#setPaddingRigth( String )
	 */
	public void setPaddingLeft( final String paddingLeft ) {
		tag.add( "paddingLeft",nval( getRequestParameter( paddingLeft ),null ) );
	}

	/**
	 * 【TAG】バーの間の右区切りスペースをピクセルで指定します(初期値:null)。
	 *
	 * @og.tag
	 * バー表示で連続して同じ色で繋げる場合、くっついて表示されます。
	 * これを、区別できるように、右区切りスペースをピクセルで指定します。
	 *
	 * 初期値は、null(属性を出力しない) です。
	 *
	 * @og.rev 5.6.4.3 (2013/05/24) 新規追加
	 *
	 * @param   paddingRigth 左区切りスペース
	 * @see		#setPaddingLeft( String )
	 */
	public void setPaddingRigth( final String paddingRigth ) {
		tag.add( "paddingRigth",nval( getRequestParameter( paddingRigth ),null ) );
	}

	/**
	 * iBar タグがユニークになるように id 値を返します。
	 *
	 * iBar は、レコードに複数作成されるため、まず、iBar作成都度のユニークIDと
	 * それをViewFormで展開後にユニークな番号になる様にするための行番号が組み合わされた
	 * ID が必要です。
	 * iBatXXX_[I] 形式のIDを作成します。
	 * 別スレッドによる同時アクセスで整合性が破綻(同じ番号が返る)しても、問題ありません。
	 *
	 * @return  ユニークな id 値
	 */
	private static String getTagId() {
		final String id = "IBar" + ( tagId++ ) + "_[I]";

		if( tagId > 999 ) { tagId = 100; }

		return id ;
	}

	/**
	 * タグの名称を、返します。
	 * 自分自身のクラス名より、自動的に取り出せないため、このメソッドをオーバーライドします。
	 *
	 * @return  タグの名称
	 * @og.rtnNotNull
	 */
	@Override
	protected String getTagName() {
		return "iBar" ;
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION		)
				.println( "tag"			,tag.makeTag()	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
