/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.db.AbstractRenderer;
import org.opengion.hayabusa.db.CellRenderer;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.resource.ResourceFactory;
import org.opengion.hayabusa.resource.ResourceManager;

/**
 * COLUMN レンデラーは、データの値をカラム名と認識して、動的カラムを
 * 表示するクラスです。
 *
 * データの値を、動的カラムのそのカラム属性を持ったフォームを表示します。
 *
 *  カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.group データ表示
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Renderer_COLUMN extends AbstractRenderer {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.0.4.0 (2014/11/28)" ;

	private final String lang ;
	private final String name ;			// 3.5.2.0 (2003/10/20) エラー時のキー

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.5.2.0 (2003/10/20) エラー時の表示で、name を使用します。
	 *
	 */
	public Renderer_COLUMN() {
		lang   = null;
		name   = null;
	}

	/**
	 * デフォルトコンストラクター。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.5.2.0 (2003/10/20) エラー時の表示で、name を使用します。
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	private Renderer_COLUMN( final DBColumn clm ) {
		lang = clm.getLang();
		name = clm.getName();
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.1.2.1 (2003/04/10) synchronized を、削除します。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellRendererオブジェクト
	 * @og.rtnNotNull
	 */
	public CellRenderer newInstance( final DBColumn clm ) {
		return new Renderer_COLUMN( clm );
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * @og.rev 3.5.2.0 (2003/10/20) 値で動的カラムが作成できない場合に、エラーメッセージを追加。
	 * @og.rev 6.0.4.0 (2014/11/28) エラーで、Exceptionではなく、エラーの値を返す。
	 *
	 * @param	value 入力値
	 *
	 * @return	データの表示用文字列
	 */
	@Override
	public String getValue( final String value ) {
		final ResourceManager resource = ResourceFactory.newInstance( lang ) ;
		final DBColumn dbColumn = resource.getDBColumn( value );

		final String rtn ;			// 6.0.4.0 (2014/11/28)
		if( dbColumn == null ) {
			final String errMsg = "指定のカラムの値[]で、動的カラムは作成できません。"
						+ CR
						+ "  name=[" + name + "]"
						+ "  value=[" + value + "]";
			System.err.println( errMsg );
			rtn = "<span class=\"error\">" + name + "[" + value + "]" + "</span>";
		}
		else {
			rtn = dbColumn.getLabel();		// 6.0.4.0 (2014/11/28)
		}

		return rtn;
	}

	/**
	 * データ出力用の文字列を作成します。
	 * ファイル等に出力する形式を想定しますので、HTMLタグを含まない
	 * データを返します。
	 * 基本は、#getValue( String ) をそのまま返します。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) データ出力用のレンデラー
	 *
	 * @param   value 入力値
	 *
	 * @return  データ出力用の文字列
	 * @og.rtnNotNull
	 * @see		#getValue( String )
	 */
	@Override
	public String getWriteValue( final String value ) {
		return value==null ? "" : value;
	}
}
