/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.view;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;

/**
 * 検索結果を自動的に表形式に変換する、テーブル作成クラスです。
 *
 * ユーザー単位に表示するカラムの順番、表示可非を指定できるように対応します。
 * setColumnDisplay( final String columnName ) に、指定された順番に
 * 表示するというHTMLFormatTable の簡易版として用意します。
 * 各HTMLのタグに必要な setter/getterメソッドのみ，追加定義しています。
 *
 * AbstractViewForm を継承している為,ロケールに応じたラベルを出力させる事が出来ます。
 *
 * @og.group 画面表示
 * @og.rev 5.1.6.0 (2010/05/01) 新規作成
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ViewForm_HTMLSeqClmTable extends ViewForm_HTMLTable {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.5.4.2 (2012/07/13)" ;

	private int[]	clmNo		= null;		// 5.1.6.0 (2010/05/01)
	private int		clmCnt		= -1;		// 5.1.6.0 (2010/05/01)

	private String	viewClms	= null;		//

	/**
	 * DBTableModel から HTML文字列を作成して返します。
	 * startNo(表示開始位置)から、pageSize(表示件数)までのView文字列を作成します。
	 * 表示残りデータが pageSize 以下の場合は,残りのデータをすべて出力します。
	 *
	 * @og.rev 5.5.4.2 (2012/07/13) editName指定時の編集対応
	 *
	 * @param  startNo	  表示開始位置
	 * @param  pageSize   表示件数
	 *
	 * @return	DBTableModelから作成された HTML文字列
	 */
	@Override
	public String create( final int startNo, final int pageSize )  {
		if( getRowCount() == 0 ) { return ""; }	// 暫定処置

		if( clmCnt < 0 ) {
			String errMsg = "ViewForm_HTMLSeqClmTable クラスの実行前に、setColumnDisplay しておく必要があります。";
			throw new HybsSystemException( errMsg );
		}

		headerLine	 = null;
		int lastNo = getLastNo( startNo, pageSize );
		int blc = getBackLinkCount();
		int hsc = getHeaderSkipCount();		// 3.5.2.0 (2003/10/20)
		int hscCnt = 1;						// 3.5.2.0 (2003/10/20)

		StringBuilder out = new StringBuilder( HybsSystem.BUFFER_LARGE );

		out.append( getCountForm( startNo,pageSize ) );
		out.append( getHeader() );

		String ckboxTD = "  <td>";

		out.append("<tbody>").append( HybsSystem.CR );
		int bgClrCnt = 0;
//		int clmCnt = getColumnCount();	// 3.5.5.7 (2004/05/10)
		for( int row=startNo; row<lastNo; row++ ) {
			if( isSkip( row ) || isSkipNoEdit( row ) ) { continue; } // 4.3.1.0 (2008/09/08)
			out.append("<tr").append( getBgColorCycleClass( bgClrCnt++,row ) );
			if( isNoTransition() ) { // 4.3.3.0 (2008/10/01)
				out.append( getHiddenRowValue( row ) );
			}
			out.append(">");	// 3.7.0.3 (2005/03/01)
			out.append( HybsSystem.CR );
			// 3.5.5.0 (2004/03/12) No 欄そのものの作成判断追加
			if( isNumberDisplay() ) {
				out.append( makeCheckbox( ckboxTD, row, blc ) ).append( HybsSystem.CR );
			}
			// 5.1.6.0 (2010/05/01)
//			for(int column = 0; column < clmCnt; column++) {
			for(int clm = 0; clm < clmCnt; clm++) {
				int column = clmNo[clm];
				if( isColumnDisplay( column ) ) {
					out.append("  <td>");
					out.append( getValueLabel(row,column) );
					out.append("</td>").append( HybsSystem.CR );
				}
			}
			
			
			// 5.5.4.2 (2012/07/13) mustとmuntAnyでwritableのものはdisplay:noneで出力する(可能な限り余分なものは出力しない)
			for(int column = 0; column < clmCnt; column++) {
				if( !isColumnDisplay( column ) && ( isMustColumn( column ) || isMustAnyColumn(column) ) && isColumnWritable( column) ) {
					out.append("  <td style=\"display:none\">");
					out.append( getValueLabel(row,column) );
					out.append("</td>").append( HybsSystem.CR );
				}
			}
			
			
			out.append("</tr>").append( HybsSystem.CR );

			// 3.5.2.0 (2003/10/20) ヘッダー繰り返し属性( headerSkipCount )を採用
			if( hsc > 0 && hscCnt % hsc == 0 ) {
				out.append( getHeadLine() );
				hscCnt = 1;
			}
			else {
				hscCnt ++ ;
			}
		}
		out.append("</tbody>").append( HybsSystem.CR );
		out.append("</table>").append( HybsSystem.CR );

		out.append( getScrollBarEndDiv() );	// 3.8.0.3 (2005/07/15)

		return out.toString();
	}

	/**
	 * DBTableModel から テーブルのタグ文字列を作成して返します。
	 *
	 * @og.rev 3.5.1.0 (2003/10/03) Noカラムに、numberType 属性を追加
	 * @og.rev 3.5.2.0 (2003/10/20) ヘッダー繰り返し部をgetHeadLine()へ移動
	 * @og.rev 3.5.3.1 (2003/10/31) VERCHAR2 を VARCHAR2 に修正。
	 * @og.rev 3.5.5.0 (2004/03/12) No 欄そのものの作成判断ロジックを追加
	 * @og.rev 3.5.6.5 (2004/08/09) thead に、id="header" を追加
	 * @og.rev 4.0.0.0 (2005/01/31) DBColumn の 属性(CLS_NM)から、DBTYPEに変更
	 * @og.rev 4.0.0.0 (2005/01/31) 新規作成(getColumnClassName ⇒ getColumnDbType)
	 * @og.rev 5.9.1.2 (2015/10/23) 自己終了警告対応
	 *
	 * @return	テーブルのタグ文字列
	 */
	@Override
	protected String getTableHead() {
		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

		// 3.5.5.0 (2004/03/12) No 欄そのものの作成判断追加
		if( isNumberDisplay() ) {
//			buf.append("<colgroup class=\"X\" />");		// 4.0.0 (2005/01/31)
//			buf.append("<colgroup class=\"BIT\" />");
//			buf.append("<colgroup class=\"S9\" />");		// 4.0.0 (2005/01/31)
			buf.append("<colgroup class=\"X\" ><!-- --></colgroup>");		// 5.9.1.2 (2015/10/23)
			buf.append("<colgroup class=\"BIT\" ><!-- --></colgroup>");
			buf.append("<colgroup class=\"S9\" ><!-- --></colgroup>");		
			buf.append( HybsSystem.CR );
		}

//		int clmCnt = getColumnCount();	// 3.5.5.7 (2004/05/10)
		// 5.1.6.0 (2010/05/01)
//		for(int column = 0; column < clmCnt; column++) {
		for(int clm = 0; clm < clmCnt; clm++) {
			int column = clmNo[clm];
			if( isColumnDisplay( column ) ) {
				buf.append("<colgroup class=\"" );
				buf.append( getColumnDbType(column) );		// 4.0.0 (2005/01/31)
//				buf.append("\"/>");
				buf.append("\"><!-- --></colgroup>");		// 5.9.1.2 (2015/10/23)
				buf.append( HybsSystem.CR );
			}
		}

	// 3.5.2.0 (2003/10/20) ヘッダー繰り返し部をgetHeadLine()へ移動
		buf.append("<thead id=\"header\">").append( HybsSystem.CR );	// 3.5.6.5 (2004/08/09)
		buf.append( getHeadLine() );
		buf.append("</thead>").append( HybsSystem.CR );

		return buf.toString();
	}

	/**
	 * ヘッダー繰り返し部を、getTableHead()メソッドから分離。
	 *
	 * @og.rev 3.5.2.0 (2003/10/20) 新規作成
	 * @og.rev 3.5.4.3 (2004/01/05) useCheckControl 属性の機能を追加
	 * @og.rev 3.5.4.5 (2004/01/23) thタグの属性設定出来る様に新規追加。
	 * @og.rev 3.5.4.6 (2004/01/30) numberType="none" 時の処理を追加(Noラベルを出さない)
	 * @og.rev 3.5.4.7 (2004/02/06) ヘッダーにソート機能用のリンクを追加します。
	 * @og.rev 3.7.0.1 (2005/01/31) 全件チェックコントロール処理変更
	 *
	 * @param	thTag タグの文字列
	 *
	 * @return	テーブルのタグ文字列
	 */
	@Override
	protected String getHeadLine( final String thTag ) {
		if( headerLine != null ) { return headerLine; }		// キャッシュを返す。

		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

//		buf.append("<tr class=\"row_h\"").append(" >").append( HybsSystem.CR );
		buf.append("<tr class=\"row_hu\"").append(" >").append( HybsSystem.CR );

		// 3.5.5.0 (2004/03/12) No 欄そのものの作成判断追加
		if( isNumberDisplay() ) {
			// 3.5.4.3 (2004/01/05) 追加分
			if( isUseCheckControl() && "checkbox".equals( getSelectedType() ) ) {
			// 3.5.4.5 (2004/01/23) thタグの属性設定出来る様に変更。
				buf.append( thTag ).append("></th>");
				buf.append( thTag ).append(">").append( getAllCheckControl() ).append("</th>");
				buf.append( thTag ).append(">").append( getNumberHeader() ).append("</th>");	// 3.5.4.6 (2004/01/30)
			}
			else {
			// 3.5.4.5 (2004/01/23) thタグの属性設定出来る様に変更。
				buf.append( thTag ).append(" colspan='3'>").append( getNumberHeader() ).append("</th>");	// 3.5.4.6 (2004/01/30)
			}
		}

		buf.append( HybsSystem.CR );
//		int clmCnt = getColumnCount();	// 3.5.5.7 (2004/05/10)
		// 5.1.6.0 (2010/05/01)
//		for(int column = 0; column < clmCnt; column++) {
		for(int clm = 0; clm < clmCnt; clm++) {
			int column = clmNo[clm];
			if( isColumnDisplay( column ) ) {
			// 3.5.4.5 (2004/01/23) thタグの属性設定出来る様に変更。
				buf.append( thTag ).append(">");
				buf.append( getSortedColumnLabel(column) );
				buf.append("</th>").append( HybsSystem.CR );
			}
		}
		buf.append("</tr>").append( HybsSystem.CR );

		headerLine = buf.toString();
		return headerLine;
	}

	/**
	 * 表示可能カラム名を、カンマ区切りで与えます。
	 * 例："OYA,KO,HJO,SU,DYSET,DYUPD"
	 * setColumnDisplay( int column,boolean rw ) の簡易版です。
	 * null を与えた場合は,なにもしません。
	 *
	 * @param	columnName	カラム名
	 */
	@Override
	public void setColumnDisplay( final String columnName ) {
		super.setColumnDisplay( columnName );

		if( columnName != null ) {
			String[] clmNames = StringUtil.csv2Array( columnName );
			clmCnt = clmNames.length;
			clmNo  = new int[clmCnt];
			for( int i=0; i<clmCnt; i++ ) {
				clmNo[i] = getColumnNo( clmNames[i] );
			}
		}
		this.viewClms = columnName;
	}

	/**
	 * ビューで表示したカラムの一覧をカンマ区切りで返します。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規追加
	 * @og.rev 5.8.6.0 (2015/04/03) オリジナルカラム対応
	 *
	 * @return	ビューで表示したカラムの一覧
	 */
	@Override
	public String getViewClms() {
		if( viewClms == null ) { return super.getViewClms(); }	// 5.8.6.0 (2015/04/03)
		return viewClms;
	}
}
