/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.db.AbstractRenderer;
import org.opengion.hayabusa.db.CellRenderer;
import org.opengion.hayabusa.db.DBColumn;

/**
 * YMDH レンデラーは、カラムのデータを日時(年/月/日 時:分:秒)表示する場合に使用するクラスです。
 *
 * このクラスは、不変オブジェクトとして、共有されます。
 *
 * @og.group データ表示
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Renderer_YMDH extends AbstractRenderer {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.0.4.0 (2014/11/28)" ;

	private static final CellRenderer dbCell = new Renderer_YMDH() ;

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.1.2.1 (2003/04/10) synchronized を、削除します。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellRendererオブジェクト
	 * @og.rtnNotNull
	 */
	public CellRenderer newInstance( final DBColumn clm ) {
		return dbCell;
	}

	/**
	 * データの表示用文字列を返します。
	 * ここでは、null か ゼロ文字列の場合は、ゼロ文字列を返します。
	 * ８ケタか、１４ケタ以外のデータは、エラー表示を行います。
	 * 
	 * ８ケタの場合は、YYYYMMDD として、YYYY/MM/DD 00:00:00 を返します。
	 * １４ケタの場合は、YYYY/MM/DD HH:MM:SS を返します。
	 *
	 * @og.rev 3.8.1.1 (2005/11/21) フォーマットエラー時には、class="error" を付けて表示させる。
	 * @og.rev 5.5.7.2 (2012/10/09) 処理ロジックの見直し
	 *
	 * @param   value 入力値
	 *
	 * @return  データの表示用文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final String value ) {
		return getValue( value , true );
	}

	/**
	 * データ出力用の文字列を作成します。
	 * ファイル等に出力する形式を想定しますので、HTMLタグを含まない
	 * データを返します。
	 * 基本は、#getValue( String ) をそのまま返します。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) データ出力用のレンデラー
	 *
	 * @param   value 入力値
	 *
	 * @return  データ出力用の文字列
	 * @og.rtnNotNull
	 * @see		#getValue( String )
	 */
	@Override
	public String getWriteValue( final String value ) {
		return getValue( value , false );
	}

	/**
	 * データ表示用/出力用の文字列を作成します。
	 * 第二引数の isView == true で、データ表示用文字列を、false で
	 * データ出力用の文字列を作成します。
	 * 処理の共通化を行うためのメソッドです。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) ロジックの共通化
	 *
	 * @param   value  入力値
	 * @param   isView データ表示用かどうか(true:表示用/false:出力用)
	 *
	 * @return  データ表示用/出力用の文字列
	 * @og.rtnNotNull
	 * @see		#getValue( String )
	 */
	private String getValue( final String value , final boolean isView ) {
//		// 6.3.9.1 (2015/11/27) A method should have only one exit point, and that should be the last statement in the method.(PMD)
//		final StringBuilder rtn = new StringBuilder(19);
//
//		if( value != null && !value.isEmpty() ) {
//			rtn.append( value );
//
//			// xxxx/xx/xx xx:xx:xx
//			if( value.length() == 14 ) {
//				rtn.append( value )
//					.insert( 16,':' )
//					.insert( 13,':' )
//					.insert( 10,' ' )
//					.insert(  7,'/' )
//					.insert(  4,'/' );
//			}
//			else if( value.length() == 8 ) {
//				rtn.append( value )
//					.insert(  7,'/' )
//					.insert(  4,'/' )
//					.append( " 00:00:00" );
//			}
//			else {
//				if( isView ) {
//					rtn.append( "<span class=\"error\">" ).append( value ).append( "</span>" );
//				}
//			}
//		}
//		return rtn.toString();

		if( value == null || value.isEmpty() ) { return ""; }

		// 6.3.9.1 (2015/11/27) Found 'DD'-anomaly for variable(PMD)
		String rtn = value;

		char[] ch1 = new char[] { '0','0','0','0','/','0','0','/','0','0',' ','0','0',':','0','0',':','0','0' };
		if( rtn.length() == 8 || rtn.length() == 14 ) {
			ch1[0] = rtn.charAt(0);
			ch1[1] = rtn.charAt(1);
			ch1[2] = rtn.charAt(2);
			ch1[3] = rtn.charAt(3);

			ch1[5] = rtn.charAt(4);
			ch1[6] = rtn.charAt(5);

			ch1[8] = rtn.charAt(6);
			ch1[9] = rtn.charAt(7);

			if( rtn.length() == 14 ) {
				ch1[11] = rtn.charAt(8);
				ch1[12] = rtn.charAt(9);

				ch1[14] = rtn.charAt(10);
				ch1[15] = rtn.charAt(11);

				ch1[17] = rtn.charAt(12);
				ch1[18] = rtn.charAt(13);
			}

			rtn = new String( ch1 );
		}
		else {
			if( isView ) {
				rtn = "<span class=\"error\">" + value + "</span>";
			}
		}
		return rtn;
	}
}
