/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import static org.opengion.fukurou.util.StringUtil.nval;

// import java.io.IOException;
// import java.io.ObjectInputStream;
// import java.io.ObjectOutputStream;

import org.opengion.hayabusa.html.ViewAjaxTreeTableParam;

/**
 * viewタグの viewFormType が HTMLAjaxTreeTable の場合にパラメータを設定します。
 *
 * ツリービューを表示する、ViewForm_HTMLAjaxTreeTable クラスに対して、各種パラメータを
 * 設定します。
 * パラメータが設定されていない場合は、ViewAjaxTreeTableParam の初期値が使用されます。
 * (パラメータを使用するには、viewタグのuseParam 属性をtrueに設定する必要があります。)
 *
 * ここで定義された子データを取得するためのJSPは、別に作成する必要があります。
 *
 * また、levelClmに設定されたカラムは、レベルと展開を表示するための画像イメージが表示されます。
 * 展開レベルに応じて階層を視覚的に表示する場合は、levelClmに設定されたカラムのレンデラーを
 * 定義して下さい。
 * (例:LVLのコードリソースとして、1→- 2→-- 3→--- 等)
 *
 * 各属性は、{&#064;XXXX} 変数が使用できます。
 * これは、ServletRequest から、XXXX をキーに値を取り出し,この変数に割り当てます。
 * つまり、このXXXXをキーにリクエストすれば、この変数に値をセットすることができます。
 *
 * http://localhost/query.jsp?KEY1=VLA1&amp;KEY2=VAL2
 *
 * のようなリクエストで、{&#064;KEY1} とすれば、 VAL1 がセットされます。
 *
 * @og.formSample
 * ●形式：&lt;og:ajaxTreeParam keys="YYY,XXX,ZZZ" childJsp="getChildTag.jsp" /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:ajaxTreeParam
 *       childSearchKeys    【TAG】子データを取得するためのJSPに渡す引数のカラム名をセットします(初期値:"")
 *       childSearchJsp     【TAG】子データを取得するためのJSPをセットします(初期値:"getChildTag.jsp")
 *       levelClm           【TAG】階層レベルとして処理(展開を制御するための画像イメージを表示)するカラム名をセットします(初期値:"LVL")
 *       imgCollapsed       【TAG】折りたたまれた状態を表現するための画像ファイル名をセットします(初期値:"/jsp/image/collapsed.gif")
 *       imgExpanded        【TAG】展開された状態を表現するための画像ファイル名をセットします(初期値:"/jsp/image/expanded.gif")
 *       imgNoSub           【TAG】展開後子データが存在しない状態を表現するための画像ファイル名をセットします(初期値:"/jsp/image/nosub.gif")
 *       expandAll          【TAG】初期状態で全展開を行うかを指定します(初期値:false=全展開しない)
 *       childViewStartNo   【TAG】子データを表示するためのStartNoを指定します(初期値:-1)
 *       expCtrlClm         【TAG】初期全展開した際に最下位の要素の状態(展開済 or 下位あり)を制御するためのカラムを指定します(初期値:EXPAND_CONTROL)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *     ViewFormTag の viewFormType が、HTMLAjaxTreeTable の場合に使用します。
 *     useParam 属性を設定しておかないと、使用されません。
 *     &lt;og:view
 *         viewFormType = &quot;HTMLAjaxTreeTable&quot;
 *         command      = &quot;{&#064;command}&quot;
 *         startNo      = &quot;0&quot;
 *         pageSize     = &quot;20&quot;
 *         <b>useParam     = &quot;true&quot;</b>
 *     &gt;
 *         &lt;og:ajaxTreeParam
 *             childSearchkeys = "YYY,XXX,ZZZ"
 *             childSearchJsp  = "getChildTag.jsp"
 *             levelClm        = "LVL"
 *             imgCollapsed    = "collapsed.gif"
 *             imgExpanded     = "expanded.gif"
 *             imgNoSub        = "nosub.gif"
 *        /&gt;
 *     &lt;/og:view &gt;
 *
 * @og.group 画面表示
 *
 * @version  4.0
 * @author	 Hiroki Nakamura
 * @since    JDK5.0,
 */
// public class ViewAjaxTreeParamTag extends ViewParamTag {
public class ViewAjaxTreeParamTag extends ViewParamImpl {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.3.5.0 (2008/02/01)" ;

	private static final long serialVersionUID = 435020080201L ;

	/**
	 * 【TAG】子データを取得するためのJSPに渡す引数のカラム名をセットします(初期値:"")。
	 *
	 * @og.tag
	 * 子データを取得するためのJSPに渡す引数のカラム名をセットします。
	 * 親データから子データを展開する際に、ここで設定されたキーの値をDBTableModelから
	 * 取得し、子データ検索用のJSPに渡します。
	 * 子データ検索用のJSPでは通常のリクエスト変数({&#064;カラム名})として値を取得することが可能です。
	 *
	 * @param	keys 子データ検索カラム
	 */
	public void setChildSearchKeys( final String keys ) {
		putParam( ViewAjaxTreeTableParam.CHILD_SEARCH_KEYS,
				  nval( getRequestParameter( keys ),"" ) );
	}

	/**
	 * 【TAG】子データを取得するためのJSPをセットします(初期値:"getChildTag.jsp")。
	 *
	 * @og.tag
	 * 子データを取得するためのJSPをセットします。
	 * このJSPで子データを取得し、viewタグでHTML出力した結果をパースし、子データとして
	 * 親データの直下に要素を挿入します。
	 * 子データ検索用のJSPでは通常のリクエスト変数({&#064;カラム名})として値を取得することが可能です。
	 *
	 * @param	jsp JSPファイル名
	 */
	public void setChildSearchJsp( final String jsp ) {
		putParam( ViewAjaxTreeTableParam.CHILD_SEARCH_JSP,
				  nval( getRequestParameter( jsp ),"getChildTag.jsp" ) );
	}

	/**
	 * 【TAG】階層レベルとして処理(展開を制御するための画像イメージを表示)するカラム名をセットします(初期値:"LVL")。
	 *
	 * @og.tag
	 * 階層レベルとして処理(展開を制御するための画像イメージを表示)するカラム名をセットします。
	 * 子データを取得した際に、ここで指定されたカラムは、親レベルに対して自動的に+1されます。
	 *
	 * @param	clm レベルカラム
	 */
	public void setLevelClm( final String clm ) {
		putParam( ViewAjaxTreeTableParam.LVL_CLM_KEY,
				  nval( getRequestParameter( clm ),"LVL" ) );
	}

	/**
	 * 【TAG】折りたたまれた状態を表現するための画像ファイル名をセットします(初期値:"/jsp/image/collapsed.gif")。
	 *
	 * @og.tag
	 * 折りたたまれた状態を表現するための画像ファイル名をセットします。
	 * 「折りたたまれた状態」とは、未展開の状態、及び展開後に折りたたんだ状態の両方を指します。
	 *
	 * @param	img 画像ファイル名
	 */
	public void setImgCollapsed( final String img ) {
		putParam( ViewAjaxTreeTableParam.IMG_COLLAPSED,
				  nval( getRequestParameter( img ),"collapsed.gif" ) );
	}

	/**
	 * 【TAG】展開された状態を表現するための画像ファイル名をセットします(初期値:"/jsp/image/expanded.gif")。
	 *
	 * @og.tag
	 * 展開された状態を表現するための画像ファイル名をセットします。
	 *
	 * @param	img 画像ファイル名
	 */
	public void setImgExpanded( final String img ) {
		putParam( ViewAjaxTreeTableParam.IMG_EXPANDED,
				  nval( getRequestParameter( img ),"expanded.gif" ) );
	}

	/**
	 * 【TAG】展開後子データが存在しない状態を表現するための画像ファイル名をセットします(初期値:"/jsp/image/nosub.gif")。
	 *
	 * @og.tag
	 * 展開後子データが存在しない状態を表現するための画像ファイル名をセットします。
	 *
	 * @param	img 画像ファイル名
	 */
	public void setImgNoSub( final String img ) {
		putParam( ViewAjaxTreeTableParam.IMG_NO_SUB,
				  nval( getRequestParameter( img ),"nosub.gif" ) );
	}

	/**
	 * 【TAG】初期状態で全展開を行うかを指定します(初期値:false=全展開しない)。
	 *
	 * @og.tag
	 * 初期状態で全展開を行うかを指定します。DBTableModel上のデータが展開済みのデータ
	 * であると仮定し、全展開を行った状態で表示します。
	 * ※ この属性をtrueに指定しても子データ取得用のJSPが再帰的に呼び出される訳では
	 *    ありません。子データの取得は、予めqueryで行っておく必要があります。
	 *
	 * @og.rev 4.3.3.0 (2008/10/01) 新規作成
	 *
	 * @param	flag 全展開を行うか [true:行う/false:行わない]
	 */
	public void setExpandAll( final String flag ) {
		putParam( ViewAjaxTreeTableParam.EXPAND_ALL,
				  nval( getRequestParameter( flag ),"false" ) );
	}

	/**
	 * 【TAG】子データを表示するためのStartNoを指定します(初期値:-1)。
	 *
	 * @og.tag
	 * 子データを取得するためのJSP内で使用します。
	 * 子データを取得し、元のDBTableModelにマージした後、子データのみを表示するため、
	 * ここでstartNoを指定して差分データを表示します。
	 *
	 * @og.rev 4.3.3.0 (2008/10/01) 新規作成
	 *
	 * @param	no 子データ表示開始番号
	 */
	public void setChildViewStartNo( final String no ) {
		putParam( ViewAjaxTreeTableParam.CHILD_VIEW_START_NO,
				  nval( getRequestParameter( no ),"-1" ) );
	}

	/**
	 * 【TAG】初期全展開した際に最下位の要素の状態(展開済 or 下位あり)を制御するためのカラムを指定します(初期値:EXPAND_CONTROL)。
	 *
	 * @og.tag
	 * 初期全展開した際に最下位の要素の状態(展開済 or 下位あり)を制御するためのカラムを指定します。
	 * ここで指定されたカラムの値により、全展開された結果の最下位の要素の状態指定することができます。
	 * この機能を利用することで、第3レベルまで存在するデータの内、第2レベルまでを展開しておく、
	 * ということを実現することができます。
	 * ここで指定したカラムが持つ値の意味は以下の通りです。
	 *  '0' or 'false' 展開済 (初期値)
	 *  '1' or 'true'  未展開 ※展開ボタンを押すことで更に下位展開が可能
	 *
	 * なお、この機能は、全展開時(expandAll="true")のみ有効であり、通常の第1レベル展開の場合は、
	 * このカラムは無視されます。
	 *
	 * @og.rev 4.3.5.0 (2008/02/01) 新規作成
	 *
	 * @param	key 最下位展開 [0,false:展開済/1,true:未展開]
	 */
	public void setExpCtrlClm( final String key ) {
		putParam( ViewAjaxTreeTableParam.EXPAND_CONTROL_CLM_KEY,
				  nval( getRequestParameter( key ),"EXPAND_CONTROL" ) );
	}

	/**
	 * タグの名称を、返します。
	 * 自分自身のクラス名より、自動的に取り出せないため、このメソッドをオーバーライドします。
	 *
	 * @og.rev 4.3.1.0 (2008/09/03) 新規追加
	 *
	 * @return  タグの名称
	 * @og.rtnNotNull
	 */
	@Override
	protected String getTagName() {
		return "ajaxTreeParam" ;
	}

//	/**
//	 * シリアライズ用のカスタムシリアライズ書き込みメソッド。
//	 *
//	 * @og.rev 4.3.1.0 (2008/09/03) 新規追加
//	 * @serialData 一部のオブジェクトは、シリアライズされません。
//	 *
//	 * @param	strm	ObjectOutputStreamオブジェクト
//	 * @throws IOException	入出力エラーが発生した場合
//	 */
//	private void writeObject( final ObjectOutputStream strm ) throws IOException {
//		strm.defaultWriteObject();
//	}

//	/**
//	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
//	 *
//	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
//	 *
//	 * @og.rev 4.3.1.0 (2008/09/03) 新規追加
//	 * @serialData 一部のオブジェクトは、シリアライズされません。
//	 *
//	 * @param	strm	ObjectInputStreamオブジェクト
//	 * @see #release2()
//	 * @throws IOException	シリアライズに関する入出力エラーが発生した場合
//	 * @throws ClassNotFoundException	クラスを見つけることができなかった場合
//	 */
//	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
//		strm.defaultReadObject();
//	}
}
