/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.table;

/**
 * TableFilter_TABLE_FIREBIRD は、TableUpda インターフェースを継承した、DBTableModel 処理用の
 * 実装クラスです。とくに、Firebird用のDB定義スクリプトを作成します。
 *
 * ここでは、テーブル一覧の検索結果より、GF05 のテーブルカラム定義テーブルから
 * 必要な情報を取得し、テーブル作成スクリプトを作成します。
 * 出力ファイルは、テーブル名＋"S.sql" という命名規則で作成します。
 * 検索では、(SYSTEM_ID,TBLSYU,TABLE_NAME,NAME_JA,TABLESPACE_NAME,INITIAL_EXTENT,COMMENTS)
 * の項目を取得する必要があります。
 *
 * @og.rev 5.1.1.0 (2009/12/01) 新規作成
 *
 * @version  0.9.0  2000/10/17
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.1,
 */
public class TableFilter_TABLE_FIREBIRD extends TableFilter_TABLE {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.0.2.3 (2014/10/10)" ;

	/**
	 * 各行部分(カラム定義)の処理を実行します。
	 *
	 * @og.rev 5.1.1.0 (2009/12/01) 各カラムの説明を追加
	 * @og.rev 5.3.9.0 (2011/09/01) PRIMARY KEY の作成は、INDEX側で行うこととする。
	 * @og.rev 5.5.8.2 (2012/11/09) NUMBER の小数の扱いを、DECIMAL にする。
	 * @og.rev 5.6.9.2 (2013/10/18) useLen が、nullやゼロ文字列の場合の考慮
	 *
	 * @param	data	１行分のデータ配列
	 * @param	first	最初の行かどうか[true:最初/false:それ以降]
	 *
	 * @return	各行部分(カラム定義)配列
	 */
	@Override
	protected String[] makeLineList( final String[] data,final boolean first ) {

		final String clsName = data[GF05_CLS_NAME] ;
		String useLen  = data[GF05_USE_LENGTH] ;
		if( useLen != null && useLen.contains( "." ) ) { useLen = useLen.replace( '.' , ',' ); }

		data[GF05_USE_LENGTH] = null;				// 標準では、桁数をクリアしておく。

		// 5.1.1.0 (2009/12/01) 各カラムの説明文を 入れたくない場合は、null を設定する。
		if( isXml ) { data[GF05_NAME_JA] = null; }

		if( "UNIQ".equalsIgnoreCase( data[GF05_CLM] ) || "UNIQSEQ".equalsIgnoreCase( data[GF05_CLM] ) ) {
			data[GF05_CLS_NAME]     = "BIGINT";
			data[GF05_DATA_DEFAULT] = null;
			data[GF05_NOT_NULL]     = "1";
		}
		else if( clsName == null || clsName.isEmpty() || clsName.startsWith( "VARCHAR" ) ) {		// 6.1.0.0 (2014/12/26) refactoring
			data[GF05_CLS_NAME]   = "VARCHAR" ;
			data[GF05_USE_LENGTH] = useLen ;
		}
		else if( clsName.startsWith( "CLOB" ) ) {
			data[GF05_CLS_NAME]     = "BLOB SUB_TYPE 1";
			data[GF05_DATA_DEFAULT] = null;
			data[GF05_NOT_NULL]     = null;
		}
		// 5.6.9.2 (2013/10/18) useLen が、nullやゼロ文字列の場合の考慮
		else if( clsName.startsWith( "NUMBER" ) && useLen != null && useLen.length() > 0 ) {

			if( useLen.contains( "," ) ) {		// カンマがある
				data[GF05_CLS_NAME] = "DECIMAL";		// 5.5.8.2 (2012/11/09) 小数は、DECIMAL
				data[GF05_USE_LENGTH] = useLen ;
			}
			else if( Integer.parseInt( useLen ) < 10 ) {	// １０桁以下
				data[GF05_CLS_NAME] = "INTEGER";		// 5.5.8.2 (2012/11/09) NUMBER のデフォルトは、INTEGER
			}
			else if( Integer.parseInt( useLen ) < 18 ) {	// １８桁以下
				data[GF05_CLS_NAME] = "BIGINT";
			}
		}

		// 5.5.8.2 (2012/11/09) 4000 文字なら、VARCHAR のままで登録できる。
	//	// 暫定追加
	//	if( "4000".equalsIgnoreCase( data[GF05_USE_LENGTH] ) ) {
	//		data[GF05_CLS_NAME]     = "BLOB";
	//		data[GF05_DATA_DEFAULT] = null;
	//		data[GF05_NOT_NULL]     = null;
	//		data[GF05_USE_LENGTH]   = null;
	//	}

		return super.makeLineList( data,first );
	}

	/**
	 * 定義の最後の部分の処理を実行します。
	 *
	 * @og.rev 6.0.2.3 (2014/10/10) isXml で、CR + EXEC_END_TAG のキャッシュを作成します。
	 *
	 * @param	clmNo	カラム番号配列
	 * @param	data	１行分のデータ配列
	 *
	 * @return	定義の最後の部分
	 * @og.rtnNotNull
	 */
	@Override
	protected String makeEndLine( final int[] clmNo,final String[] data ) {
		return ")" + execEndTag ;						// 6.0.2.3 (2014/10/10) 改行を含める。
	}

	/**
	 * ユニークシーケンスの作成処理を実行します。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) シーケンス名を[TABLE_NAME]S00に変更
	 * @og.rev 6.0.2.3 (2014/10/10) isXml で、CR + EXEC_END_TAG のキャッシュを作成します。
	 *
	 * @param	clmNo	カラム番号配列
	 * @param	data	１行分のデータ配列
	 *
	 * @return	ユニークシーケンス
	 * @og.rtnNotNull
	 */
	@Override
	protected String makeUniqSeq( final int[] clmNo,final String[] data ) {
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );

		buf.append( CR );
		if( isXml ) { buf.append( EXEC_START_TAG ).append( CR ); }

		// 5.1.9.0 (2010/08/01) シーケンス名を[TABLE_NAME]S00に変更
		buf.append( "CREATE SEQUENCE " ).append( data[clmNo[TABLE_NAME]] ).append( "S00 " )
			// 6.0.2.3 (2014/10/10) isXml で、CR + EXEC_END_TAG のキャッシュを使用
			.append( execEndTag );

		return buf.toString();
	}

	/**
	 * ユニークシーケンスと関連付けるトリガの作成処理を実行します。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) トリガー名を[TABLE_NAME]T00に変更
	 *
	 * @param	clmNo		カラム番号配列
	 * @param	data		１行分のデータ配列
	 * @param   uniqName	ユニークトリガ名
	 *
	 * @return	ユニークシーケンスと関連付けるトリガ
	 * @og.rtnNotNull
	 */
	@Override
	protected String makeUniqTrig( final int[] clmNo,final String[] data, final String uniqName ) {
		final String TBL_NAME = data[clmNo[TABLE_NAME]] ;
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
			.append( CR );
		if( isXml ) { buf.append( EXEC_START_TAG ).append( CR ); }

		// 5.1.9.0 (2010/08/01) トリガー名を[TABLE_NAME]T00に変更
		buf.append( "CREATE TRIGGER " ).append( TBL_NAME )
			.append( "T00 FOR ").append( TBL_NAME ).append( CR )
			.append( "  BEFORE INSERT AS ").append( CR )
			.append( "  BEGIN " ).append( CR )
			.append( "    NEW." ).append( uniqName )
				// 5.1.9.0 (2010/08/01) シーケンス名を[TABLE_NAME]S00に変更
			.append( " = GEN_ID( " ).append( TBL_NAME ).append( "S00,1 );" ).append( CR )
			.append( "  END; " ).append( CR );

		if( isXml )	{ buf.append( EXEC_END_TAG ); }		// ここはキャッシュではなく、EXEC_END_TAG を使う。

		return buf.toString();
	}
}
