/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.io;

import javax.swing.tree.TreeModel;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeNode;
import javax.swing.tree.DefaultMutableTreeNode;
import java.io.File;

/**
 * 内部に TreeModel を持ったファイル階層表現を表すクラスです。
 * ルートディレクトリを指定して、それ以下のディレクトリ階層を構築します。
 * このクラスは，すべてのデータを走査してから、TreeModel を構築しますので、
 * パフォーマンス的には劣ります。
 *
 * @og.group その他出力
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class FileTreeModel {
	private TreeModel	model	;			// 6.3.9.1 (2015/11/27) 修飾子を、なし → private に変更。
	private String		dir		;			// 6.3.9.1 (2015/11/27) 修飾子を、なし → private に変更。

	/**
	 * デフォルトコンストラクター
	 *
	 */
	public FileTreeModel() {
		initialise( null );
	}

	/**
	 * ルートディレクトリを指定して、TreeModel を構築するコンストラクター
	 *
	 * @param   dir ルートディレクトリ文字列
	 */
	public FileTreeModel( final String dir ) {
		initialise( dir );
	}

	/**
	 * ルートディレクトリを指定して、TreeModel を構築します。
	 *
	 * @param   dir ルートディレクトリ文字列
	 */
	public void setDirectory( final String dir ) {
		initialise( dir );
	}

	/**
	 * ルートディレクトリを指定して、TreeModel を構築します。
	 *
	 * @param   dir ルートディレクトリ文字列
	 */
	private void initialise( final String dir ) {
		this.dir = dir;
		if( this.dir == null ) { this.dir = "."; }
		final TreeNode root = makeTree( new File(this.dir) );
		model = new DefaultTreeModel( root );
	}

	/**
	 * TreeModel を取得します。
	 * コンストラクター または、setDirectory()メソッドによって構築された
	 * ディレクトリ階層を TreeModel にマッピングして返します。
	 *
	 * @return	ルートディレクトリ文字列
	 */
	public TreeModel getTreeModel() {
		return model;
	}

	/**
	 * 内部的に ディレクトリ階層を表現した TreeNode を返します。
	 *
//	 * @param	file	ルートディレクトリのファイルオブジェクト
	 * @param	dir		ルートディレクトリのファイルオブジェクト
	 *
	 * @return	ディレクトリ階層を表現したTreeNode
	 */
	private DefaultMutableTreeNode makeTree( final File dir ) {
		final DefaultMutableTreeNode node = new DefaultMutableTreeNode( dir.getName() );
		if( dir.isDirectory() ) {
			final String[] list = dir.list();
			// 6.3.9.0 (2015/11/06) null になっている可能性がある(findbugs)
			if( list != null ) {
				for( final String file : list ) {
					node.add( makeTree( new File( dir, file ) ) );
				}
			}
//			for( int i=0; i<list.length; i++ ) {
//				node.add( makeTree( new File(dir, list[i]) ) );
//			}
		}
		return node;
	}

	/**
	 * Tree の表示用メソッド
	 *
	 * これは、テスト用に使用するための Tree を標準出力に 出力するメソッドです。
	 *
	 * @param    root	トップレベルのTreeNodeオブジェクト(階層的に印字します。)
	 * @param    model	TreeNodeを含む TreeModelオブジェクト
	 * @param    level	階層レベル。一番トップを 0 レベルとする。
	 */
	public void printTree( final TreeNode root,final TreeModel model,final int level ) {
		final int num = model.getChildCount( root );
		TreeNode[] nodes = new TreeNode[num];
		for( int i=0; i<num; i++ ) {
			nodes[i] = (TreeNode)model.getChild( root,i );
			if( nodes[i].isLeaf() ) {
				System.out.println( level + ":" + nodes[i].toString() );
			}
			else {
				System.out.println( level + ":" + nodes[i].toString() );
				printTree( nodes[i],model,level+1 );
			}
		}
	}

	/**
	 * main メソッド
	 *
	 * これは、テスト用に使用するための main メソッドです。
	 *
	 * @param  args  起動時の引数 args[0] にルートディレクトリ名
	 */
	public static void main( final String[] args ) {
		final FileTreeModel fmodel = new FileTreeModel( args[0] );
		final TreeModel model = fmodel.getTreeModel();
		final TreeNode root = (TreeNode)model.getRoot() ;
		fmodel.printTree( root,model,0 );
	}
}
