/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.remote;

import java.util.Map;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.resource.ResourceFactory;
import org.opengion.hayabusa.resource.ResourceManager;
import static org.opengion.fukurou.util.HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring

/**
 * RemoteControllableインタフェイスを実装した
 * サーブレット経由で遠隔リソース更新を行うためのクラスです。
 *
 * POSTキーとしてcommandとCLM、langが必要です。
 * commandは更新リソースの種別(COLUMN,LABEL,CODE,GUI)
 * CLMはCLM値をカンマで区切ったCSV形式
 * langは更新対象の言語(例：ja)です。
 * 詳しくはremoteControlメソッドのJavaDocをご覧下さい。
 *
 * @og.rev 4.1.0.0 (2007/12/20) 新規作成
 *
 * @version  4.1
 * @author   Masakazu Takahashi
 * @since    JDK6.0,
 *
 */
public class ClearResource implements RemoteControllable {
	/**
	 * RemoteControllableインタフェイスの実装メソッドです。
	 * ClearResourceでは受け取ったパラメータによってコンテキストのリソースの再読込をします。
	 *
	 * POSTキーとして受け取るキーと値は次の通りとなります
	 * <table border="1" frame="box" rules="all" >
	 *   <caption>POSTキーとして受け取るキーと値</caption>
	 *   <tr><th>キー		</th><th>内容		</th><th>値								 </th></tr>
	 *   <tr><td>command	</td><td>更新種別	</td><td>COLUMN,LABEL,CODE,GUI			 </td></tr>
	 *   <tr><td>lang		</td><td>言語		</td><td>更新リソースの言語				 </td></tr>
	 *   <tr><td>CLM		</td><td>更新キー	</td><td>キーカラムを","で区切ったCSV形式</td></tr>
	 * </table>
	 *
	 * 動作はcommand == "GUI"の場合とそれ以外の場合に分かれます。
	 * 但し、リソースの更新はResourceManagerのメソッドを利用する部分は共通です。
	 * この時、langによって更新対象の言語を選択します(nullの場合はja)
	 * ①commandが"GUI"の場合
	 * 　画面リソースを更新する場合はcommand="GUI"で渡します。
	 * 　GUIに限ってCLMパラメータは不要です。
	 * 　ResourceManager.guiClear()がコールされて画面リソースがクリアされます。
	 * 　(同時に画面リソースのラベルリソースも再読込します)
	 * ②commandが"GUI"以外の場合
	 * 　commandが"GUI"以外の場合の動作はどれも同じです。
	 * 　受け取ったCLMパラメータをCSV分解し、ループで回して
	 * 　ResourceManager.clear(key)をコールします。
	 * 　
	 * 返す値は XML形式の文字列です。
	 * &lt;clear-resource&gt;
	 *   &lt;command&gt;command引数&lt;/command&gt;
	 *   &lt;lang&gt;lang引数&lt;/lang&gt;
	 *   &lt;keys&gt;
	 *     &lt;key&gt;CLM引数の更新キー１&lt;/key&gt;
	 *     &lt;key&gt;CLM引数の更新キー２&lt;/key&gt;
	 *     ...
	 *   &lt;/keys&gt;
	 * &lt;/clear-resource&gt;
	 * となります。
	 *
	 * @param	valMap	 サーブレットが受け取ったキーと値のマップ
	 *
	 * @return	XML形式の実行結果
	 * @og.rtnNotNull
	 */
	public String remoteControl( final Map<String,String> valMap ) {
		final String 			command = valMap.get( "command" );
		final String 			clms	= valMap.get( "CLM" );
		final String 			lang	= valMap.get( "lang" );
		final ResourceManager rscM	= ResourceFactory.newInstance( lang );

		final StringBuilder rtnStr = new StringBuilder( BUFFER_MIDDLE )
			.append( "<clear-resource><command>" ).append( command )
			.append( "</command><lang>"	).append( lang )
			.append( "</lang><keys>" );

		if( "GUI".equals(command) ) {
			rscM.guiClear();
			rtnStr.append( " <key>").append( clms ).append( "</key>" );
		}
		else {
			final String[] vals = StringUtil.csv2Array( clms );
			for( int i=0; i<vals.length; i++ ) {
				rscM.clear( vals[i] );
				rtnStr.append( " <key>" ).append( vals[i] ).append( "</key>" );
			}
		}

		rtnStr.append( " </keys></clear-resource>" );
		return rtnStr.toString();
	}
}
