/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.TableFilter;
import org.opengion.fukurou.db.Transaction;
// import org.opengion.fukurou.db.TransactionReal;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)

import static org.opengion.fukurou.util.StringUtil.nval ;

// import java.io.ObjectOutputStream;
// import java.io.ObjectInputStream;
// import java.io.IOException;
import java.util.Map;

/**
 * TableFilter のサブクラスをCALLしてDBTableModelにアクセスするタグです。
 *
 * DBTableModel を TableFilter のサブクラス(classIdで指定)に渡して処理を実行します。
 * クラスを作成する場合は、org.opengion.hayabusa.db.TableFilter インターフェースを継承した
 * クラスにする必要があります。また、classId 属性には、システムリソース で
 * 設定した TableFilter.XXXX の XXXX を指定します。
 *
 * BODY部分は、SQLを記述する為だけに使っていましたが、CSS定義形式の書式で、keys,vals を記述
 * できるようにします。
 * これは、下記のようなパラメータを、keys="KEY,KEY2,KEY3" vals='AAAA,"BB,CC,DD",EE' のような記述形式と
 *   {
 *        KEY1 : AAAA ;
 *        KEY2 : BB,CC,DD ;
 *        KEY3 : EE ;
 *        ・・・・・・
 *   }
 * のような、CSS形式に類似の形式でも記述できるようにしました。
 * keys,vals と CSS定義形式パラメータを同時に指定した場合は、両方とも有効です。
 * ただし、キーが重複した場合は、不定と考えてください。
 * 現時点では、CSS定義形式パラメータが優先されますが、これは、単に内部パラメータMapへの
 * 登録順が、CSS定義形式パラメータが後の為、上書きされるためです。
 *
 * ※ このタグは、Transaction タグの対象です。
 *
 * @og.formSample
 * ●形式：&lt;og:tableFilter classId="…" /&gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:tableFilter
 *       classId          ○【TAG】データベース処理を実行するクラスパスを指定します(必須)。
 *       tableId            【TAG】(通常は使いません)DBTableModel sessionに登録されているキーを指定します
 *       modifyType         【TAG】データ処理の方法(A:追加 C:更新 D:削除)を指定します
 *       keys               【TAG】リンク先に渡すキーを指定します
 *       vals               【TAG】keys属性に対応する値をCSV形式で複数指定します
 *       selectedAll        【TAG】データを全件選択済みとして処理するかどうか[true/false]を指定します(初期値:false)
 *       stopZero           【TAG】検索結果が０件のとき処理を続行するかどうか[true/false]を指定します(初期値:false[続行する])
 *       scope              【TAG】キャッシュする場合のスコープ[request/page/session/applicaton]を指定します(初期値:session)
 *       dbid               【TAG】(通常は使いません)Queryオブジェクトを作成する時のDB接続IDを指定します
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null) 5.7.7.2 (2014/06/20)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null) 5.7.7.2 (2014/06/20)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:判定しない) 5.7.7.2 (2014/06/20)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:判定しない) 5.7.7.2 (2014/06/20)
 *       caseIf             【TAG】指定の値が、true/TRUE文字列の場合は、このタグは使用されます(初期値:判定しない)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:tableFilter&gt;
 *
 * ●使用例
 *    ・引数/プロシジャーを直接書く場合
 *    【entry.jsp】
 *        &lt;og:tableFilter
 *            classId     = "WL_LOGICSET"         :TableFilter のサブクラス(実行クラス)
 *            tableId     = "WL0000"              :登録元のDBTableModelのsession/request変数内の取得キー
 *            keys        = "AA,BB,CC"            :実行クラスへの引数のキー
 *            vals        = "{&#064;AA},{&#064;BB},{&#064;CC}"   :実行クラスへの引数の値
 *            selectedAll = "false/true"          :処理対象の行を全行選択するかどうか(初期値:false)
 *            modifyType  = "A/C/D"               :処理の方法(A:追加 C:更新 D:削除)を指定します。初期値は自動です。
 *        /&gt;
 *
 *    ・BODY部分に、CSS形式のパラメータ（keys,vals）を記述する例
 *
 *        &lt;og:tableFilter
 *            classId     = "WL_LOGICSET"         :TableFilter のサブクラス(実行クラス)
 *            tableId     = "WL0000"              :登録元のDBTableModelのsession/request変数内の取得キー
 *            selectedAll = "false/true"          :処理対象の行を全行選択するかどうか(初期値:false)
 *            modifyType  = "A/C/D"               :処理の方法(A:追加 C:更新 D:削除)を指定します。初期値は自動です。
 *        &gt;
 *               {
 *                   AA    :  {&#064;AA}
 *                   BB    :  {&#064;BB}
 *                   CC    :  {&#064;CC}
 *               }
 *        &lt;/og:tableFilter&gt;
 *
 * @og.group その他
 * @og.rev 3.8.5.0 (2006/03/20) 新規作成
 *
 * @version  0.9.0  2000/10/17
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.1,
 */
public class TableFilterTag extends CommonTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.3.9.0 (2015/11/06)" ;
	private static final long serialVersionUID = 639020151106L ;

	private static final String errMsgId	= HybsSystem.ERR_MSG_KEY;
	private transient DBTableModel	table	;

	private String		tableId		= HybsSystem.TBL_MDL_KEY;
	private String		classId		;
	private String		modifyType	;
	private String[]	keys		;
	private String[]	vals		;

	private   String	dbid		; // 4.2.4.0 (2008/06/23)
	private   String	sql			; // 5.6.5.2 (2013/06/21) bodyからSQL文のみを切り出す。
	private   Map<String,String>  paramMap;	// 5.6.5.2 (2013/06/21) bodyからparamMapを取りだし。

	private boolean		selectedAll	;
	private boolean		stopZero	;	// 5.7.6.2 (2014/05/16) stopZero属性追加

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 5.7.7.2 (2014/06/20) caseKey,caseVal,caseNN,caseNull 属性を追加
	 *
	 * @return	後続処理の指示( EVAL_BODY_BUFFERED )
	 */
	@Override
	public int doStartTag() {
		// 5.7.7.2 (2014/06/20) caseKey,caseVal,caseNN,caseNull 属性を追加
		if( !useTag() ) { return SKIP_BODY ; }

		table = (DBTableModel)getObject( tableId );

		if( keys != null && vals != null && keys.length != vals.length ) {
			final String errMsg = "keys と vals の設定値の数が異なります。: " + CR
						+ "keys.length=[" + keys.length + "] , "
						+ "keys.length=[" + StringUtil.array2line( keys,"," ) + "]"
						+ CR
						+ "vals.length=[" + vals.length + "] , "
						+ "vals.length=[" + StringUtil.array2line( vals,"," ) + "]";
			throw new HybsSystemException( errMsg );
		}

		startQueryTransaction( tableId );
		return EVAL_BODY_BUFFERED ;		// Body を評価する
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 5.6.5.2 (2013/06/21) bodyローカル化、sql、paramMap 追加
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		final String body = nval( getBodyString(),null );

		// paramMapの取り出し
		paramMap = StringUtil.cssParse( body );

		// SQL文の切り出し classId="DBSELECT" の場合のみの処理
		if( "DBSELECT".equalsIgnoreCase( classId ) && body != null ) {
			final int ad1 = body.indexOf( '{' );
			final int ad2 = body.indexOf( '}' );

			if( ad1 >= 0 && ad2 >= 0 ) {
				sql = body.substring( 0,ad1 ).trim() + body.substring( ad2+1 ).trim();
			}
			else {
				sql = body.trim();
			}
		}

		return SKIP_BODY ;
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 4.2.3.0 (2008/06/23) DBIDとボディー部分の記述を下位クラスに渡す用に修正
	 * @og.rev 4.3.7.4 (2009/07/01) Resouceオブジェクトを下位クラスに渡す用に修正
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応
	 * @og.rev 5.2.1.0 (2010/10/01) debugPrint() メソッドの処理条件見直し
	 * @og.rev 5.3.7.0 (2011/07/01) TransactionReal の引数変更 、Transaction対応で、close処理を入れる。
	 * @og.rev 5.6.5.2 (2013/06/21) bodyローカル化、sql、paramMap 追加
	 * @og.rev 5.7.6.2 (2014/05/16) table件数が変わる場合、"DB.COUNT" キーでリクエストに再セットする。
	 * @og.rev 5.7.7.2 (2014/06/20) caseKey,caseVal,caseNN,caseNull 属性を追加
	 * @og.rev 6.0.4.0 (2014/11/28) NullPointerException が発生するので、事前にチェックします。
	 * @og.rev 6.3.6.1 (2015/08/28) Transaction でAutoCloseableを使用したtry-with-resources構築に対応。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		// デバッグ時には、オブジェクト内部情報を表示する。
		debugPrint();	// 5.2.1.0 (2010/10/01) debugPrint() メソッド自体に、isDebug() が組み込まれている。

		// 5.7.7.2 (2014/06/20) caseKey,caseVal,caseNN,caseNull 属性を追加
		if( !useTag() ) { return EVAL_PAGE ; }

		int rtnCode = EVAL_PAGE;	// try ～ finally の関係で、変数化しておく

		final int[] rowNo = getParameterRows();

		// 5.1.9.0 (2010/08/01) Transaction 対応
//		Transaction tran = null;
		final TableFilter filter ;
		// 5.3.7.0 (2011/07/01) Transaction対応で、close処理を入れる。
//		try {
//			final TransactionTag tranTag = (TransactionTag)findAncestorWithClass( this,TransactionTag.class );
//			if( tranTag == null ) {
//				tran = new TransactionReal( getApplicationInfo() );		// 5.3.7.0 (2011/07/01) 引数変更
//			}
//			else {
//				tran = tranTag.getTransaction();
//			}

		// 6.3.6.1 (2015/08/28) Transaction でAutoCloseableを使用したtry-with-resources構築に対応。
		try( final Transaction tran = getTransaction() ) {
			// 5.7.6.2 (2014/05/16) table件数が変わる場合、"DB.COUNT" キーでリクエストに再セットする。
			final int rowCnt1 = table == null ? -1 : table.getRowCount();

			final String cls = HybsSystem.sys( "TableFilter_" + classId );

			// NullPointerException が発生するので、事前にチェックします。
			if( cls == null ) {
				final String errMsg = "TableFilter クラスが見つかりません。class=" + "TableFilter_" + classId;
				throw new RuntimeException( errMsg );
			}

			filter = (TableFilter)HybsSystem.newInstance( cls );

			filter.setDBTableModel( table );
			filter.setParameterRows( rowNo );
			filter.setModifyType( modifyType );
			filter.setKeysVals( keys,vals );
	//		filter.setApplicationInfo( getApplicationInfo() );	// 3.8.7.0 (2006/12/15)
			filter.setTransaction( tran );						// 5.1.9.0 (2010/08/01) Transaction 対応
			filter.setDebug( isDebug() );
			filter.setDbid( dbid );					// 4.2.4.0 (2008/06/23)
			filter.setSql( sql );					// 5.6.5.2 (2013/06/21) sql 追加
			filter.setParamMap( paramMap );			// 5.6.5.2 (2013/06/21) paramMap 追加
			filter.setResource( getResource() );	// 4.3.7.4 (2009/07/01)

			table = filter.execute();

			// 5.7.6.2 (2014/05/16) table件数が変わる場合、"DB.COUNT" キーでリクエストに再セットする。
			final int rowCnt2 = table == null ? -1 : table.getRowCount();
			if( rowCnt1 != rowCnt2 ) {
				setRequestAttribute( "DB.COUNT" , String.valueOf( rowCnt2 ) );
			}

			final int errCode = filter.getErrorCode();
			final ErrorMessage errMessage = filter.getErrorMessage();

			if( errCode >= ErrorMessage.NG )  { 	// 異常
				rtnCode = SKIP_PAGE;
			}

			// 5.7.6.2 (2014/05/16) 件数０件(または、table==null)かつ stopZero = true
			if( rowCnt2 <= 0 && stopZero ) { return SKIP_PAGE; }

			final String err = TaglibUtil.makeHTMLErrorTable( errMessage,getResource() );
			if( err != null && err.length() > 0 ) {
				jspPrint( err );
				setSessionAttribute( errMsgId,errMessage );
			}
			else {
				removeSessionAttribute( errMsgId );
			}
			tran.commit();				// 6.3.6.1 (2015/08/28)
		}
//		finally {
//			if( tran != null ) { tran.close(); }
//		}

		if( table != null && ! commitTableObject( tableId, table ) ) {
			rtnCode = SKIP_PAGE ;
		}

		return rtnCode ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 5.6.5.2 (2013/06/21) body廃止、sql、paramMap 追加
	 * @og.rev 5.7.6.2 (2014/05/16) stopZero属性追加
	 */
	@Override
	protected void release2() {
		super.release2();
		table		= null;
		tableId		= HybsSystem.TBL_MDL_KEY;
		classId		= null;
		modifyType	= null;
		keys		= null;
		vals		= null;
		selectedAll	= false;
		stopZero	= false;	// 5.7.6.2 (2014/05/16) stopZero属性追加
		dbid		= null;		// 4.2.4.0 (2008/06/23)
		sql			= null;		// 5.6.5.2 (2013/06/21) bodyからSQL文のみを切り出す。
		paramMap	= null;		// 5.6.5.2 (2013/06/21) bodyからparamMapを取りだす。
	}

	/**
	 * 表示データの HybsSystem.ROW_SEL_KEY を元に、選ばれた 行を処理の対象とします。
	 *
	 * @og.rev 6.3.9.0 (2015/11/06) コンストラクタで初期化されていないフィールドを null チェックなしで利用している(findbugs)
	 *
	 * @return	選択行の配列
	 * @og.rtnNotNull
	 */
	@Override
	protected int[] getParameterRows() {
		final int[] rowNo ;
		// 6.3.9.0 (2015/11/06) コンストラクタで初期化されていないフィールドを null チェックなしで利用している(findbugs)
//		if( selectedAll ) {
		if( selectedAll && table != null ) {
			final int rowCnt = table.getRowCount();
			rowNo = new int[rowCnt];
			for( int i=0; i<rowCnt; i++ ) {
				rowNo[i] = i;
			}
		} else {
			rowNo = super.getParameterRows();		// 4.0.0 (2005/01/31)
		}
		return rowNo;
	}

	/**
	 * 【TAG】データベース処理を実行するテーブルフィルターのクラスIDを指定します。
	 *
	 * @og.tag
	 * ここで指定するクラスIDは、システムリソース にて TableFilter の
	 * サブクラス(インターフェース継承)として指定する必要があります。
	 *
	 * クラス自身は、org.opengion.hayabusa.db.TableFilter インターフェースを継承している必要があります。
	 * {@og.doc03Link tableFilter TableFilter_**** クラス}
	 *
	 * @param	id 実行クラスID
	 * @see		org.opengion.hayabusa.db.TableFilter  TableFilter インターフェース
	 */
	public void setClassId( final String id ) {
		classId = nval( getRequestParameter( id ),classId );
	}

	/**
	 * 【TAG】(通常は使いません)結果のDBTableModelを、sessionに登録するときのキーを指定します
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @og.tag
	 * 検索結果より、DBTableModelオブジェクトを作成します。これを、下流のviewタグ等に
	 * 渡す場合に、通常は、session を利用します。その場合の登録キーです。
	 * query タグを同時に実行して、結果を求める場合、同一メモリに配置される為、
	 * この tableId 属性を利用して、メモリ空間を分けます。
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @param	id テーブルID (sessionに登録する時のID)
	 */
	public void setTableId( final String id ) {
		tableId = nval( getRequestParameter( id ),tableId );
	}

	/**
	 * 【TAG】データを全件選択済みとして処理するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * 全てのデータを選択済みデータとして扱って処理します。
	 * 全件処理する場合に、指定します。(true/false)
	 * デフォルト false です。
	 *
	 * @param  all 全件選択済み [true:全件選択済み/false:通常]
	 */
	public void setSelectedAll( final String all ) {
		selectedAll = nval( getRequestParameter( all ),selectedAll );
	}

	/**
	 * 【TAG】検索結果が０件のとき処理を停止するかどうか[true/false]を指定します(初期値:false[続行する])。
	 *
	 * @og.tag
	 * 初期値は、false(続行する)です。
	 *
	 * @og.rev 5.7.6.2 (2014/05/16) 新規追加
	 *
	 * @param  cmd ０件時停止可否 [true:処理を中止する/false:続行する]
	 */
	public void setStopZero( final String cmd ) {
		stopZero = nval( getRequestParameter( cmd ),stopZero );
	}

	/**
	 * 【TAG】データ処理の方法[A:追加/C:更新/D:削除]を指定します(初期値:自動判定)。
	 *
	 * @og.tag
	 * 通常は、DBTableModel に自動設定されている modifyType を元に、データ処理方法を
	 * 選別します。(A:追加 C:更新 D:削除)
	 * この場合、行単位で modifyType の値を取得して判別する必要がありますが、一般には
	 * 処理対象は、全件おなじ modifyType である可能性が高いです。
	 * また、selectedAll などで強制的に全件処理対象とする場合は、modifyType に値が
	 * 設定さていません。その様な場合に外部より modifyType を指定します。
	 * 初期値は、自動判定 です。
	 *
	 * @param  type 処理方法 [A:追加/C:更新/D:削除]
	 */
	public void setModifyType( final String type ) {
		modifyType = nval( getRequestParameter( type ),modifyType );

		if( modifyType != null && !"A".equals( modifyType ) && !"C".equals( modifyType ) && !"D".equals( modifyType ) ) {
			final String errMsg = "modifyType は A:追加 C:更新 D:削除 のどれかを指定してください。: " + CR
						+ "modifyType=[" + modifyType + "]";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】リンク先に渡すキーを指定します。
	 *
	 * @og.tag
	 * 戻る時に、検索時のキャッシュに指定した引数以外に指定したり、別の値に置き換えたり
	 * する場合のキーを設定できます。CSV形式で複数指定できます。
	 * vals 属性には、キーに対応する値を、設定してください。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @param	key リンク先に渡すキー
	 */
	public void setKeys( final String key ) {
		keys = getCSVParameter( key );
	}

	/**
	 * 【TAG】names属性に対応する値をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * キーに設定した値を、CSV形式で複数して出来ます。
	 * 指定順序は、キーと同じにしておいて下さい。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @param	val names属性に対応する値 (CSV形式)
	 */
	public void setVals( final String val ) {
		vals = getCSVParameter( val );
	}

	/**
	 * 【TAG】(通常は使いません)Queryオブジェクトを作成する時のDB接続IDを指定します。
	 *
	 * @og.tag
	 * Queryオブジェクトを作成する時のDB接続IDを指定します。
	 * これは、システムリソースで、DEFAULT_DB_URL 等で指定している データベース接続先
	 * 情報に、XX_DB_URL を定義することで、 dbid="XX" とすると、この 接続先を使用して
	 * データベースにアクセスできます。
	 *
	 * @param	id データベース接続ID
	 */
	public void setDbid( final String id ) {
		dbid = nval( getRequestParameter( id ),dbid );
	}

//	/**
//	 * シリアライズ用のカスタムシリアライズ書き込みメソッド。
//	 *
//	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
//	 * @serialData 一部のオブジェクトは、シリアライズされません。
//	 *
//	 * @param	strm	ObjectOutputStreamオブジェクト
//	 * @throws IOException	入出力エラーが発生した場合
//	 */
//	private void writeObject( final ObjectOutputStream strm ) throws IOException {
//		strm.defaultWriteObject();
//	}

//	/**
//	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
//	 *
//	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
//	 *
//	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
//	 * @serialData 一部のオブジェクトは、シリアライズされません。
//	 *
//	 * @param	strm	ObjectInputStreamオブジェクト
//	 * @see #release2()
//	 * @throws IOException	シリアライズに関する入出力エラーが発生した場合
//	 * @throws ClassNotFoundException	クラスを見つけることができなかった場合
//	 */
//	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
//		strm.defaultReadObject();
//	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION		)
				.println( "tableId"			,tableId		)
				.println( "classId"			,classId		)
				.println( "modifyType"		,modifyType		)
				.println( "selectedAll"		,selectedAll	)
				.println( "keys"			,keys			)
				.println( "vals"			,vals			)
				.println( "dbid"			,dbid			) // 4.2.4.0 (2008/06/23)
				.println( "sql"				,sql			) // 5.6.5.2 (2013/06/21)
				.fixForm().toString() ;
	}
}
