/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.db.AbstractDBType;
import org.opengion.hayabusa.db.DBTypeCheckUtil;

/**
 * 全角のみで構成される文字列を扱う為の、カラム属性を定義します。
 *
 * 全角文字とは、「c &lt; 0x7f || ( 0xff65 &lt;= c &amp;&amp; c &lt; 0xffa0 ) 以外」
 * の文字で構成される文字列のことです。
 *
 * タイプチェックとして、以下の条件を判定します。
 * ・全角文字チェック「c &lt; 0x7f || ( 0xff65 &lt;= c &amp;&amp; c &lt; 0xffa0 ) 以外」エラー
 * ・文字列長は、Byte換算での文字数との比較
 * ・文字パラメータの 正規表現チェック
 *
 * @og.group データ属性
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DBType_K extends AbstractDBType {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.2.2.0 (2010/11/01)" ;

	/**
	 * 半角スペースで固定長(半角換算の数)に変換した文字列を返します。
	 * 半角スペース埋めは、文字が半角、全角混在でもかまいません。
	 * なお、エラーチェックは行われません。
	 * 実行前に、必ず valueCheck( String value ,int len ) が行われる必要があります。
	 *
	 * @og.rev 3.5.4.5 (2004/01/23) エンコード指定に変更します。
	 *
	 * @param	value	 ＦＩＬＬ埋めする文字列
	 * @param	sizeX	 整数部分の文字列の長さ
	 * @param	sizeY	 小数部分の文字列の長さ
	 * @param   encode   固定長で変換する文字エンコード
	 *
	 * @return	ＦＩＬＬ埋めした新しい文字列
	 */
	@Override
	public String valueFill( final String value ,final int sizeX ,final int sizeY,final String encode ) {
		final int len = (sizeY == 0) ? sizeX : sizeX + sizeY + 1;

		return StringUtil.stringKFill( value,len,encode );
	}

	/**
	 * データが登録可能かどうかをチェックします。
	 * データがエラーの場合は、そのエラー内容を返します。
	 *
	 * @og.rev 3.0.1.3 (2003/03/11) DBTypeCheckUtilクラスを利用するように修正
	 * @og.rev 3.6.0.0 (2004/09/22) dbType パラメータを引数に追加
	 * @og.rev 5.2.2.0 (2010/11/01) 厳密にチェック(isStrict=true)するフラグを追加
	 *
	 * @param   key		キー
	 * @param   value	値
	 * @param   sizeX	整数部分の文字列の長さ
	 * @param   sizeY	小数部分の文字列の長さ
	 * @param   typeParam	dbType パラメータ
	 * @param   isStrict	厳密にチェックするかどうか[true:する/false:標準的]
	 *
	 * @return  エラー内容
	 * @og.rtnNotNull
	 */
	@Override
	public ErrorMessage valueCheck( final String key ,final String value ,
									final int sizeX ,final int sizeY ,final String typeParam ,final boolean isStrict) {

		final ErrorMessage msg = new ErrorMessage();
		if( value == null || value.isEmpty() ) { return msg; }

		final int len = (sizeY == 0) ? sizeX : sizeX + sizeY + 1;
		String check = DBTypeCheckUtil.byteLengthCheck( value,len );
		if( check != null ) {
			// 文字列の長さが指定の長さよりも長いです。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0006",key,value,check,String.valueOf( len ) );
		}

		final char[] chs = value.toCharArray() ;
		for( int i=0; i<chs.length; i++ ) {
			if( chs[i] < 0x7f || ( 0xff65 <= chs[i] && chs[i] < 0xffa0 ) ) {
				// 全てが全角文字ではありません。
				msg.addMessage( 0,ErrorMessage.NG,"ERR0007",key,value );
				break;
			}
		}

		// 3.6.0.0 (2004/09/22) dbType パラメータを使用したマッチチェック
		check = DBTypeCheckUtil.matcheCheck( value,typeParam );
		if( check != null ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,check );
		}

		return msg;
	}
}
