/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.db;

import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ParameterMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Locale;

import org.opengion.fukurou.util.ApplicationInfo;
import org.opengion.fukurou.system.OgRuntimeException ;		// 6.4.2.0 (2016/01/29)
import org.opengion.fukurou.system.Closer;
import org.opengion.fukurou.util.StringUtil;
import static org.opengion.fukurou.system.HybsConst.CR;				// 6.1.0.0 (2014/12/26) refactoring

/**
 * データベース関連の便利なメソッドを集めた簡易ユーティリティークラスです。
 * 全てのメソッドは、static メソッドになっています。
 *
 * @og.rev 2.1.1.1 (2002/11/15) Serializable インターフェースを削除する。
 * @og.rev 4.0.0.0 (2007/10/16) DBアクセス関係のメソッドのみをパッケージ移動(hayabusa/db ⇒ fukurou/db)
 * @og.rev 6.0.4.0 (2014/11/28) ResultSetValue クラスで、ResultSet から値を取得する処理を移植
 *
 * @og.group ＤＢ/Shell制御
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class DBUtil {

	/**
	 * インスタンスを作らないので、コンストラクタは、private に設定します。
	 */
	private DBUtil() {}

	/**
	 * 初期データベースに接続して、Queryを実行します(互換性確保のため残しています)。
	 *
	 * ステートメントと引数により、Prepared クエリーの検索のみ実行します。
	 * 結果は,すべて文字列に変換されて格納されます。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 4.0.0.0 (2007/10/10) dbid の初期値を、"DEFAULT" から null に変更
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応します。
	 * @og.rev 5.3.7.0 (2011/07/01) TransactionReal の引数変更
	 * @og.rev 5.3.8.0 (2011/08/01) TransactionReal と close() 処理をセットで実行する。
	 * @og.rev 6.3.6.1 (2015/08/28) Transaction でAutoCloseableを使用したtry-with-resources構築に対応。
	 *
	 * @param   stmt ステートメント文字列
	 * @param   args オブジェクトの引数配列
	 * @param   appInfo アプリ情報オブジェクト
	 *
	 * @return  検索結果の配列
	 */
	public static String[][] dbExecute( final String stmt ,final String[] args ,final ApplicationInfo appInfo ) {
		try( final Transaction tran = new TransactionReal( appInfo ) ) {
			return dbExecute( stmt, args, tran, null, false );
		}

//		final Transaction tran = new TransactionReal( appInfo );			// 5.3.7.0 (2011/07/01) 引数変更
//
//		// 5.3.8.0 (2011/08/01) TransactionReal と close() 処理をセットで実行する。
//		try {
//			return dbExecute( stmt, args, tran, null, false );
//		}
//		finally {
//			// エラー発生時は、tran.rollback() が呼ばれているため、close()時にエラーフラグをセットする必要はない。
//			tran.close();
//		}
	}

	/**
	 * 初期データベースに接続して、Queryを実行します(Transaction 対応)。
	 *
	 * ステートメントと引数により、Prepared クエリーの検索のみ実行します。
	 * 結果は,すべて文字列に変換されて格納されます。
	 * ここでは、Transactionオブジェクトから、Connection を取り出して使用します。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) 新規作成 Transaction 対応
	 *
	 * @param   stmt ステートメント文字列
	 * @param   args オブジェクトの引数配列
	 * @param   tran Transactionオブジェクト
	 *
	 * @return  検索結果の配列
	 */
	public static String[][] dbExecute( final String stmt ,final String[] args ,final Transaction tran ) {
		return dbExecute( stmt, args, tran, null, false );
	}

	/**
	 * 検索するデータベースを指定して、Queryを実行します(互換性確保のため残しています)。
	 *
	 * ステートメントと引数により、Prepared クエリーの検索のみ実行します。
	 * 結果は,すべて文字列に変換されて格納されます。
	 * 追加：検索以外のSQLも実行できます。結果は、null を返します。
	 *
	 * @og.rev 3.0.0.0 (2002/12/25) 検索のみのクエリーから、何でもありのクエリーに変更
	 * @og.rev 2.3.1.3 (2003/01/28) Open Cursor が、大量に残る件の対応。ResultSet を close()
	 * @og.rev 3.8.0.8 (2005/10/03) エラーメッセージの出力順をメッセージ＋Queryに変更します。
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 4.0.0.1 (2007/12/03) try ～ catch ～ finally をきちんと行う。
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応します。
	 * @og.rev 5.3.7.0 (2011/07/01) TransactionReal の引数変更
	 * @og.rev 5.3.8.0 (2011/08/01) TransactionReal と close() 処理をセットで実行する。
	 * @og.rev 6.3.6.1 (2015/08/28) Transaction でAutoCloseableを使用したtry-with-resources構築に対応。
	 *
	 * @param   stmt ステートメント文字列
	 * @param   args オブジェクトの引数配列
	 * @param   appInfo アプリ情報オブジェクト
	 * @param   dbid 接続先ID
	 *
	 * @return  検索結果の配列
	 */
	public static String[][] dbExecute( final String stmt ,final String[] args, final ApplicationInfo appInfo, final String dbid ) {
		try( final Transaction tran = new TransactionReal( appInfo ) ) {
			return dbExecute( stmt, args, tran, dbid, false  );
		}

//		final Transaction tran = new TransactionReal( appInfo );			// 5.3.7.0 (2011/07/01) 引数変更
//
//		// 5.3.8.0 (2011/08/01) TransactionReal と close() 処理をセットで実行する。
//		try {
//			return dbExecute( stmt, args, tran, dbid, false  );
//		}
//		finally {
//			// エラー発生時は、tran.rollback() が呼ばれているため、close()時にエラーフラグをセットする必要はない。
//			tran.close();
//		}
	}

	/**
	 * 検索するデータベースを指定して、Queryを実行します(Transaction 対応)。
	 *
	 * ステートメントと引数により、Prepared クエリーの検索のみ実行します。
	 * 結果は,すべて文字列に変換されて格納されます。
	 * 追加：検索以外のSQLも実行できます。結果は、null を返します。
	 * ここでは、Transactionオブジェクトから、Connection を取り出して使用します。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) 新規作成 Transaction 対応
	 *
	 * @param   stmt ステートメント文字列
	 * @param   args オブジェクトの引数配列
	 * @param   tran Transactionオブジェクト
	 * @param   dbid 接続先ID
	 *
	 * @return  検索結果の配列
	 */
	public static String[][] dbExecute( final String stmt ,final String[] args, final Transaction tran , final String dbid ) {
		return dbExecute( stmt, args, tran, dbid, false );
	}

	/**
	 * 検索するデータベースを指定して、Queryを実行します(互換性確保のため残しています)。
	 *
	 * ステートメントと引数により、Prepared クエリーの検索のみ実行します。
	 * 結果は,すべて文字列に変換されて格納されます。
	 * 追加：検索以外のSQLも実行できます。結果は、null を返します。
	 *
	 * @og.rev 4.3.7.0 (2009/06/01) 新規作成
	 * @og.rev 5.1.2.0 (2010/01/01) setObject に ParameterMetaData の getParameterType を渡す。(PostgreSQL対応)
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応します。
	 * @og.rev 5.3.7.0 (2011/07/01) TransactionReal の引数変更
	 * @og.rev 5.3.8.0 (2011/08/01) TransactionReal と close() 処理をセットで実行する。
	 * @og.rev 6.3.6.1 (2015/08/28) Transaction でAutoCloseableを使用したtry-with-resources構築に対応。
	 *
	 * @param   stmt ステートメント文字列
	 * @param   args オブジェクトの引数配列
	 * @param   appInfo アプリ情報オブジェクト
	 * @param   dbid 接続先ID
	 * @param   useHeader 1行目にヘッダーを含めるか
	 *
	 * @return  検索結果の配列
	 */
	public static String[][] dbExecute( final String stmt ,final String[] args, final ApplicationInfo appInfo, final String dbid, final boolean useHeader ) {
		try( final Transaction tran = new TransactionReal( appInfo ) ) {
			return dbExecute( stmt, args, tran, dbid, useHeader );
		}

//		final Transaction tran = new TransactionReal( appInfo );			// 5.3.7.0 (2011/07/01) 引数変更
//
//		// 5.3.8.0 (2011/08/01) TransactionReal と close() 処理をセットで実行する。
//		try {
//			return dbExecute( stmt, args, tran, dbid, useHeader );
//		}
//		finally {
//			// エラー発生時は、tran.rollback() が呼ばれているため、close()時にエラーフラグをセットする必要はない。
//			tran.close();
//		}
	}

	/**
	 * 検索するデータベースを指定して、Queryを実行します(Transaction 対応)。
	 *
	 * ステートメントと引数により、Prepared クエリーの検索のみ実行します。
	 * 結果は,すべて文字列に変換されて格納されます。
	 * 追加：検索以外のSQLも実行できます。結果は、null を返します。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) 新規作成 Transaction 対応
	 * @og.rev 5.3.8.0 (2011/08/01) Transaction を引数で受け取った場合は、close() しない。
	 * @og.rev 5.3.8.0 (2011/08/01) useParamMetaData を ConnectionFactory経由で取得。(PostgreSQL対応)、setNull 対応
	 *
	 * @param   stmt ステートメント文字列
	 * @param   args オブジェクトの引数配列
	 * @param   tran Transactionオブジェクト
	 * @param   dbid 接続先ID
	 * @param   useHeader 1行目にヘッダーを含めるか
	 *
	 * @return  検索結果の配列
	 */
	public static String[][] dbExecute( final String stmt ,final String[] args, final Transaction tran, final String dbid, final boolean useHeader ) {
		PreparedStatement pstmt = null;
		ResultSet resultSet = null;
		String[][] rtn = null;
		try {
			final Connection conn = tran.getConnection( dbid );				// 5.1.9.0 (2010/08/01) Transaction 対応
			pstmt = conn.prepareStatement( stmt );
			if( args != null ) {
				// 5.1.1.0 (2009/11/11) setObject に ParameterMetaData の getParameterType を渡す。(PostgreSQL対応)
				final boolean useParamMetaData = ConnectionFactory.useParameterMetaData( dbid );	// 5.3.8.0 (2011/08/01)
				if( useParamMetaData ) {
					final ParameterMetaData pMeta = pstmt.getParameterMetaData();
					for( int i=0; i<args.length; i++ ) {
						final int type = pMeta.getParameterType( i+1 );
						// 5.3.8.0 (2011/08/01) setNull 対応
						final String val = args[i];
						if( val == null || val.isEmpty() ) {
							pstmt.setNull( i+1, type );
						}
						else {
							pstmt.setObject( i+1, val, type );
						}
					}
				}
				else {
					for( int i=0; i<args.length; i++ ) {
						pstmt.setObject( i+1,args[i] );
					}
				}
			}
			if( pstmt.execute() ) {									// 6.1.0.0 (2014/12/26) refactoring
				resultSet = pstmt.getResultSet();
				rtn = DBUtil.resultToArray( resultSet,useHeader );	// 4.3.7.0 (2009/06/01)
			}
			// 6.3.6.1 (2015/08/28) 検索・登録関係なく、commit() する。
//			else {
				tran.commit();			// 5.1.9.0 (2010/08/01) Transaction 対応
//			}
		}
		catch ( SQLException ex ) {
			tran.rollback();			// 5.1.9.0 (2010/08/01) Transaction 対応
			final String errMsg = ex.getMessage() + ":" + ex.getSQLState() + CR
							+ "SQL=[" + stmt + "]" + CR
							+ "ARG=[" + StringUtil.array2csv( args ) + "]" + CR
							+ "DBID=[" + dbid + "]" + CR;
			throw new OgRuntimeException( errMsg,ex );
		}
		finally {
			Closer.resultClose( resultSet );
			Closer.stmtClose( pstmt );
		}
		return rtn;
	}

	/**
	 * 初期データベースに接続して、CallableStatement(PL/SQL)を実行します(互換性確保のため残しています)。
	 * ステートメントと引数により、CallableStatement クエリーを実行します。
	 * 結果は,ステータスとエラーメッセージを返します。便宜的に、String配列に
	 * 設定して返します。
	 * ステートメント文字列には、 { call PLSQL( ?,?,?･･･ ) } となります。
	 * 第一引数、第二引数は、OUT属性で、結果(STATUS)とエラー時の内容(ERR_CODE)を返します。
	 * 第三引数以降の ? には、オブジェクトの引数配列 が順に割り当てられます。
	 *
	 * @og.rev 3.8.0.0 (2005/06/07) 新規追加
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 4.0.0.0 (2007/10/10) dbid の初期値を、"DEFAULT" から null に変更
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応します。
	 * @og.rev 5.3.7.0 (2011/07/01) TransactionReal の引数変更
	 * @og.rev 5.3.8.0 (2011/08/01) TransactionReal と close() 処理をセットで実行する。
	 * @og.rev 6.3.6.1 (2015/08/28) Transaction でAutoCloseableを使用したtry-with-resources構築に対応。
	 *
	 * @param   stmt ステートメント文字列
	 * @param   args オブジェクトの引数配列
	 * @param   appInfo アプリ情報オブジェクト
	 *
	 * @return  実行結果([0]=ステータス、[1]=エラーメッセージ
	 */
	public static String[] dbCallExecute( final String stmt ,final String[] args, final ApplicationInfo appInfo ) {
		try( final Transaction tran = new TransactionReal( appInfo ) ) {
			return dbCallExecute( stmt ,args, tran, null );
		}

//		final Transaction tran = new TransactionReal( appInfo );			// 5.3.7.0 (2011/07/01) 引数変更
//
//		// 5.3.8.0 (2011/08/01) TransactionReal と close() 処理をセットで実行する。
//		try {
//			return dbCallExecute( stmt ,args, tran, null );
//		}
//		finally {
//			// エラー発生時は、tran.rollback() が呼ばれているため、close()時にエラーフラグをセットする必要はない。
//			tran.close();
//		}
	}

	/**
	 * 初期データベースに接続して、CallableStatement(PL/SQL)を実行します(Transaction 対応)。
	 * ステートメントと引数により、CallableStatement クエリーを実行します。
	 * 結果は,ステータスとエラーメッセージを返します。便宜的に、String配列に
	 * 設定して返します。
	 * ステートメント文字列には、 { call PLSQL( ?,?,?･･･ ) } となります。
	 * 第一引数、第二引数は、OUT属性で、結果(STATUS)とエラー時の内容(ERR_CODE)を返します。
	 * 第三引数以降の ? には、オブジェクトの引数配列 が順に割り当てられます。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) 新規作成 Transaction 対応
	 *
	 * @param   stmt ステートメント文字列
	 * @param   args オブジェクトの引数配列
	 * @param   tran Transactionオブジェクト
	 *
	 * @return  実行結果([0]=ステータス、[1]=エラーメッセージ
	 */
	public static String[] dbCallExecute( final String stmt ,final String[] args, final Transaction tran ) {
		return dbCallExecute( stmt ,args, tran, null );
	}

	/**
	 * 検索するデータベースを指定して、CallableStatement(PL/SQL)を実行します(互換性確保のため残しています)。
	 * ステートメントと引数により、CallableStatement クエリーを実行します。
	 * 結果は,ステータスとエラーメッセージを返します。便宜的に、String配列に
	 * 設定して返します。
	 * ステートメント文字列には、 { call PLSQL( ?,?,?･･･ ) } となります。
	 * 第一引数、第二引数は、OUT属性で、結果(STATUS)とエラー時の内容(ERR_CODE)を返します。
	 * 第三引数以降の ? には、オブジェクトの引数配列 が順に割り当てられます。
	 * 検索するデータベースは、DEFAULT です。
	 *
	 * @og.rev 3.8.0.0 (2005/06/07) 新規追加
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 4.0.0.1 (2007/12/03) try ～ catch ～ finally をきちんと行う。
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応します。
	 * @og.rev 5.3.7.0 (2011/07/01) TransactionReal の引数変更
	 * @og.rev 5.3.8.0 (2011/08/01) TransactionReal と close() 処理をセットで実行する。
	 * @og.rev 6.3.6.1 (2015/08/28) Transaction でAutoCloseableを使用したtry-with-resources構築に対応。
	 *
	 * @param   stmt ステートメント文字列
	 * @param   args オブジェクトの引数配列
	 * @param   appInfo アプリ情報オブジェクト
	 * @param   dbid 接続先ID
	 *
	 * @return  実行結果([0]=ステータス、[1]=エラーメッセージ
	 */
	public static String[] dbCallExecute( final String stmt ,final String[] args, final ApplicationInfo appInfo ,final String dbid ) {
		try( final Transaction tran = new TransactionReal( appInfo ) ) {
			return dbCallExecute( stmt ,args, tran, dbid );
		}

//		final Transaction tran = new TransactionReal( appInfo );			// 5.3.7.0 (2011/07/01) 引数変更
//
//		// 5.3.8.0 (2011/08/01) TransactionReal と close() 処理をセットで実行する。
//		try {
//			return dbCallExecute( stmt ,args, tran, dbid );
//		}
//		finally {
//			// エラー発生時は、tran.rollback() が呼ばれているため、close()時にエラーフラグをセットする必要はない。
//			tran.close();
//		}
	}

	/**
	 * 検索するデータベースを指定して、CallableStatement(PL/SQL)を実行します(Transaction 対応)。
	 * ステートメントと引数により、CallableStatement クエリーを実行します。
	 * 結果は,ステータスとエラーメッセージを返します。便宜的に、String配列に
	 * 設定して返します。
	 * ステートメント文字列には、 { call PLSQL( ?,?,?･･･ ) } となります。
	 * 第一引数、第二引数は、OUT属性で、結果(STATUS)とエラー時の内容(ERR_CODE)を返します。
	 * 第三引数以降の ? には、オブジェクトの引数配列 が順に割り当てられます。
	 * 検索するデータベースは、DEFAULT です。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) 新規作成 Transaction 対応
	 * @og.rev 5.3.8.0 (2011/08/01) Transaction を引数で受け取った場合は、close() しない。
	 *
	 * @param   stmt ステートメント文字列
	 * @param   args オブジェクトの引数配列
	 * @param   tran Transactionオブジェクト
	 * @param   dbid 接続先ID
	 *
	 * @return  実行結果([0]=ステータス、[1]=エラーメッセージ
	 */
	public static String[] dbCallExecute( final String stmt ,final String[] args, final Transaction tran ,final String dbid ) {
		CallableStatement callStmt = null ;

		String[] rtn = new String[2] ;

		try {
			final Connection conn = tran.getConnection( dbid );				// 5.1.9.0 (2010/08/01) Transaction 対応
			callStmt = conn.prepareCall( stmt );

			callStmt.registerOutParameter( 1, Types.INTEGER );
			callStmt.registerOutParameter( 2, Types.VARCHAR );
			if( args != null ) {
				for( int i=0; i<args.length; i++ ) {
					callStmt.setObject( i+3,args[i] );
				}
			}
			callStmt.execute();

			rtn[0] = String.valueOf( callStmt.getInt(1) );	// 結果ステータス
			rtn[1] = callStmt.getString(2);					// 内容(エラーメッセージ)

			tran.commit();					// 5.1.9.0 (2010/08/01) Transaction 対応
		}
		catch ( SQLException ex ) {
			tran.rollback();				// 5.1.9.0 (2010/08/01) Transaction 対応
			final String errMsg = ex.getMessage() + ":" + ex.getSQLState() + CR
							+ "SQL=[" + stmt + "]" + CR
							+ "ARG=[" + StringUtil.array2csv( args ) + "]" + CR
							+ "DBID=[" + dbid + "]" + CR;
			throw new OgRuntimeException( errMsg,ex );
		}
		finally {
			Closer.stmtClose( callStmt );
			// 5.3.8.0 (2011/08/01) Transaction を引数で受け取った場合は、close() しない。
		}
		return rtn;
	}

	/**
	 * SQL文の実行結果において、データの件数を取得します(互換性確保のため残しています)。
	 * ステートメントと引数により、Prepared クエリーの検索を実行します。
	 * 結果は、件数を数値で返します。
	 * あくまで、存在チェックに必要な処理のみ行っているため、通常の検索より高速です。
	 *
	 * @og.rev 3.5.0.0 (2003/09/17) 新規作成
	 * @og.rev 3.8.0.8 (2005/10/03) エラーメッセージの出力順をメッセージ＋Queryに変更します。
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 4.0.0.1 (2007/12/03) try ～ catch ～ finally をきちんと行う。
	 * @og.rev 5.1.2.0 (2010/01/01) setObject に ParameterMetaData の getParameterType を渡す。(PostgreSQL対応)
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応します。
	 * @og.rev 5.3.7.0 (2011/07/01) TransactionReal の引数変更
	 * @og.rev 5.3.8.0 (2011/08/01) TransactionReal と close() 処理をセットで実行する。
	 * @og.rev 6.3.6.1 (2015/08/28) Transaction でAutoCloseableを使用したtry-with-resources構築に対応。
	 *
	 * @param   stmt ステートメント文字列
	 * @param   args オブジェクトの引数配列
	 * @param   appInfo アプリ情報オブジェクト
	 * @param   dbid 接続先ID
	 *
	 * @return  検索結果(データの件数)
	 */
	public static int dbExist( final String stmt ,final String[] args, final ApplicationInfo appInfo , final String dbid ) {
		try( final Transaction tran = new TransactionReal( appInfo ) ) {
			return dbExist( stmt ,args, tran , dbid );
		}

//		final Transaction tran = new TransactionReal( appInfo );			// 5.3.7.0 (2011/07/01) 引数変更
//
//		// 5.3.8.0 (2011/08/01) TransactionReal と close() 処理をセットで実行する。
//		try {
//			return dbExist( stmt ,args, tran , dbid );
//		}
//		finally {
//			// エラー発生時は、tran.rollback() が呼ばれているため、close()時にエラーフラグをセットする必要はない。
//			tran.close();
//		}
	}

	/**
	 * SQL文の実行結果において、データの件数を取得します(Transaction 対応)。
	 * ステートメントと引数により、Prepared クエリーの検索を実行します。
	 * 結果は、件数を数値で返します。
	 * あくまで、存在チェックに必要な処理のみ行っているため、通常の検索より高速です。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) 新規作成 Transaction 対応
	 * @og.rev 5.3.8.0 (2011/08/01) Transaction を引数で受け取った場合は、close() しない。
	 * @og.rev 5.3.8.0 (2011/08/01) useParamMetaData を ConnectionFactory経由で取得。(PostgreSQL対応)、setNull 対応
	 *
	 * @param   stmt ステートメント文字列
	 * @param   args オブジェクトの引数配列
	 * @param   tran Transactionオブジェクト
	 * @param   dbid 接続先ID
	 *
	 * @return  検索結果(データの件数)
	 */
	public static int dbExist( final String stmt ,final String[] args, final Transaction tran , final String dbid ) {
		PreparedStatement pstmt = null;
		ResultSet resultSet = null;
		int rtnCnt = -1;

		try {
			final Connection conn = tran.getConnection( dbid );				// 5.1.9.0 (2010/08/01) Transaction 対応
			pstmt = conn.prepareStatement( stmt );
			if( args != null ) {
				// 5.1.2.0 (2010/01/01) setObject に ParameterMetaData の getParameterType を渡す。(PostgreSQL対応)
				final boolean useParamMetaData = ConnectionFactory.useParameterMetaData( dbid );	// 5.3.8.0 (2011/08/01)
				if( useParamMetaData ) {
					final ParameterMetaData pMeta = pstmt.getParameterMetaData();
					for( int i=0; i<args.length; i++ ) {
						final int type = pMeta.getParameterType( i+1 );
						// 5.3.8.0 (2011/08/01) setNull 対応
						final String val = args[i];
						if( val == null || val.isEmpty() ) {
							pstmt.setNull( i+1, type );
						}
						else {
							pstmt.setObject( i+1, val, type );
						}
					}
				}
				else {
					for( int i=0; i<args.length; i++ ) {
						pstmt.setObject( i+1,args[i] );
					}
				}
			}

			resultSet = pstmt.executeQuery();
			if( resultSet.next() ) {
				rtnCnt = resultSet.getInt(1);
			}
			tran.commit();				// 6.3.6.1 (2015/08/28) 検索・登録関係なく、commit() する。
		}
		catch ( SQLException ex ) {
			tran.rollback();			// 6.3.6.1 (2015/08/28) 検索・登録関係なく処理する。
			final String errMsg = ex.getMessage() + ":" + ex.getSQLState() + CR
							+ "SQL=[" + stmt + "]" + CR
							+ "ARG=[" + StringUtil.array2csv( args ) + "]" + CR
							+ "DBID=[" + dbid + "]" + CR;
			throw new OgRuntimeException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}
		finally {
			Closer.resultClose( resultSet );
			Closer.stmtClose( pstmt );
		}
		return rtnCnt;
	}

	/**
	 * ResultSet より、結果の文字列配列を作成します。
	 *
	 * 結果は,すべて文字列に変換されて格納されます。
	 * 移動したメソッドで使われているのでこれも移動
	 *
	 * @og.rev 3.1.0.0 (2003/03/20) Vector を使用している箇所で、非同期でも構わない箇所を、ArrayList に置換え。
	 * @og.rev 3.8.0.8 (2005/10/03) エラーメッセージの出力順をメッセージ＋Queryに変更します。
	 * @og.rev 4.0.0.0 (2005/01/31) private ⇒ public , ヘッダー情報の取得有無フラグの追加
	 * @og.rev 5.6.7.0 (2013/07/27) CLOB 対応
	 * @og.rev 6.0.4.0 (2014/11/28) ResultSetValue クラスで、ResultSet から値を取得する処理を行う。
	 *
	 * @param   resultSet ResultSetオブジェクト
	 * @param   useHeader true:ヘッダーを第一行に含める/false:含めない
	 *
	 * @return  ResultSetの検索結果配列
	 * @throws	java.sql.SQLException データベース・アクセス・エラーが発生した場合
	 */
	public static String[][] resultToArray( final ResultSet resultSet,final boolean useHeader ) throws SQLException {
		final ArrayList<String[]> data = new ArrayList<>();

		final ResultSetValue rsv = new ResultSetValue( resultSet );

		if( useHeader ) { data.add( rsv.getNames() ); }

		while( rsv.next() ) {
			data.add( rsv.getValues() );
		}

		final int size = data.size();
		String[][] rtn = new String[size][];
		for( int i=0; i<size; i++ ) {
			rtn[i] = data.get(i);
		}

		return rtn;
	}

	/**
	 * コネクションオブジェクトからデータベースのProductNameを取り出します。
	 * ProductName は、小文字化して返します。
	 * また、処理エラーが発生した場合は、"none" を返します。
	 * ここでは、SQLException は、発生させません。
	 *
	 * @og.rev 5.6.7.0 (2013/07/27) 新規追加
	 * @og.rev 5.6.7.4 (2013/08/30) ProductNameの小文字化対応
	 *
	 * @param conn コネクションオブジェクト
	 *
	 * @return データベースのProductName
	 */
	public static String getProductName( final Connection conn ) {
		String dbName ;
		try {
			dbName = conn.getMetaData().getDatabaseProductName().toLowerCase( Locale.JAPAN );	// 5.6.7.4 (2013/08/30)
		}
		catch( SQLException ex ) {
			dbName = "none";
		}
		return dbName ;
	}
}
