/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.process;

import org.opengion.fukurou.system.OgRuntimeException ;		// 6.4.2.0 (2016/01/29)
import org.opengion.fukurou.util.Argument;
import org.opengion.fukurou.util.SystemParameter;
import org.opengion.fukurou.system.LogWriter;

import org.opengion.fukurou.util.HybsEntry ;
import org.opengion.fukurou.system.Closer;
import org.opengion.fukurou.util.StringUtil;		// 5.7.2.3 (2014/01/31)
import org.opengion.fukurou.db.ConnectionFactory;

import java.util.Map ;
import java.util.LinkedHashMap ;
import java.util.Locale ;

import java.sql.Connection;
import java.sql.Statement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;

/**
 * Process_DBReaderは、データベースから読み取った内容を、LineModel に設定後、
 * 下流に渡す、FirstProcess インターフェースの実装クラスです。
 *
 * データベースから読み取った内容より、LineModelを作成し、下流(プロセス
 * チェインは、チェインしているため、データは上流から下流へと渡されます。)
 * に渡します。ここで指定できるのは、検索系SQL のみです。
 *
 * データベース接続先等は、ParamProcess のサブクラス(Process_DBParam)に
 * 設定された接続(Connection)を使用します。
 *
 * 引数文字列中にスペースを含む場合は、ダブルコーテーション("") で括って下さい。
 * 引数文字列の 『=』の前後には、スペースは挟めません。必ず、-key=value の様に
 * 繋げてください。
 *
 * SQL文には、{&#064;DATE.YMDH}等のシステム変数が使用できます。
 *
 * @og.formSample
 *  Process_DBReader -dbid=DBGE -sql="select * from GEA08"
 *
 *   [ -dbid=DB接続ID       ] ：-dbid=DBGE (例: Process_DBParam の -configFile で指定する DBConfig.xml ファイルで規定)
 *   [ -sql=検索SQL文       ] ：-sql="select * from GEA08"
 *   [ -sqlFile=検索SQLﾌｧｲﾙ ] ：-sqlFile=select.sql
 *                                 -sql= を指定しない場合は、ファイルで必ず指定してください。
 *   [ -sql_XXXX=固定値     ] ：-sql_SYSTEM_ID=GE
 *                                SQL文中の{&#064;XXXX}文字列を指定の固定値で置き換えます。
 *                                WHERE SYSTEM_ID='{&#064;SYSTEM_ID}' ⇒ WHERE SYSTEM_ID='GE'
 *   [ -asClms=置換ｶﾗﾑ名    ] ：-asClms="FGJ:CDJ SEQ123:UNIQ" 元ｶﾗﾑ名:新ｶﾗﾑ名 のｽﾍﾟｰｽ区切り
 *   [ -fetchSize=100       ] ：フェッチする行数(初期値:100)
 *   [ -display=[false/true]] ：結果を標準出力に表示する(true)かしない(false)か(初期値:false[表示しない])
 *   [ -debug=[false/true]  ] ：デバッグ情報を標準出力に表示する(true)かしない(false)か(初期値:false[表示しない])
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Process_DBReader extends AbstractProcess implements FirstProcess {
	private static final String SQL_KEY  = "sql_" ;

	private Connection	connection	;
	private Statement	stmt		;
	private ResultSet	resultSet	;
	private LineModel	newData		;
	private int			count		;
	private int			fetchSize	= 100;

	private String		dbid		;
	private boolean		display		;		// false:表示しない
	private boolean		debug		;		// 5.7.3.0 (2014/02/07) デバッグ情報

	private static final Map<String,String> MUST_PROPARTY   ;		// ［プロパティ］必須チェック用 Map
	private static final Map<String,String> USABLE_PROPARTY ;		// ［プロパティ］整合性チェック Map

	static {
		MUST_PROPARTY = new LinkedHashMap<>();

		USABLE_PROPARTY = new LinkedHashMap<>();
		USABLE_PROPARTY.put( "dbid",	"Process_DBParam の -configFile で指定する DBConfig.xml ファイルで規定" );
		USABLE_PROPARTY.put( "sql",		"検索SQL文(sql or sqlFile 必須)例: \"select * from GEA08\"" );
		USABLE_PROPARTY.put( "sqlFile",	"検索SQLファイル(sql or sqlFile 必須)例: select.sql" );
		USABLE_PROPARTY.put( "sql_",		"SQL文中の{&#064;XXXX}文字列を指定の固定値で置き換えます。" +
									CR + "WHERE SYSTEM_ID='{&#064;SYSTEM_ID}' ⇒ WHERE SYSTEM_ID='GE'" );
		// 5.7.2.3 (2014/01/31) asClms 追加
		USABLE_PROPARTY.put( "asClms",	"元ｶﾗﾑ名:新ｶﾗﾑ名 のｽﾍﾟｰｽ区切りでｶﾗﾑ名の置換を行う" );
		USABLE_PROPARTY.put( "fetchSize","フェッチする行数 (初期値:100)" );
		USABLE_PROPARTY.put( "display",	"結果を標準出力に表示する(true)かしない(false)か" +
										CR + "(初期値:false:表示しない)" );
		USABLE_PROPARTY.put( "debug",	"デバッグ情報を標準出力に表示する(true)かしない(false)か" +
										CR + "(初期値:false:表示しない)" );		// 5.7.3.0 (2014/02/07) デバッグ情報
	}

	/**
	 * デフォルトコンストラクター。
	 * このクラスは、動的作成されます。デフォルトコンストラクターで、
	 * super クラスに対して、必要な初期化を行っておきます。
	 *
	 */
	public Process_DBReader() {
		super( "org.opengion.fukurou.process.Process_DBReader",MUST_PROPARTY,USABLE_PROPARTY );
	}

	/**
	 * プロセスの初期化を行います。初めに一度だけ、呼び出されます。
	 * 初期処理(ファイルオープン、ＤＢオープン等)に使用します。
	 *
	 * @og.rev 5.7.2.2 (2014/01/24) SQL実行エラーを少し詳細に出力します。
	 * @og.rev 5.7.2.3 (2014/01/31) asClms 追加
	 *
	 * @param   paramProcess データベースの接続先情報などを持っているオブジェクト
	 */
	public void init( final ParamProcess paramProcess ) {
		final Argument arg = getArgument();

		String sql		= arg.getFileProparty("sql","sqlFile",true);

		// 5.7.2.3 (2014/01/31) asClms 追加
		final String asClms	= arg.getProparty("asClms");

		final String fSize	= arg.getProparty("fetchSize");
		display			= arg.getProparty("display",display);
		debug			= arg.getProparty("debug",debug);				// 5.7.3.0 (2014/02/07) デバッグ情報

		dbid			= arg.getProparty("dbid");
		connection		= paramProcess.getConnection( dbid );

		// 3.8.0.1 (2005/06/17) SQL文の {@XXXX} 文字列の固定値への置き換え
		final HybsEntry[] entry	=arg.getEntrys(SQL_KEY);		//配列
		final SystemParameter sysParam = new SystemParameter( sql );
		sql = sysParam.replace( entry );

		// SQL文の {@XXXX} 文字列の固定値への置き換え
		if( fSize != null ) { fetchSize = Integer.parseInt( fSize ); }

		try {
			stmt = connection.createStatement();
			if( fetchSize > 0 ) { stmt.setFetchSize( fetchSize ); }
			resultSet = stmt.executeQuery( sql );

			// 5.7.2.3 (2014/01/31) asClms 処理を追加。
			newData = createLineModel( resultSet,asClms );

			if( display ) { println( newData.nameLine() ); }
		}
		catch (SQLException ex) {
			// 5.7.2.2 (2014/01/24) SQL実行エラーを少し詳細に出力します。
			final String errMsg = "Query の実行に問題があります。" + CR
								+ "errMsg=[" + ex.getMessage() + "]" + CR
								+ "errCode=[" + ex.getErrorCode() + "] State=[" + ex.getSQLState() + "]" + CR
								+ "dbid=[" + dbid + "]" + CR
								+ "sql =[" + sql + "]" ;
			throw new OgRuntimeException( errMsg,ex );
		}
	}

	/**
	 * プロセスの終了を行います。最後に一度だけ、呼び出されます。
	 * 終了処理(ファイルクローズ、ＤＢクローズ等)に使用します。
	 *
	 * @og.rev 4.0.0.0 (2007/11/27) commit,rollback,remove 処理を追加
	 *
	 * @param   isOK トータルで、OKだったかどうか[true:成功/false:失敗]
	 */
	public void end( final boolean isOK ) {
		final boolean flag1 = Closer.resultClose( resultSet );
		resultSet  = null;
		final boolean flag2 = Closer.stmtClose( stmt );
		stmt       = null;

		ConnectionFactory.remove( connection,dbid );

		if( !flag1 || !flag2 ) {
			final String errMsg = "ステートメントをクローズ出来ません。";
			throw new OgRuntimeException( errMsg );
		}
	}

	/**
	 * このデータの処理において、次の処理が出来るかどうかを問い合わせます。
	 * この呼び出し１回毎に、次のデータを取得する準備を行います。
	 *
	 * @og.rev 5.7.2.2 (2014/01/24) SQL実行エラーを少し詳細に出力します。
	 *
	 * @return	処理できる:true / 処理できない:false
	 */
	public boolean next() {
		try {
			return resultSet.next() ;
		}
		catch (SQLException ex) {
			final String errMsg = "ネクストすることが出来ません。"
								+ "errMsg=[" + ex.getMessage() + "]" + CR
								+ "errCode=[" + ex.getErrorCode() + "] State=[" + ex.getSQLState() + "]" + CR ;
			throw new OgRuntimeException( errMsg,ex );
		}
	}

	/**
	 * 最初に、 行データである LineModel を作成します
	 * FirstProcess は、次々と処理をチェインしていく最初の行データを
	 * 作成して、後続の ChainProcess クラスに処理データを渡します。
	 *
	 * @og.rev 5.7.2.2 (2014/01/24) SQL実行エラーを少し詳細に出力します。
	 *
	 * @param	rowNo	処理中の行番号
	 *
	 * @return	処理変換後のLineModel
	 */
	public LineModel makeLineModel( final int rowNo ) {
		count++ ;
		try {
			for( int clm=0; clm<newData.size(); clm++ ) {
				final Object obj = resultSet.getObject(clm+1);
				if( obj == null ) {
		//			newData.setValue( clm, "" );
					newData.setValue( clm, null );
				}
				else {
					newData.setValue( clm, obj );
				}
			}
			newData.setRowNo( rowNo );
			if( display ) { println( newData.dataLine() ); }
		}
		catch (SQLException ex) {
			// 5.7.2.2 (2014/01/24) SQL実行エラーを少し詳細に出力します。
			final String errMsg = "データを処理できませんでした。[" + rowNo + "]件目 " + CR
					+ "errMsg=[" + ex.getMessage() + "]" + CR
					+ "errCode=[" + ex.getErrorCode() + "] State=[" + ex.getSQLState() + "]" + CR
					+ "dbid=[" + dbid + "]" + CR
					+ "data=[" + newData.dataLine() + "]" + CR ;
			throw new OgRuntimeException( errMsg,ex );
		}
		return newData;
	}

	/**
	 * 内部で使用する LineModel を作成します。
	 * このクラスは、プロセスチェインの基点となりますので、新規 LineModel を返します。
	 * Exception 以外では、必ず LineModel オブジェクトを返します。
	 * 第２引数は、カラム名の置き換え指示です。null の場合は、何もしません。
	 * 通常は、SELECT CLM1 AS CLM2 FROM *** とする箇所を、CLM1:CLM2 と指定する事で
	 * SELECT CLM1 FROM *** のまま、以降の処理を CLM2 で扱えます。
	 *
	 * @og.rev 5.7.2.2 (2014/01/24) SQL実行エラーを少し詳細に出力します。
	 * @og.rev 5.7.2.3 (2014/01/31) asClms 追加
	 *
	 * @param	rs	データベースカーソル(リザルトセット)
	 * @param	asClms	別名カラム指定 (元ｶﾗﾑ名:新ｶﾗﾑ名 のｽﾍﾟｰｽ区切り文字列)
	 *
	 * @return	データベースから取り出して変換した LineModel
	 * @throws RuntimeException カラム名を取得できなかった場合。
	 */
	private LineModel createLineModel( final ResultSet rs , final String asClms ) {
		final LineModel model = new LineModel();

		try {
			final ResultSetMetaData metaData	= rs.getMetaData();

			final int size =  metaData.getColumnCount();
			model.init( size );

			for( int clm=0; clm<size; clm++ ) {
				String name = metaData.getColumnLabel(clm+1).toUpperCase(Locale.JAPAN) ;
				// 5.7.2.3 (2014/01/31) asClms 追加
				if( asClms != null ) {
					// asClms の null判定も、toUpperCase 処理も行っているが、判りにくいので。
					name = StringUtil.caseReplace( name,asClms,false );
				}
				model.setName( clm,name );
			}
		}
		catch (SQLException ex) {
			// 5.7.2.2 (2014/01/24) SQL実行エラーを少し詳細に出力します。
			final String errMsg = "ResultSetMetaData から、カラム名を取得できませんでした。" + CR
								+ "errMsg=[" + ex.getMessage() + "]" + CR
								+ "errCode=[" + ex.getErrorCode() + "] State=[" + ex.getSQLState() + "]" + CR
								+ "dbid=[" + dbid + "]" + CR ;
			throw new OgRuntimeException( errMsg,ex );
		}
		return model;
	}

	/**
	 * プロセスの処理結果のレポート表現を返します。
	 * 処理プログラム名、入力件数、出力件数などの情報です。
	 * この文字列をそのまま、標準出力に出すことで、結果レポートと出来るような
	 * 形式で出してください。
	 *
	 * @return   処理結果のレポート
	 * @og.rtnNotNull
	 */
	public String report() {
		// 6.0.2.5 (2014/10/31) refactoring
		return "[" + getClass().getName() + "]" + CR
				+ TAB + "DBID        : " + dbid + CR
				+ TAB + "Input Count : " + count ;

	}

	/**
	 * このクラスの使用方法を返します。
	 *
	 * @return	このクラスの使用方法
	 * @og.rtnNotNull
	 */
	public String usage() {
		final StringBuilder buf = new StringBuilder( BUFFER_LARGE )
			.append( "Process_DBReaderは、データベースから読み取った内容を、LineModel に設定後、"	).append( CR )
			.append( "下流に渡す、FirstProcess インターフェースの実装クラスです。"					).append( CR )
			.append( CR )
			.append( "データベースから読み取った内容より、LineModelを作成し、下流(プロセス"			).append( CR )
			.append( "チェインは、チェインしているため、データは上流から下流へと渡されます。)"		).append( CR )
			.append( "に渡します。ここで指定できるのは、検索系SQL のみです。"						).append( CR )
			.append( CR )
			.append( "データベース接続先等は、ParamProcess のサブクラス(Process_DBParam)に"			).append( CR )
			.append( "設定された接続(Connection)を使用します。"										).append( CR )
			.append( CR )
			.append( "引数文字列中に空白を含む場合は、ダブルコーテーション(\"\") で括って下さい。"	).append( CR )
			.append( "引数文字列の 『=』の前後には、空白は挟めません。必ず、-key=value の様に"		).append( CR )
			.append( "繋げてください。"																).append( CR )
			.append( CR )
			.append( "SQL文には、{@DATE.YMDH}等のシステム変数が使用できます。"						).append( CR )
			.append( CR ).append( CR )
			.append( getArgument().usage() ).append( CR );

		return buf.toString();
	}

	/**
	 * このクラスは、main メソッドから実行できません。
	 *
	 * @param	args	コマンド引数配列
	 */
	public static void main( final String[] args ) {
		LogWriter.log( new Process_DBReader().usage() );
	}
}
