/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.process;

import org.opengion.fukurou.system.OgRuntimeException ;		// 6.4.2.0 (2016/01/29)
import org.opengion.fukurou.util.Argument;
import org.opengion.fukurou.util.SystemParameter;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.system.LogWriter;
import org.opengion.fukurou.util.HybsEntry ;
import org.opengion.fukurou.system.Closer;
import org.opengion.fukurou.model.Formatter;
import org.opengion.fukurou.db.ConnectionFactory;

import java.util.Map ;
import java.util.LinkedHashMap ;
import java.util.Set ;
import java.util.HashSet ;

import java.sql.Connection;
import java.sql.Statement;
import java.sql.PreparedStatement;
import java.sql.ParameterMetaData;
import java.sql.SQLException;

/**
 * Process_DBWriter は、上流から受け取ったデータをデータベースに書き込む
 * CainProcess インターフェースの実装クラスです。
 *
 * 上流(プロセスチェインのデータは上流から下流へと渡されます。)から受け取った
 * LineModel を元に、データベースへの書き込みを行います。
 *
 * データベース接続先等は、ParamProcess のサブクラス(Process_DBParam)に
 * 設定された接続(Connection)を使用します。
 *
 * 引数文字列中にスペースを含む場合は、ダブルコーテーション("") で括って下さい。
 * 引数文字列の 『=』の前後には、スペースは挟めません。必ず、-key=value の様に
 * 繋げてください。
 *
 * SQL文には、{&#064;DATE.YMDH}等のシステム変数が使用できます。
 *
 * @og.formSample
 *  Process_DBWriter -dbid=DBGE -table=GE41
 *
 *   [ -dbid=DB接続ID            ] ： -dbid=DBGE (例: Process_DBParam の -configFile で指定する DBConfig.xml ファイルで規定)
 *   [ -table=登録テーブルID     ] ： ＳＱＬ文を指定する場合は不要。INSERT する場合のテーブルID
 *   [ -sql=検索SQL文            ] ： -sql="UPDATE GE41 SET NAME_JA = [NAME_JA],LABEL_NAME = [LABEL_NAME]
 *                                         WHERE SYSTEM_ID = [SYSTEM_ID] AND CLM = [CLM]"
 *   [ -sqlFile=登録SQLﾌｧｲﾙ      ] ： -sqlFile=update.sql
 *                                 ：   -sql や -sqlFile が指定されない場合は、-table で指定のテーブルに全カラム insert です。
 *   [ -sql_XXXX=固定値          ] ： -sql_SYSTEM_ID=GE
 *                                     SQL文中の{&#064;XXXX}文字列を指定の固定値で置き換えます。
 *                                     WHERE SYSTEM_ID='{&#064;SYSTEM_ID}' ⇒ WHERE SYSTEM_ID='GE'
 *   [ -const_XXXX=固定値        ] ： -const_FGJ=1
 *                                     LineModel のキー(const_ に続く文字列)の値に、固定値を設定します。
 *                                     キーが異なれば、複数のカラム名を指定できます。
 *   [ -omitClms=AAA,BBB,…      ] ： -omitClms=UNIQ,FGJ,DYSET
 *                                     -table 属性でINSERT文を自動作成する場合、取り除くカラム名を
 *                                     CSV形式で複数指定できます。
 *   [ -initSql=開始時SQL文      ] ： -initSql="DELETE FROM GE41 WHERE FGJ = '9'"
 *   [ -initSqlFile=開始時SQLﾌｧｲﾙ] ： -initSqlFile=update.sql
 *   [ -endSql=終了時SQL文       ] ： -endSql="UPDATE GE41 SET FGJ = '1'"
 *   [ -endSqlFile=終了時SQLﾌｧｲﾙ ] ： -endSqlFile=update.sql
 *   [ -commitCnt=commit処理指定 ] ： 指定数毎にコミットを発行します。0 の場合は、終了までコミットしません。
 *   [ -display=[false/true]     ] ： 結果を標準出力に表示する(true)かしない(false)か(初期値:false[表示しない])
 *   [ -debug=[false/true]       ] ：デバッグ情報を標準出力に表示する(true)かしない(false)か(初期値:false[表示しない])
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Process_DBWriter extends AbstractProcess implements ChainProcess {
	private static final String CNST_KEY = "const_" ;
	private static final String SQL_KEY  = "sql_" ;

	private Connection	connection	;
	private PreparedStatement pstmt	;
	private ParameterMetaData pMeta ;		// 5.1.1.0 (2009/11/11) setObject に、Type を渡す。(PostgreSQL対応)
	private boolean useParamMetaData;		// 5.1.1.0 (2009/11/11) setObject に、Type を渡す。(PostgreSQL対応)

	private String		dbid		;
	private String		sql			;
	private String		endSql		;		// 5.7.2.2 (2014/01/24) 追加
	private String		table		;
	private int[]		clmNos		;		// ファイルのヘッダーのカラム番号
	private int			commitCnt	;		// コミットするまとめ件数
	private boolean		display		;		// false:表示しない
	private boolean		debug		;		// 5.7.3.0 (2014/02/07) デバッグ情報

	private String[]	cnstClm		;		// 固定値を設定するカラム名
	private int[]		cnstClmNos	;		// 固定値を設定するカラム番号
	private String[]	constVal	;		// カラム番号に対応した固定値

	private boolean		firstRow	= true;	// 最初の一行目
	private int			count		;
	private String[]	omitClms	;		// 4.0.0.0 (2007/09/21) table 指定時に取り除くカラム

	private static final Map<String,String> MUST_PROPARTY   ;		// ［プロパティ］必須チェック用 Map
	private static final Map<String,String> USABLE_PROPARTY ;		// ［プロパティ］整合性チェック Map

	static {
		MUST_PROPARTY = new LinkedHashMap<>();

		USABLE_PROPARTY = new LinkedHashMap<>();
		USABLE_PROPARTY.put( "dbid",	"Process_DBParam の -configFile で指定する DBConfig.xml ファイルで規定" );
		USABLE_PROPARTY.put( "table",		"INSERT する場合のテーブルID SQL文を指定する場合は不要。" );
		USABLE_PROPARTY.put( "sql",			"更新SQL文(sql or sqlFile 必須)" +
									CR + "例: \"UPDATE GE41 " +
									CR + "SET NAME_JA = [NAME_JA],LABEL_NAME = [LABEL_NAME] " +
									CR + "WHERE SYSTEM_ID = [SYSTEM_ID] AND CLM = [CLM]\"" );
		USABLE_PROPARTY.put( "sqlFile",		"登録SQLファイル(sql or sqlFile 必須)例: update.sql" );
		USABLE_PROPARTY.put( "sql_",		"SQL文中の{&#064;XXXX}文字列を指定の固定値で置き換えます。" +
									CR + "WHERE SYSTEM_ID='{&#064;SYSTEM_ID}' ⇒ WHERE SYSTEM_ID='GE'" );
		USABLE_PROPARTY.put( "const_",	"LineModel のキー(const_ に続く文字列)の値に、固定値を" +
									CR + "設定します。キーが異なれば、複数のカラム名を指定できます。" +
									CR + "例: -sql_SYSTEM_ID=GE" );
		// 4.0.0.0 (2007/09/21) 属性を追加
		USABLE_PROPARTY.put( "omitClms",	"-table 属性でINSERT文を自動作成する場合、取り除くカラム名を" +
									CR + "CSV形式で複数指定できます。" +
									CR + "例: -omitClms=UNIQ,FGJ,DYSET" );
		USABLE_PROPARTY.put( "initSql"	 ,	"開始時に一度だけ実行されるSQL文を指定します。" );			// 5.7.2.2 (2014/01/24) 追加
		USABLE_PROPARTY.put( "initSqlFile",	"開始時に一度だけ実行されるSQLファイルを指定します。" );	// 5.7.2.2 (2014/01/24) 追加
		USABLE_PROPARTY.put( "endSql"	 ,	"終了時に一度だけ実行されるSQL文を指定します。" );			// 5.7.2.2 (2014/01/24) 追加
		USABLE_PROPARTY.put( "endSqlFile" ,	"終了時に一度だけ実行されるSQLファイルを指定します。" );	// 5.7.2.2 (2014/01/24) 追加
		USABLE_PROPARTY.put( "commitCnt",	"指定数毎にコミットを発行します。" +
									CR + "0 の場合は、終了までコミットしません(初期値:0)" );
		USABLE_PROPARTY.put( "display",	"結果を標準出力に表示する(true)かしない(false)か" +
										CR + "(初期値:false:表示しない)" );
		USABLE_PROPARTY.put( "debug",	"デバッグ情報を標準出力に表示する(true)かしない(false)か" +
										CR + "(初期値:false:表示しない)" );		// 5.7.3.0 (2014/02/07) デバッグ情報
	}

	/**
	 * デフォルトコンストラクター。
	 * このクラスは、動的作成されます。デフォルトコンストラクターで、
	 * super クラスに対して、必要な初期化を行っておきます。
	 *
	 */
	public Process_DBWriter() {
		super( "org.opengion.fukurou.process.Process_DBWriter",MUST_PROPARTY,USABLE_PROPARTY );
	}

	/**
	 * プロセスの初期化を行います。初めに一度だけ、呼び出されます。
	 * 初期処理(ファイルオープン、ＤＢオープン等)に使用します。
	 *
	 * @og.rev 4.0.0.0 (2007/09/21) omitClms 属性を追加
	 * @og.rev 5.1.1.0 (2009/11/11) setObject に ParameterMetaData の getParameterType を渡す。(PostgreSQL対応)
	 * @og.rev 5.3.8.0 (2011/08/01) useParamMetaData を ConnectionFactory経由で取得。(PostgreSQL対応)
	 * @og.rev 5.7.2.2 (2014/01/24) initSql,initSqlFile,endSql,endSqlFile 追加
	 *
	 * @param   paramProcess データベースの接続先情報などを持っているオブジェクト
	 */
	public void init( final ParamProcess paramProcess ) {
		final Argument arg = getArgument();

		table		= arg.getProparty("table");
		sql			= arg.getFileProparty("sql","sqlFile",false);
		endSql		= arg.getFileProparty("endSql","endSqlFile",false);		// 5.7.2.2 (2014/01/24) 追加
		commitCnt	= arg.getProparty("commitCnt",commitCnt);
		display		= arg.getProparty("display",display);
		debug		= arg.getProparty("debug",debug);				// 5.7.3.0 (2014/02/07) デバッグ情報

		dbid		= arg.getProparty("dbid");
		connection	= paramProcess.getConnection( dbid );
		// 5.1.1.0 (2009/11/11) setObject に ParameterMetaData の getParameterType を渡す。(PostgreSQL対応)
		useParamMetaData = ConnectionFactory.useParameterMetaData( dbid );	// 5.3.8.0 (2011/08/01)

		// 取り除くカラム名リストを配列に変換します。
		final String tempClms	= arg.getProparty("omitClms",null);
		if( tempClms != null ) {
			omitClms = StringUtil.csv2Array( tempClms );
		}

		if( sql == null && table == null ) {
			final String errMsg = "sql を指定しない場合は、table を必ず指定してください。";
			throw new OgRuntimeException( errMsg );
		}

		// 3.8.0.1 (2005/06/17) {@DATE.XXXX} 変換処理の追加
		// {@DATE.YMDH} などの文字列を、yyyyMMddHHmmss 型の日付に置き換えます。
		// SQL文の {@XXXX} 文字列の固定値への置き換え
		final HybsEntry[] entry	=arg.getEntrys(SQL_KEY);		// 配列
		final SystemParameter sysParam = new SystemParameter( sql );
		sql = sysParam.replace( entry );

		// 5.7.2.2 (2014/01/24) initSql,endSql にも{@XXXX} 文字列の置き換えを行います。
		String initSql = arg.getFileProparty("initSql","initSqlFile",false);	// 5.7.2.2 (2014/01/24) 追加
		if( initSql != null ) {
			final SystemParameter sysParam2 = new SystemParameter( initSql );
			initSql = sysParam2.replace( entry );
			execSql( initSql );
		}
		if( endSql != null ) {
			final SystemParameter sysParam3 = new SystemParameter( endSql );
			endSql = sysParam3.replace( entry );
		}

		final HybsEntry[] cnstKey = arg.getEntrys( CNST_KEY );		// 配列
		final int csize	= cnstKey.length;
		cnstClm		= new String[csize];
		constVal	= new String[csize];
		for( int i=0; i<csize; i++ ) {
			cnstClm[i]  = cnstKey[i].getKey();
			constVal[i] = cnstKey[i].getValue();
		}
	}

	/**
	 * プロセスの終了を行います。最後に一度だけ、呼び出されます。
	 * 終了処理(ファイルクローズ、ＤＢクローズ等)に使用します。
	 *
	 * @og.rev 4.0.0.0 (2007/11/27) commit,rollback,remove 処理を追加
	 * @og.rev 5.1.1.0 (2009/11/11) pMeta のクリア
	 * @og.rev 5.7.2.2 (2014/01/24) endSql 処理の追加
	 *
	 * @param   isOK トータルで、OKだったかどうか[true:成功/false:失敗]
	 */
	public void end( final boolean isOK ) {
		final boolean flag = Closer.stmtClose( pstmt );
		pstmt = null;
		pMeta = null;		// 5.1.1.0 (2009/11/11)

		// 5.7.2.2 (2014/01/24) endSql の実行
		Throwable th2 = null;
		if( isOK && endSql != null ) {
			try { execSql( endSql ); } catch (Throwable th) { th2 = th ; }
		}

		// 5.7.2.2 (2014/01/24) すべて異常がない場合のみ、処理する様に変更。
		if( isOK && flag && th2 == null ) {
			Closer.commit( connection );
		}
		else {
			Closer.rollback( connection );
		}
		ConnectionFactory.remove( connection,dbid );

		if( !flag ) {
			final String errMsg = "ステートメントをクローズ出来ません。";
			throw new OgRuntimeException( errMsg );
		}

		// 5.7.2.2 (2014/01/24) endSql の実行失敗時の処理
		if( th2 != null ) {
			final String errMsg = "endSql の実行に失敗しました。sql=[" + endSql + "]" + CR
								+ th2.getMessage() + CR ;
			throw new OgRuntimeException( errMsg,th2 );
		}
	}

	/**
	 * 引数の LineModel を処理するメソッドです。
	 * 変換処理後の LineModel を返します。
	 * 後続処理を行わない場合(データのフィルタリングを行う場合)は、
	 * null データを返します。つまり、null データは、後続処理を行わない
	 * フラグの代わりにも使用しています。
	 * なお、変換処理後の LineModel と、オリジナルの LineModel が、
	 * 同一か、コピー(クローン)かは、各処理メソッド内で決めています。
	 * ドキュメントに明記されていない場合は、副作用が問題になる場合は、
	 * 各処理ごとに自分でコピー(クローン)して下さい。
	 *
	 * @og.rev 5.1.1.0 (2009/11/11) setObject に ParameterMetaData の getParameterType を渡す。(PostgreSQL対応)
	 * @og.rev 5.3.8.0 (2011/08/01) useParamMetaData  setNull 対応(PostgreSQL対応)
	 * @og.rev 5.7.2.2 (2014/01/24) SQL実行エラーを少し詳細に出力します。
	 *
	 * @param	data	オリジナルのLineModel
	 *
	 * @return	処理変換後のLineModel
	 */
	public LineModel action( final LineModel data ) {
		count++ ;
		try {
			if( firstRow ) {
				pstmt = makePrepareStatement( table,data );
				// 5.1.1.0 (2009/11/11) setObject に ParameterMetaData の getParameterType を渡す。(PostgreSQL対応)
				if( useParamMetaData ) {
					pMeta = pstmt.getParameterMetaData();
				}

				final int size   = cnstClm.length;
				cnstClmNos = new int[size];
				for( int i=0; i<size; i++ ) {
					cnstClmNos[i] = data.getColumnNo( cnstClm[i] );
				}

				firstRow = false;
				if( display ) { println( data.nameLine() ); }		// 5.7.3.0 (2014/02/07) デバッグ情報
			}

			// 固定値置き換え処理
			for( int j=0; j<cnstClmNos.length; j++ ) {
				data.setValue( cnstClmNos[j],constVal[j] );
			}

			// 5.1.1.0 (2009/11/11) setObject に ParameterMetaData の getParameterType を渡す。(PostgreSQL対応)
			if( useParamMetaData ) {
				for( int i=0; i<clmNos.length; i++ ) {
					final int type = pMeta.getParameterType( i+1 );
					// 5.3.8.0 (2011/08/01) setNull 対応
					final Object val = data.getValue(clmNos[i]);
					if( val == null || ( val instanceof String && ((String)val).isEmpty() ) ) {
						pstmt.setNull( i+1, type );
					}
					else {
						pstmt.setObject( i+1, val, type );
					}
				}
			}
			else {
				for( int i=0; i<clmNos.length; i++ ) {
					pstmt.setObject( i+1,data.getValue(clmNos[i]) );
				}
			}

			pstmt.execute();
			if( commitCnt > 0 && ( count%commitCnt == 0 ) ) {
				Closer.commit( connection );
			}
		}
		catch (SQLException ex) {
			// 5.7.2.2 (2014/01/24) SQL実行エラーを少し詳細に出力します。
			final String errMsg = "SQL を実行できませんでした。" + CR
								+ "errMsg=[" + ex.getMessage() + "]" + CR
								+ "errCode=[" + ex.getErrorCode() + "] State=[" + ex.getSQLState() + "]" + CR
								+ "dbid=[" + dbid + "]" + CR
								+ "sql =[" + sql + "]" + CR
								+ "data=[" + data.dataLine() + "]" + CR ;
			throw new OgRuntimeException( errMsg,ex );
		}
		if( display ) { println( data.dataLine() ); }	// 5.1.2.0 (2010/01/01) display の条件変更
		return data;
	}

	/**
	 * 内部で使用する PreparedStatement を作成します。
	 * 引数指定の SQL または、LineModel から作成した SQL より構築します。
	 *
	 * @og.rev 4.0.0.0 (2007/09/21) omitClms 属性を追加
	 * @og.rev 5.7.2.2 (2014/01/24) SQL実行エラーを少し詳細に出力します。
	 * @og.rev 6.2.3.0 (2015/05/01) CSV形式の作成を、String#join( CharSequence , CharSequence... )を使用。
	 *
	 * @param	table 	処理対象のテーブルID
	 * @param	data	処理対象のLineModel
	 *
	 * @return  PreparedStatementオブジェクト
	 */
	private PreparedStatement makePrepareStatement( final String table,final LineModel data ) {
		if( sql == null ) {
			String[] names = data.getNames();

			// カラムを取り除く場合
			if( omitClms != null ) {
				final Set<String> set = new HashSet<>();
				for( int i=0; i<names.length; i++ ) {
					set.add( names[i] );
				}
				for( int i=0; i<omitClms.length; i++ ) {
					set.remove( omitClms[i] );
				}
				names = set.toArray( new String[set.size()] );
			}
			final int size = names.length;

		// 6.2.3.0 (2015/05/01) CSV形式の作成を、String#join( CharSequence , CharSequence... )を使用。
			final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
				.append( "INSERT INTO " ).append( table ).append( " (" )
				.append( String.join( "," , names ) )		// 6.2.3.0 (2015/05/01)
				.append( " ) VALUES ( ?" );
			for( int i=1; i<size; i++ ) {
				buf.append( ",?" );
			}
			buf.append( " )" );
			sql = buf.toString();

			// カラム番号を設定します。
			clmNos = new int[size];
			for( int i=0; i<size; i++ ) {
				clmNos[i] = data.getColumnNo( names[i] );		// 4.0.0.0 (2007/09/21)
			}
		}
		else {
			final Formatter format = new Formatter( data );
			format.setFormat( sql );
			sql = format.getQueryFormatString();
			clmNos = format.getClmNos();
		}

		final PreparedStatement ps ;
		try {
			ps = connection.prepareStatement( sql );
		}
		catch (SQLException ex) {
			// 5.7.2.2 (2014/01/24) SQL実行エラーを少し詳細に出力します。
			final String errMsg = "PreparedStatement を取得できませんでした。" + CR
								+ "errMsg=[" + ex.getMessage() + "]" + CR
								+ "errCode=[" + ex.getErrorCode() + "] State=[" + ex.getSQLState() + "]" + CR
								+ "dbid =[" + dbid + "]" + CR
								+ "sql  =[" + sql + "]" + CR
								+ "table=[" + table + "]" + CR
								+ "data =[" + data.dataLine() + "]" + CR ;
			throw new OgRuntimeException( errMsg,ex );
		}

		return ps;
	}

	/**
	 * SQL処理を実行します。
	 * 主に、initSql,endSqlの実行用です。
	 * ここでは、エラーが発生しても、connection は閉じません。
	 * 最終的に、endメソッドで処理されるためです。
	 *
	 * @og.rev 5.7.2.2 (2014/01/24) 新規追加
	 *
	 * @param   sql 実行するSQL文
	 */
	private void execSql( final String sql ) {
		Statement stmt = null;
		try {
			stmt = connection.createStatement();
			stmt.execute( sql );
		}
		catch (SQLException ex) {
			// 5.7.2.2 (2014/01/24) SQL実行エラーを少し詳細に出力します。
			final String errMsg = "SQL を実行できませんでした。" + CR
								+ "errMsg=[" + ex.getMessage() + "]" + CR
								+ "errCode=[" + ex.getErrorCode() + "] State=[" + ex.getSQLState() + "]" + CR
								+ "dbid=[" + dbid + "]" + CR
								+ "sql =[" + sql + "]" + CR ;
			throw new OgRuntimeException( errMsg,ex );
		}
		finally {
			// connection は、endメソッドで処理されます。
			Closer.stmtClose( stmt );
		}
	}

	/**
	 * プロセスの処理結果のレポート表現を返します。
	 * 処理プログラム名、入力件数、出力件数などの情報です。
	 * この文字列をそのまま、標準出力に出すことで、結果レポートと出来るような
	 * 形式で出してください。
	 *
	 * @return   処理結果のレポート
	 */
	public String report() {
		final String report = "[" + getClass().getName() + "]" + CR
							+ TAB + "DBID         : " + dbid + CR
							+ TAB + "Output Count : " + count ;

		return report ;
	}

	/**
	 * このクラスの使用方法を返します。
	 *
	 * @return	このクラスの使用方法
	 * @og.rtnNotNull
	 */
	public String usage() {
		final StringBuilder buf = new StringBuilder( BUFFER_LARGE )
			.append( "Process_DBWriter は、上流から受け取ったデータをデータベースに書き込む"		).append( CR )
			.append( "CainProcess インターフェースの実装クラスです。"								).append( CR )
			.append( CR )
			.append( "上流(プロセスチェインのデータは上流から下流へと渡されます。)から"				).append( CR )
			.append( "受け取った LineModel を元に、データベースへの書き込みを行います。"			).append( CR )
			.append( CR )
			.append( "データベース接続先等は、ParamProcess のサブクラス(Process_DBParam)に"			).append( CR )
			.append( "設定された接続(Connection)を使用します。"										).append( CR )
			.append( CR )
			.append( "引数文字列中に空白を含む場合は、ダブルコーテーション(\"\") で括って下さい。"	).append( CR )
			.append( "引数文字列の 『=』の前後には、空白は挟めません。必ず、-key=value の様に"		).append( CR )
			.append( "繋げてください。"																).append( CR )
			.append( CR )
			.append( "SQL文には、{@DATE.YMDH}等のシステム変数が使用できます。"						).append( CR )
			.append( CR ).append( CR )
			.append( getArgument().usage() ).append( CR );

		return buf.toString();
	}

	/**
	 * このクラスは、main メソッドから実行できません。
	 *
	 * @param	args	コマンド引数配列
	 */
	public static void main( final String[] args ) {
		LogWriter.log( new Process_DBWriter().usage() );
	}
}
