/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import org.opengion.fukurou.system.OgRuntimeException ;		// 6.4.2.0 (2016/01/29)
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.resource.ResourceManager;
import org.opengion.fukurou.db.ConnectionFactory;
import org.opengion.fukurou.db.ResultSetValue;				// 6.0.4.0 (2014/11/28)
import org.opengion.fukurou.util.ApplicationInfo;

import java.sql.DatabaseMetaData ;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;

/**
 * 【検索】DatabaseMetaData の情報を検索するタグです。
 *
 * データベースに関する包括的な情報を提供する、DatabaseMetaData の内容を
 * 表示する、タグです。テスト的に使用します。
 *
 * @og.formSample
 * ●形式：
 *       ・&lt;og:databaseMetaData /&gt;
 * ●body：なし
 *
 * ●使用例
 *       &lt;og:databaseMetaData /&gt;
 *
 * @og.group テーブル管理
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DBMetaData {
	private String			dbid		;
	private ResourceManager resource	;
	private ApplicationInfo appInfo		;	// 3.8.7.0 (2006/12/15)

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public DBMetaData() {
		super();		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。
	}

	/**
	 * DatabaseMetaData を作成する時のDB接続IDを指定します。
	 *
	 * @param	id データベース接続ID
	 */
	public void setDbid( final String id ) {
		dbid = id;
	}

	/**
	 * リソースマネージャーをセットします。
	 * リソースマネージャーが設定されていない、または、所定のキーの DBColumn が
	 * リソースに存在しない場合は、デフォルトの DBColumn オブジェクトを作成します。
	 *
	 * @param	resource リソースマネージャー
	 */
	public void setResourceManager( final ResourceManager resource ) {
		this.resource = resource;
	}

	/**
	 * アクセスログ取得の為,ApplicationInfoオブジェクトを設定します。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) 新規追加
	 *
	 * @param   appInfo アプリ情報オブジェクト
	 */
	public void setApplicationInfo( final ApplicationInfo appInfo ) {
		this.appInfo = appInfo;
	}

	/**
	 * ResultSet より、DBTableModel を作成して返します。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) SQLException を throws するように変更。
	 * @og.rev 6.0.4.0 (2014/11/28) ResultSetValue を使用するように変更。
	 * @og.rev 6.3.9.0 (2015/11/06) コンストラクタで初期化されていないフィールドを null チェックなしで利用している(findbugs)
	 *
	 * @param	resultSet ResultSetオブジェクト
	 *
	 * @return	作成された DBTableModelオブジェクト
	 * @throws	java.sql.SQLException データベース・アクセス・エラーが発生した場合
	 */
	private DBTableModel makeDBTableModel( final ResultSet resultSet ) throws SQLException {

		final ResultSetValue rsv = new ResultSetValue( resultSet );

		final int clmSize =  rsv.getColumnCount();
//		final String[] names = rsv.getNames();

		final DBTableModel table = DBTableModelUtil.newDBTable() ;
		table.init( clmSize );

		// 6.3.9.0 (2015/11/06) コンストラクタで初期化されていないフィールドを null チェックなしで利用している(findbugs)
		if( resource == null ) {
			final String errMsg = "#setResourceManager(ResourceManager)を先に実行しておいてください。" ;
			throw new OgRuntimeException( errMsg );
		}

		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid declaring a variable if it is unreferenced before a possible exit point.
		final String[] names = rsv.getNames();
		for( int clmNo=0; clmNo<clmSize; clmNo++ ) {
			final DBColumn clm = resource.makeDBColumn( names[clmNo] );
			table.setDBColumn( clmNo,clm );
		}

		// データ部の設定
		while( rsv.next() ) {
			table.addColumnValues( rsv.getValues() ) ;
		}
		return table ;
	}

	/**
	 * このデータベースで使用可能なスキーマ名を取得します。
	 * 結果はスキーマ名で順序付けられます。
	 * スキーマ列は次のようになります。
	 *
	 * ・<b>TABLE_SCHEM</b> String ⇒ スキーマ名
	 * ・<b>TABLE_CATALOG</b> String ⇒ カタログ名 (null の可能性がある)
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 *
	 * @return	スキーマ名をDBTableModelオブジェクトにラップ
	 * @see java.sql.DatabaseMetaData#getSchemas()
	 */
	public DBTableModel getSchemas() {
		final DBTableModel table ;
		Connection conn = null ;
		try {
			conn = ConnectionFactory.connection( dbid,appInfo );
			final DatabaseMetaData metaData = conn.getMetaData();
			// ====== table 求めの個所のみ、異なります。
			table = makeDBTableModel( metaData.getSchemas() );
			// ====== ここまで
		}
		catch ( SQLException ex) {
			ConnectionFactory.remove( conn,dbid );
			conn = null;
			throw new HybsSystemException( ex );
		}
		finally {
			ConnectionFactory.close( conn,dbid );
		}
		return table ;
	}

	/**
	 * 指定されたカタログで使用可能なテーブルに関する記述を取得します。
	 * カタログ、スキーマ、テーブル名および型の条件に一致するテーブルの記述だけが返されます。
	 * それらは、TABLE_TYPE、TABLE_SCHEM、TABLE_NAME によって順序付けられます。
	 *
	 * 各テーブルの記述には次の列があります。
	 *
	 * ・<b>TABLE_CAT</b> String ⇒ テーブルカタログ (null の可能性がある)
	 * ・<b>TABLE_SCHEM</b> String ⇒ テーブルスキーマ (null の可能性がある)
	 * ・<b>TABLE_NAME</b> String ⇒ テーブル名
	 * ・<b>TABLE_TYPE</b> String ⇒ テーブルの型。典型的な型は、"TABLE"、"VIEW"、"SYSTEM TABLE"、"GLOBAL TEMPORARY"、"LOCAL TEMPORARY"、"ALIAS"、"SYNONYM" である
	 * ・<b>REMARKS</b> String ⇒ テーブルに関する説明
	 * ・<b>TYPE_CAT</b> String ⇒ の型のカタログ (null の可能性がある)
	 * ・<b>TYPE_SCHEM</b> String ⇒ の型のスキーマ (null の可能性がある)
	 * ・<b>TYPE_NAME</b> String ⇒ の型名 (null の可能性がある)
	 * ・<b>SELF_REFERENCING_COL_NAME</b> String ⇒ 型付きテーブルの指定された「識別子」列の名前 (null の可能性がある)
	 * ・<b>REF_GENERATION</b> String ⇒ SELF_REFERENCING_COL_NAME の値の作成方法を指定する。値は、"SYSTEM"、"USER"、"DERIVED" (null の可能性がある)
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 *
	 * @param	catalog	カタログ名。(カタログ名と一致、"" はカタログなし、null は、カタログ名無条件)
	 * @param	schema	スキーマ名パターン。(スキーマ名と一致、"" はスキーマなし、null は、スキーマ名無条件)
	 * @param	tableName	テーブル名パターン。
	 *
	 * @return	テーブルに関する記述をDBTableModelオブジェクトにラップ
	 * @see java.sql.DatabaseMetaData#getSchemas()
	 */
	public DBTableModel getTables( final String catalog,
									final String schema,
									final String tableName ) {
		final DBTableModel table ;
		Connection conn = null ;
		try {
			conn = ConnectionFactory.connection( dbid,appInfo );
			final DatabaseMetaData metaData = conn.getMetaData();
			// ====== table 求めの個所のみ、異なります。
			// types String[] 組み込むテーブルの型のリスト。null はすべての型を返す
			table = makeDBTableModel( metaData.getTables(catalog, schema, tableName, null) );
			// ====== ここまで
		}
		catch ( SQLException ex) {
			ConnectionFactory.remove( conn,dbid );
			conn = null;
			throw new HybsSystemException( ex );
		}
		finally {
			ConnectionFactory.close( conn,dbid );
		}
		return table ;
	}

	/**
	 * 指定されたカタログで使用可能なテーブル列の記述を取得します。
	 * カタログ、スキーマ、テーブル名、および列名の条件に一致する列の
	 * 記述だけが返されます。
	 * それらは、TABLE_SCHEM、TABLE_NAME、ORDINAL_POSITION によって順序付けられます。
	 * 各列の説明を次にします
	 *
	 * ・<b>TABLE_CAT</b> String ⇒ テーブルカタログ (null の可能性がある)
	 * ・<b>TABLE_SCHEM</b> String ⇒ テーブルスキーマ (null の可能性がある)
	 * ・<b>TABLE_NAME</b> String ⇒ テーブル名
	 * ・<b>COLUMN_NAME</b> String ⇒ 列名
	 * ・<b>DATA_TYPE</b> short ⇒ java.sql.Types からの SQL の型
	 * ・<b>TYPE_NAME</b> String ⇒ データソース依存の型名。UDT の場合、型名は完全指定
	 * ・<b>COLUMN_SIZE</b> int ⇒ 列サイズ。char や date の型については最大文字数、numeric や decimal の型については精度
	 * ・<b>BUFFER_LENGTH</b> - 未使用
	 * ・<b>DECIMAL_DIGITS</b> int ⇒ 小数点以下の桁数
	 * ・<b>NUM_PREC_RADIX</b> int ⇒ 基数 (通常は、10 または 2 のどちらか)
	 * ・<b>NULLABLE</b> int ⇒ NULL は許されるか
	 * ・<b>columnNoNulls</b> - NULL 値を許さない可能性がある
	 * ・<b>columnNullable</b> - 必ず NULL 値を許す
	 * ・<b>columnNullableUnknown</b> - NULL 値を許すかどうかは不明
	 * ・<b>REMARKS</b> String ⇒ コメント記述列 (null の可能性がある)
	 * ・<b>COLUMN_DEF</b> String ⇒ デフォルト値 (null の可能性がある)
	 * ・<b>SQL_DATA_TYPE</b> int ⇒ 未使用
	 * ・<b>SQL_DATETIME_SUB</b> int ⇒ 未使用
	 * ・<b>CHAR_OCTET_LENGTH</b> int ⇒ char の型については列の最大バイト数
	 * ・<b>ORDINAL_POSITION</b> int ⇒ テーブル中の列のインデックス (1 から始まる)
	 * ・<b>IS_NULLABLE</b> String ⇒ "NO" は、列は決して NULL 値を許さないことを意味する。"YES" は NULL 値を許す可能性があることを意味する。空の文字列は不明であることを意味する
	 * ・<b>SCOPE_CATLOG</b> String ⇒ 参照属性のスコープであるテーブルのカタログ (DATA_TYPE が REF でない場合は null)
	 * ・<b>SCOPE_SCHEMA</b> String ⇒ 参照属性のスコープであるテーブルのスキーマ (DATA_TYPE が REF でない場合は null)
	 * ・<b>SCOPE_TABLE</b> String ⇒ 参照属性のスコープであるテーブル名 (DATA_TYPE が REF でない場合は null)
	 * ・<b>SOURCE_DATA_TYPE</b> short ⇒ 個別の型またはユーザ生成 Ref 型、java.sql.Types の SQL 型のソースの型 (DATA_TYPE が DISTINCT またはユーザ生成 REF でない場合は null)
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 *
	 * @param	catalog	カタログ名。(カタログ名と一致、"" はカタログなし、null は、カタログ名無条件)
	 * @param	schema	スキーマ名パターン。(スキーマ名と一致、"" はスキーマなし、null は、スキーマ名無条件)
	 * @param	tableName	テーブル名パターン。
	 * @param	columnName	列名パターン
	 *
	 * @return	テーブル列の記述をDBTableModelオブジェクトにラップ
	 * @see java.sql.DatabaseMetaData#getSchemas()
	 */
	public DBTableModel getColumns( final String catalog,
									final String schema,
									final String tableName,
									final String columnName ) {
		final DBTableModel table ;
		Connection	 conn	= null ;
		try {
			conn = ConnectionFactory.connection( dbid,appInfo );
			final DatabaseMetaData metaData = conn.getMetaData();
			// ====== table 求めの個所のみ、異なります。
			table = makeDBTableModel( metaData.getColumns(catalog, schema, tableName, columnName) );
			// ====== ここまで
		}
		catch ( SQLException ex) {
			ConnectionFactory.remove( conn,dbid );
			conn = null;
			throw new HybsSystemException( ex );
		}
		finally {
			ConnectionFactory.close( conn,dbid );
		}
		return table ;
	}

	/**
	 * 指定されたテーブルのインデックスと統計情報に関する記述を取得します。
	 * それらは、NON_UNIQUE、TYPE、INDEX_NAME、ORDINAL_POSITION によって順序付けされます。
	 * 各インデックス列の記述には次の列があります
	 *
	 * ・<b>TABLE_CAT</b> String ⇒ テーブルカタログ (null の可能性がある)
	 * ・<b>TABLE_SCHEM</b> String ⇒ テーブルスキーマ (null の可能性がある)
	 * ・<b>TABLE_NAME</b> String ⇒ テーブル名
	 * ・<b>NON_UNIQUE</b> boolean ⇒ インデックス値は一意でない値にできるか。TYPE が tableIndexStatistic の場合は false
	 * ・<b>INDEX_QUALIFIER</b> String ⇒ インデックスカタログ (null の可能性がある)。TYPE が tableIndexStatistic の場合は null
	 * ・<b>INDEX_NAME</b> String ⇒ インデックス名。TYPE が tableIndexStatistic の場合は null
	 * ・<b>TYPE</b> short ⇒ インデックスの型
	 *
	 * ・tableIndexStatistic - テーブルのインデックスの記述に連動して返されるテーブルの統計情報を識別する
	 * ・tableIndexClustered - クラスタ化されたインデックス
	 * ・tableIndexHashed - ハッシュ化されたインデックス
	 * ・tableIndexOther - インデックスのその他のスタイル
	 *
	 * ・<b>ORDINAL_POSITION</b> short ⇒ インデックス中の列シーケンス。TYPE が tableIndexStatistic の場合は 0
	 * ・<b>COLUMN_NAME</b> String ⇒ 列名。TYPE が tableIndexStatistic の場合は null
	 * ・<b>ASC_OR_DESC</b> String ⇒ 列ソートシーケンス、"A" ⇒ 昇順、"D" ⇒ 降順、
				ソートシーケンスがサポートされていない場合は、null の可能性がある。TYPE が tableIndexStatistic の場合は null
	 * ・<b>CARDINALITY</b> int ⇒ TYPE が tableIndexStatistic の場合、テーブル中の列数。そうでない場合は、インデックス中の一意の値の数
	 * ・<b>PAGES</b> int ⇒ TYPE が tableIndexStatistic の場合、テーブルで使用されるページ数。そうでない場合は、現在のインデックスで使用されるページ数
	 * ・<b>FILTER_CONDITION</b> String ⇒ もしあれば、フィルタ条件 (null の可能性がある)
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 *
	 * @param	catalog	カタログ名。(カタログ名と一致、"" はカタログなし、null は、カタログ名無条件)
	 * @param	schema	スキーマ名パターン。(スキーマ名と一致、"" はスキーマなし、null は、スキーマ名無条件)
	 * @param	tableName	テーブル名。このデータベースに格納されたテーブル名と一致しなければならない
	 * @param	unique	true の場合は、一意の値のインデックスだけを返す。false の場合は、一意であるかどうかにかかわらずインデックスを返す
	 * @param	approximate	true の場合は、結果は概数またはデータ値から外れることもある。false の場合は、正確であることが要求される
	 *
	 * @return	インデックスと統計情報に関する記述をDBTableModelオブジェクトにラップ
	 * @see java.sql.DatabaseMetaData#getSchemas()
	 */
	public DBTableModel getIndexInfo( final String catalog,
										final String schema,
										final String tableName,
										final boolean unique,
										final boolean approximate ) {
		final DBTableModel table ;
		Connection conn = null ;
		try {
			conn = ConnectionFactory.connection( dbid,appInfo );
			final DatabaseMetaData metaData = conn.getMetaData();
			// ====== table 求めの個所のみ、異なります。
			table = makeDBTableModel( metaData.getIndexInfo(catalog, schema, tableName, unique, approximate) );
			// ====== ここまで
		}
		catch ( SQLException ex) {
			ConnectionFactory.remove( conn,dbid );
			conn = null;
			throw new HybsSystemException( ex );
		}
		finally {
			ConnectionFactory.close( conn,dbid );
		}
		return table ;
	}

	/**
	 * 指定されたカタログで使用可能なストアドプロシージャに関する記述を取得します。
	 * スキーマとプロシージャ名の条件に一致するプロシージャの記述だけが返されます。
	 * それらは、PROCEDURE_SCHEM と PROCEDURE_NAME によって順序付けられます。
	 *
	 * 各プロシージャの記述には次の列があります。
	 *
	 * ・PROCEDURE_CAT String ⇒ プロシージャカタログ (null の可能性がある)
	 * ・PROCEDURE_SCHEM String ⇒ プロシージャスキーマ (null の可能性がある)
	 * ・PROCEDURE_NAME String ⇒ プロシージャ名
	 * ・将来使用するための予約
	 * ・将来使用するための予約
	 * ・将来使用するための予約
	 * ・REMARKS String ⇒ プロシージャの説明文
	 * ・PROCEDURE_TYPE short ⇒ プロシージャの種類
	 *
	 * ・procedureResultUnknown - 結果を返す可能性がある
	 * ・procedureNoResult - 結果を返さない
	 * ・procedureReturnsResult - 結果を返す
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 *
	 * @param	catalog	カタログ名。(カタログ名と一致、"" はカタログなし、null は、カタログ名無条件)
	 * @param	schema	スキーマ名パターン。(スキーマ名と一致、"" はスキーマなし、null は、スキーマ名無条件)
	 * @param	procName	プロシージャ名パターン。データベースに格納されたプロシージャ名と一致しなければならない
	 *
	 * @return	ストアドプロシージャに関する記述をDBTableModelオブジェクトにラップ
	 * @see java.sql.DatabaseMetaData#getSchemas()
	 */
	public DBTableModel getProcedures( final String catalog,
										final String schema,
										final String procName ) {
		final DBTableModel table ;
		Connection conn = null ;
		try {
			conn = ConnectionFactory.connection( dbid,appInfo );
			final DatabaseMetaData metaData = conn.getMetaData();
			// ====== table 求めの個所のみ、異なります。
			table = makeDBTableModel( metaData.getProcedures(catalog, schema, procName) );
			// ====== ここまで
		}
		catch ( SQLException ex) {
			ConnectionFactory.remove( conn,dbid );
			conn = null;
			throw new HybsSystemException( ex );
		}
		finally {
			ConnectionFactory.close( conn,dbid );
		}
		return table ;
	}
}
