/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import org.opengion.hayabusa.resource.GUIInfo;
import org.opengion.fukurou.util.StringUtil ;					// 6.2.0.0 (2015/02/27)

import static org.opengion.fukurou.util.StringUtil.nval;

/**
 * テキストフィールドの入力を補助するポップアップを作成します。
 *
 * 通常のテキストフィールドに組み込むケースと、一覧表に組み込むケースに対応しています。
 * 基本的には、columnMarker タグと同じ使い方ですが、フィールドが書き込み許可時に表示させる必要が
 * あるため、実際には、viewMarker の isRendere="false" で使用するケースが多いと思います。
 * なお、ポップアップ専用なので href="index.jsp" target="CONTENTS" がデフォルトで設定されています。
 *
 * @og.formSample
 * ●形式：&lt;og:popup gamenId="…" href="…" keys="…,…" vals="…,…" rtnKeys="…,…" /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:popup
 *       name               【TAG】この要素をリンクの対象とするための名前を指定します
 *       gamenId            【TAG】画面ID属性を登録します
 *       href               【TAG】ポップアップ先のURLを指定します(初期値:index.jsp)
 *       keys               【TAG】ポップアップ先に渡す入力キー配列をCSV形式で複数指定します
 *       vals               【TAG】ポップアップ先に渡す値配列をCSV形式で複数指定します
 *       rtnKeys            【TAG】ポップアップ先より受け取るキー配列をCSV形式で複数指定します
 *       column             【TAG】リンクをつけたいカラムを指定します
 *       onMark             【TAG】リンクを張る(true or 1)/張らない(false or 0)の設定を指定します(初期値:true)
 *       markList           【TAG】処理するマークを、"|"区切りの文字列で指定します
 *       height             【TAG】ダイアログの縦幅をピクセル数で指定します
 *       width              【TAG】ダイアログの横幅をピクセル数で指定します
 *       src                【TAG】表示させる画像のURLを指定します
 *       charset            【TAG】ポップアップ先の文字コードセットを指定します
 *       target             【TAG】ポップアップ先の表示ターゲットを指定します(初期値:CONTENTS)
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:判定しない)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:判定しない)
 *       caseIf             【TAG】指定の値が、true/TRUE文字列の場合は、このタグは使用されます(初期値:判定しない)
 *       id                 【HTML】要素に対して固有の名前(id)をつける場合に設定します
 *       lang               【HTML】要素の内容と他の属性値の言語(lang,xml:lang)を指定します
 *       dir                【HTML】文字表記の方向(dir)を指定します
 *       title              【HTML】要素に対する補足的情報(title)を設定します
 *       style              【HTML】この要素に対して適用させるスタイルシート(style)を設定します
 *       tabindex           【HTML】タブの移動順(tabindex)を指定します(0 ～ 32767)
 *       accesskey          【HTML】アクセスキー(alt+キーで直接指定)を割り当てます
 *       clazz              【HTML】要素に対して class 属性を設定します
 *       onClick            【HTML】JavaScriptのイベント onClick を設定します(例:onClick="renew('query.jsp','QUERY');")
 *       lbl                【TAG】ラベルリソースのラベルIDを指定します
 *       language           【TAG】タグ内部で使用する言語コード[ja/en/zh/…]を指定します
 *       command            【TAG】(通常使いません)処理の実行を指定する command を設定できます(初期値:NEW)
 *       append             【TAG】返ってきた値をキャレット位置に追記するかどうか(初期値：false)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *    ・普通のポップアップ
 *        &lt;og:popup gamenId="GE0002" href="index.jsp" target="CONTENTS"
 *                    keys="AAA,BBB,CCC" vals="{&#064;AAA},{&#064;B},ABC"
 *                    rtnKeys="DDD,EEE" /&gt;
 *
 *    ・QUERYの入力フィールドに組み込む
 *      ①keys、vals、rtnKeys がセットされる場合
 *      　　　　セットされる keys に対して vals の値をポップアップ画面に渡し、ポップアップ画面の戻り値を rtnKeys にセットされます。
 *        &lt;og:column name="AAA" value="{&#064;AAA}" &gt;
 *            &lt;og:popup gamenId="GE0002"
 *                        keys="AAA,BBB,CCC" vals="{&#064;AAA},{&#064;B},ABC"
 *                        rtnKeys="DDD,EEE" /&gt;
 *        &lt;/og:column&gt;
 *      ②keys、vals、rtnKeys がセットされない場合
 *      　　　　 ポップアップ画面にリクエスト変数 XNAME に外側のカラム名、XVALUE にカラムの入力値、rtnKeys にカラム名を自動的にセットされます。
 *        &lt;og:column name="AAA" value="{&#064;AAA}" &gt;
 *            &lt;og:popup gamenId="GE0002" /&gt;
 *        &lt;/og:column&gt;
 *
 *    ・RESULTの入力フィールドに組み込む(viewMarker 経由)
 *      gamenId を指定しなければこの画面へのURLを作成します。
 *      このURLは、その画面の index.jsp に直接ジャンプするためのURLです。
 *        &lt;og:viewMarker command="{&#064;command}" isRenderer="false"&gt;
 *            &lt;og:popup gamenId="GE0002" column="ABC" onMark="[ABC]" markList="1|2|3"
 *                        keys="AAA,BBB,CCC" vals="[AAA],[BBB].value,{&#064;CCC}"
 *                        rtnKeys="DDD,EEE" /&gt;
 *        &lt;/og:viewMarker&gt;
 *
 *    ・ポップアップ側のJSPファイル(view 部分)
 *
 * @og.group 画面部品
 * @og.rev 3.8.6.1 (2006/10/20) 新規作成
 *
 * @version  0.9.0	2000/10/17
 * @author	 Kazuhiko Hasegawa
 * @since	 JDK1.1,
 */
public class PopupTag extends HTMLTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.2.0 (2016/01/29)" ;
	private static final long serialVersionUID = 642020160129L ;

	private static final String DEF_BODY  = "▼" ;
//	private static final String DEF_STYLE = "text-align:center;" ;						// 6.2.0.0 (2015/02/27)
	private static final String DEF_STYLE = "width:20px;text-align:center;padding:0;" ; // 5.9.4.3 (2016/01/15) padding

	/** command 引数に渡す事の出来る コマンド  新規 {@value} */
	private static final String CMD_NEW	 = "NEW" ;		// 5.1.7.0 (2010/06/01)

	private String		command		= CMD_NEW;			// 5.1.7.0 (2010/06/01)

	private String		column		;
	private String		gamenId		;
	private String		href		= "index.jsp";
	private String		target		= "CONTENTS";
	private String		onMark		= "true";			// true または 1
	private String		markList	= "true|TRUE|1";	// true または 1
	private String		height		;
	private String		width		;
	private String		imgsrc		;
	private String[]	keys		;
	private String[]	vals		;
	private String[]	rtnKeys		;
	private String		append		= "false";			// 5.8.5.0 (2015/03/06)

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public PopupTag() {
		super();		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		// 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
		if( useTag() ) {
			makeGamenLink();

			// GAMENID属性のＵＲＬへの自動セットは行わない。
			final ViewMarkerTag viewMark = (ViewMarkerTag)findAncestorWithClass( this,ViewMarkerTag.class );
			// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
			if( viewMark == null ) {
				if( markList.indexOf( onMark ) >= 0 ) {
					jspPrint( makePopup( false ) );
				}
			}
//			if( viewMark != null ) {
			else {
				set( "onMark"  ,onMark );
				set( "markList",markList );
				set( "body"    ,makePopup( true ) );

				if( column != null && column.length() > 0 ) {
					set( "column"  ,column );	// 4.0.0 (2005/08/31) 同一カラムの複数登録を許可します。
					viewMark.addAttribute( getAttributes() );
				}
			}
//			else {
//				if( markList.indexOf( onMark ) >= 0 ) {
//					jspPrint( makePopup( false ) );
//				}
//			}
		}
		return EVAL_PAGE ;
	}

	/**
	 * 使用しません。
	 *
	 * @return	(null 固定)
	 */
	@Override
	protected String makeTag() { return null; }

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		column		= null;
		gamenId		= null;
		href		= "index.jsp";
		target		= "CONTENTS";
		onMark		= "true";			// true または 1
		markList	= "true|TRUE|1";	// true または 1
		keys		= null;
		vals		= null;
		rtnKeys		= null;
		height		= null;
		width		= null;
		imgsrc		= null;
		command		= CMD_NEW;			// 5.1.7.0 (2010/06/01)
		append		= "false";			// 5.8.5.0 (2015/03/66)
	}

	/**
	 * リンクを作成します。
	 *
	 * @og.rev 4.2.4.0 (2008/07/01) keys,vals,rtnKeysが設定されていない場合、外側のog:columnタグのname値を設定するように変更
	 * @og.rev 5.1.7.0 (2010/06/01) 引数にcommandを追加
	 * @og.rev 5.2.1.0 (2010/10/01) height、width には、"px" などの単位を付ける前提での対応
	 * @og.rev 5.2.2.0 (2010/11/01) height、width が null の場合は、シングルクオートをつけない。
	 * @og.rev 6.2.0.0 (2015/02/27) onClick パラメータを設定できるようにし、後ろに、popupを追記します。
	 * @og.rev 5.8.5.0 (2015/03/06) append追加
	 * @og.rev 6.3.9.0 (2015/11/06) コンストラクタで初期化されていないフィールドを null チェックなしで利用している(findbugs)
	 *
	 * @param	useKey リンク作成時のキー有無 [true:KEY付き/false:KEY無し]
	 *
	 * @return	リンクタグ文字列
	 * @og.rtnNotNull
	 */
	private String makePopup( final boolean useKey ) {
		final StringBuilder rtn = new StringBuilder( BUFFER_MIDDLE );
		// 5.2.1.0 (2010/10/01)  px を付けると数字ではなくなるため、シングルクオーテーションでくくる。
		rtn.append( "ogPopup( '" ).append( get( "href" ) ).append( "'," );
		// 5.2.2.0 (2010/11/01) height、width が null の場合は、シングルクオートをつけない。
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		if( height == null ) { rtn.append( "null" ); }
		else {				   rtn.append( '\'' ).append( height ).append( '\'' ); }
//		if( height != null ) { rtn.append( '\'' ).append( height ).append( '\'' ); }
//		else {				   rtn.append( "null" ); }
		rtn.append( ',' );								// 6.0.2.5 (2014/10/31) char を append する。
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		if( width == null ) {  rtn.append( "null" ); }
		else {				   rtn.append( '\'' ).append( width ).append( '\'' );  }

//		if( width != null ) {  rtn.append( '\'' ).append( width ).append( '\'' );  }
//		else {				   rtn.append( "null" ); }
		rtn.append( ',' );								// 6.0.2.5 (2014/10/31) char を append する。

		String columnName = null;						// 4.2.4.0 2008/07/01
		// 6.3.9.0 (2015/11/06) コンストラクタで初期化されていないフィールドを null チェックなしで利用している(findbugs)
//		if( keys != null && keys.length > 0 ) {
		if( keys != null && keys.length > 0 && vals != null ) {
			rtn.append( "new Array(" );
			addQuote( rtn, keys[0], false );
			rtn.append( ',' );							// 6.0.2.5 (2014/10/31) char を append する。
			addQuote( rtn, vals[0], false );
			for( int i=1; i<keys.length; i++ ) {
				rtn.append( ',' );						// 6.0.2.5 (2014/10/31) char を append する。
				addQuote( rtn, keys[i], false );
				rtn.append( ',' );						// 6.0.2.5 (2014/10/31) char を append する。
				addQuote( rtn, vals[i], false );
			}
			rtn.append( ")," );
		}
		else {
			final ColumnTag outerColumn = (ColumnTag) findAncestorWithClass( this, ColumnTag.class );
			columnName = ( outerColumn == null ) ? null : outerColumn.getName();
			// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
			if( columnName == null ) {
				rtn.append( "null," );
			}
			else {
//			if( columnName != null ) {
				rtn.append( "new Array('XNAME','" )
					.append( columnName )
					.append( "','XVALUE'," )
					.append( columnName )
					.append( ".value)," );
			}
//			else {
//				rtn.append( "null," );
//			}
		}
		rtn.append( "" );

		if( rtnKeys != null && rtnKeys.length > 0 ) {
			rtn.append( "this,new Array(" );
			addQuote( rtn, rtnKeys[0], useKey );
			for( int i=1; i<rtnKeys.length; i++ ) {
				rtn.append( ',' );						// 6.0.2.5 (2014/10/31) char を append する。
				addQuote( rtn, rtnKeys[i], useKey );
			}
			rtn.append( ')' );							// 6.0.2.5 (2014/10/31) char を append する。
		}
		else if( columnName != null ) {
			rtn.append( "this,new Array('" )
				.append( columnName )
				.append( "')" );
		}
		else {
			rtn.append( "null,null" );
		}

		// 5.1.7.0 (2010/06/01) 引数にcommandを追加
		// 5.8.5.0 (2015/03/06) append追加
		rtn.append( ",'" ).append( command ).append( "','" ).append( append ).append( "');" );

		if( imgsrc == null ) {
			set( "type"   , "button" );
			set( "body"   , nval( getMsglbl(),DEF_BODY) );
			add( "onClick", rtn.toString() , ";" );				// 6.2.0.0 (2015/02/27) onClick を追記します。
			set( "style"  , nval( get( "style" ),DEF_STYLE) );
			return XHTMLTag.button( getAttributes() ) ;
		}
		else {
			set( "src"    , imgsrc );
			set( "title"  , getMsglbl() );
			add( "onClick", rtn.toString() , ";" );				// 6.2.0.0 (2015/02/27) onClick を追記します。
			set( "style"  , get( "style" ) );
			return XHTMLTag.img( getAttributes() ) ;
		}
	}

	/**
	 * 引数 val の設定値を設定する簡易メソッド。
	 *
	 * @param	buf		結果を格納するStringBuilderオブジェクト
	 * @param	val		設定値
	 * @param	useKey リンク作成時のキー有無 [true:KEY付き/false:KEY無し]
	 *
	 * @return	(引数のバッファと同じオブジェクト)
	 * @og.rtnNotNull
	 */
	private StringBuilder addQuote( final StringBuilder buf, final String val, final boolean useKey ) {
		if( useKey ) {
			if( val != null && val.endsWith( ".value" ) ) {
				buf.append( val.substring( 0,val.length()-6 ) ).append( "__{I}.value" );
			}
			else {
				buf.append( '\'' ).append( val ).append( "__{I}'" );
			}
		}
		else {
			if( val != null && val.endsWith( ".value" ) ) {
				if( StringUtil.startsChar( val , '[' ) ) {			// 6.2.0.0 (2015/02/27) １文字 String.startsWith
					buf.append( val.substring( 1,val.length()-7 ) ).append( "__{I}.value" );
				}
				else {
					buf.append( val );
				}
			}
			else {
				buf.append( '\'' ).append( val ).append( '\'' );
			}
		}
		return buf ;
	}

	/**
	 * 画面ＩＤとhref から、指定のＵＲＬを作成します。
	 */
	private void makeGamenLink() {
		if( gamenId == null || gamenId.isEmpty() ) { return ; }

		final GUIInfo guiInfo = getGUIInfo( gamenId );		// 4.0.0 (2005/01/31)
		if( guiInfo == null ) { return ; }	// 見つからない場合は、アクセス不可

		final String address = guiInfo.getRealAddress( get( "href" ) );
		final String url = getRequestParameter( address );
		set( "href",url );

		if( get( "title" ) == null ) {
			set( "title","To:" + guiInfo.getLabel() );
		}
	}

	/**
	 * 【TAG】この要素をリンクの対象とするための名前を指定します。
	 *
	 * @og.tag この要素をリンクの対象とするための名前を指定します。
	 *
	 * @param	name 名前
	 */
	public void setName( final String name ) {
		set( "name",getRequestParameter( name ) );
	}

	/**
	 * 【TAG】リンクをつけたいカラムを指定します。
	 *
	 * @og.tag
	 * このカラム名のTableModelに対して、 ViewLink タグが存在する場合は,
	 * リンクを張ります。
	 *
	 * @param	clm カラム名
	 */
	public void setColumn( final String clm ) {
		column = nval( getRequestParameter( clm ),column );
	}

	/**
	 * 【TAG】画面ID を指定します。
	 *
	 * @og.tag
	 * gamenId 属性は、画面IDをキーに、実アドレスを求めるのに使用します。
	 * 画面IDが指定された場合は、実アドレスに変換する。指定されない場合は、
	 * href 属性をそのままリンク情報を作成します。
	 * 画面IDを指定する場合の href は、JSPファイル名(例えば、index.jsp 等)です。
	 *
	 * @param	id	画面ID
	 */
	public void setGamenId( final String id ) {
		gamenId = nval( getRequestParameter( id ),gamenId );
	}

	/**
	 * 【TAG】ポップアップ先のURLを指定します(初期値:index.jsp)。
	 *
	 * @og.tag
	 * 通常、gamenId を指定している場合は、それ以降(index.jsp , result.jsp など)を指定します。
	 * 通常のリンクの場合、GAMENID/href というアドレスに合成されます。( GE0001/result.jsp など )
	 * (正確には、画面IDではなく、対応する画面ADDRESSを使用します。)
	 * ポップアップ専用のリンクなので、初期値は、index.jsp です。
	 *
	 * @param	href ポップアップ先のURL
	 */
	public void setHref( final String href ) {
		set( "href",nval( getRequestParameter( href ),this.href ) );
	}

	/**
	 * 【TAG】ポップアップ先の文字コードセットを指定します。
	 *
	 * @og.tag ポップアップ先の文字コードセットを指定します。
	 *
	 * @param	charset 文字コードセット
	 */
	public void setCharset( final String charset ) {
		set( "charset",getRequestParameter( charset ) );
	}

	/**
	 * 【TAG】ポップアップ先の表示ターゲットを指定します(初期値:CONTENTS)。
	 *
	 * @og.tag ポップアップ先の表示ターゲットを指定します。
	 *
	 * @param	target 表示ターゲット
	 */
	public void setTarget( final String target ) {
		set( "target",nval( getRequestParameter( target ),this.target ) );
	}

	/**
	 * 【TAG】ポップアップ先に渡す入力キー配列をCSV形式で複数指定します。
	 *
	 * @og.tag ポップアップ先に渡す入力キー配列を指定します。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @param	key 渡す入力キー配列(CSV形式)
	 */
	public void setKeys( final String key ) {
		keys = getCSVParameter( key );
	}

	/**
	 * 【TAG】ポップアップ先に渡す値配列をCSV形式で複数指定します。
	 *
	 * @og.tag ポップアップ先に渡す値配列を指定します。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @param	val 渡す値配列(CSV形式)
	 */
	public void setVals( final String val ) {
		vals = getCSVParameter( val );
	}

	/**
	 * 【TAG】ポップアップ先より受け取るキー配列をCSV形式で複数指定します。
	 *
	 * @og.tag ポップアップ先より受け取るキー配列を指定します。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @param	key 受け取るキー配列(CSV形式)
	 */
	public void setRtnKeys( final String key ) {
		rtnKeys = getCSVParameter( key );
	}

	/**
	 * 【TAG】マークを処理する(true or 1)/処理しない(false or 0)の設定を指定します(初期値:true)。
	 *
	 * @og.tag
	 * マークを処理する場合は、"true"(または "1")
	 * 処理しない場合は, "true以外"(または "0")をセットします。
	 * 初期値は、 "true"(マークを処理する)です。
	 * さらに、[カラム名] で、動的にカラムの値で、マークをする、しないを
	 * 選ぶ事が可能になります。値は、"true"(または "1") で、マークします。
	 * 追記 3.5.2.0 (2003/10/20)：markList属性に、処理対象文字列郡を指定できます。
	 * これにより、マークを処理するかどうかの判断が、true,1 以外にも使用できるようになりました。
	 *
	 * @og.rev 3.5.0.0 (2003/09/17) onMark に、[カラム名] の値をセットできるように修正。
	 *
	 * @param	flag マークの処理 [true,1:処理する/それ以外:処理しない]
	 */
	public void setOnMark( final String flag ) {
		onMark = nval( getRequestParameter( flag ),onMark );
	}

	/**
	 * 【TAG】処理するマークを含むような文字列を、"|"区切りの文字列で指定します(初期値:"true|TRUE|1")。
	 *
	 * @og.tag
	 * markListで与えられた文字列に、onMark文字列(大文字/小文字の区別あり)が
	 * 含まれていれば、処理します。
	 * 例えば、"A","B","C" という文字列が、onMark で指定された
	 * 場合に処理するようにしたい場合は、"A|B|C" をセットします。
	 * markList には、[カラム名]指定が可能です。(ただし、単独 linkタグ時は除く)
	 * 初期値は、 "true|TRUE|1"です。
	 *
	 * @param	list 処理するマーク (indexOf による含む/含まない判定)
	 */
	public void setMarkList( final String list ) {
		markList = nval( getRequestParameter( list ),markList );
	}

	/**
	 * 【TAG】ダイアログの縦幅をピクセル数で指定します。
	 *
	 * @og.tag
	 * 縦幅を指定します。単位も含めて入力してください。
	 * 設定しない場合は、ogPopup JavaScript の内部初期値が採用されます。
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) height、width には、"px" などの単位を付ける前提での対応
	 *
	 * @param   ht ダイアログの縦幅(ピクセル数)
	 */
	public void setHeight( final String ht ) {
		height = nval( getRequestParameter( ht ),height );
		if( height != null && ! height.endsWith( "px" ) ) { height += "px"; }	// 5.2.1.0 (2010/10/01)
	}

	/**
	 * 【TAG】ダイアログの横幅をピクセル数で指定します。
	 *
	 * @og.tag
	 * 横幅を指定します。単位も含めて入力してください。
	 * 設定しない場合は、ogPopup JavaScript の内部初期値が採用されます。
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) height、width には、"px" などの単位を付ける前提での対応
	 *
	 * @param   wh ダイアログの横幅(ピクセル数)
	 */
	public void setWidth( final String wh ) {
		width = nval( getRequestParameter( wh ),width );
		if( width != null && ! width.endsWith( "px" ) ) { width += "px"; }	// 5.2.1.0 (2010/10/01)
	}

	/**
	 * 【TAG】表示させる画像のURLを指定します。
	 *
	 * @og.tag
	 * この属性を設定すると、自動的に イメージリンクでの onClick属性で
	 * ポップアップの起動画面を作成します。
	 * この属性が設定されていない場合は、button での起動になります。
	 *
	 * @param   src 画像のURL
	 */
	public void setSrc( final String src ) {
		imgsrc = nval( getRequestParameter( src ),imgsrc );
	}

	/**
	 * 【TAG】(通常使いません)処理の実行を指定する command を設定できます(初期値:NEW)。
	 *
	 * @og.tag
	 * ここで指定されたcommandは、ポップアップ画面の引数として渡されます。
	 * 初期値は、NEW です。
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) 新規作成
	 *
	 * @param	cmd コマンド
	 */
	public void setCommand( final String cmd ) {
		command = nval( getRequestParameter( cmd ),command );
	}

	/**
	 * 【TAG】返り値をキャレット位置に追記する(true)/しない(false)の設定を指定します(初期値:false)。
	 *
	 * @og.tag
	 * ポップアップからの返り値を、元の値と置換するのではなく、キャレット位置に挿入するかどうかを指定します。
	 * 
	 * @og.rev 5.8.5.0 (2015/02/06)
	 *
	 * @param	flg 返り値のキャレット位置に追記 [true:追記/それ以外:置換]
	 */
	public void setAppend( final String flg ) {
		append = nval( getRequestParameter( flg ),append );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "column"		,column		)
				.println( "gamenId"		,gamenId	)
				.println( "href"		,href		)
				.println( "target"		,target		)
				.println( "onMark"		,onMark		)
				.println( "markList"	,markList	)
				.println( "height"		,height		)
				.println( "width"		,width		)
				.println( "keys"		,keys		)
				.println( "vals"		,vals		)
				.println( "rtnKeys"		,rtnKeys	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString()
			+ CR
			+ super.toString() ;
	}
}
