/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.db.AbstractRenderer;
import org.opengion.hayabusa.db.CellRenderer;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.fukurou.util.StringUtil;

import java.text.DecimalFormat;

/**
 * DECIMAL レンデラーは、カラムのデータをDecimal(１０進数、小数)表示する場合に
 * 使用するクラスです。
 * 負数の場合はspanタグclass="minus"を付けて出力します。
 *
 * 表示パラメータに与えられた文字列は、java.text.DecimalFormat を
 * 使用してフォーマットされます。
 * フォーマット変換前に、カンマなどの数値変換時にエラーになる情報を削除しておきます。
 *
 * フォーマットルールは、{@link java.text.DecimalFormat} を参照願います。
 * 標準のフォーマットは"#,##0.#"です。
 *
 *  カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.rev 3.5.0.0 (2003/09/17) 新規作成
 * @og.rev 5.4.3.6 (2012/01/19) コメント修正
 * @og.group データ表示
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Renderer_DECIMAL extends AbstractRenderer {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.0.4.0 (2014/11/28)" ;

	private final DecimalFormat format ;
	private final String defValue ;
	private final String noDisplayVal ;		// 5.6.2.3 (2013/03/22)

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 5.6.2.3 (2013/03/22) noDisplayVal 変数初期化
	 */
	public Renderer_DECIMAL() {
		super();		// 6.4.1.1 (2016/01/16) PMD refactoring. It is a good practice to call super() in a constructor
		format			= null;
		defValue		= "";		// 5.1.1.0 (2009/12/01)
		noDisplayVal	= null;		// 5.5.1.0 (2012/04/03)
	}

	/**
	 * 引数つきコンストラクター。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @og.rev 5.1.1.0 (2009/12/01) 初期値がnullの場合は、defValueはnullとする。
	 * @og.rev 5.6.2.3 (2013/03/22) noDisplayVal 変数追加
	 */
	private Renderer_DECIMAL( final DBColumn clm ) {
		super();		// 6.4.1.1 (2016/01/16) PMD refactoring. It is a good practice to call super() in a constructor

		String fm = clm.getRendererParam();
		if( fm == null || fm.isEmpty() || fm.equals( "_" ) ) {
			fm = "#,##0.#";
		}
		format = new DecimalFormat( fm );

		// 5.1.1.0 (2009/12/01)
		final String defv = clm.getDefault();
		if( defv == null || defv.isEmpty() ) {
			defValue = "";
		}
		else {
			final double dd = StringUtil.parseDouble( defv );
			defValue = format.format( dd );
		}
		noDisplayVal = clm.getNoDisplayVal();	// 5.6.2.3 (2013/03/22)
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellRendererオブジェクト
	 * @og.rtnNotNull
	 */
	public CellRenderer newInstance( final DBColumn clm ) {
		return new Renderer_DECIMAL( clm );
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * ここでは、値が 0 の場合に、初期値を使うロジックが組み込まれており、
	 * 従来はこれを利用して、初期値にゼロ文字列を設定することで、"0" を
	 * 非表示文字として扱っていました。
	 * 互換性の問題があるので、既存の機能は残しますが、この処理は、noDisplayVal を
	 * 利用した処理に置き換えてください。
	 * 他の処理（NUMBER,MONEY,DBMENUなど）は、noDisplayVal を使います。
	 *
	 * @og.rev 3.8.5.2 (2006/05/31) カンマ編集された数字の対応
	 * @og.rev 5.3.1.0 (2009/12/01) 値が0の場合は、初期値を適用する。
	 * @og.rev 5.6.2.3 (2013/03/22) noDisplayVal 変数追加
	 * @og.rev 6.0.4.0 (2014/11/28) ロジックの共通化
	 *
	 * @param	value 入力値
	 *
	 * @return	データの表示用文字列
	 */
	@Override
	public String getValue( final String value ) {
		return getValue( value , true );

	}

	/**
	 * データ出力用の文字列を作成します。
	 * ファイル等に出力する形式を想定しますので、HTMLタグを含まない
	 * データを返します。
	 * 基本は、#getValue( String ) をそのまま返します。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) データ出力用のレンデラー
	 *
	 * @param   value 入力値
	 *
	 * @return  データ出力用の文字列
	 * @see		#getValue( String )
	 */
	@Override
	public String getWriteValue( final String value ) {
		return getValue( value , false );
	}

	/**
	 * データ表示用/出力用の文字列を作成します。
	 * 第二引数の isView == true で、データ表示用文字列を、false で
	 * データ出力用の文字列を作成します。
	 * 処理の共通化を行うためのメソッドです。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) ロジックの共通化
	 *
	 * @param   value  入力値
	 * @param   isView データ表示用かどうか(true:表示用/false:出力用)
	 *
	 * @return  データ表示用/出力用の文字列
	 * @see		#getValue( String )
	 */
	private String getValue( final String value , final boolean isView ) {
		// 5.6.2.3 (2013/03/22) noDisplayVal 変数追加
		if( noDisplayVal != null && noDisplayVal.equalsIgnoreCase( value ) ) { return "" ; }

		if( value == null || value.trim().isEmpty() || "0".equals( value ) ) {
			return defValue;
		}

		final double dd = StringUtil.parseDouble( value );

		String rtn ;
		synchronized( format ) {
			rtn = format.format( dd );
		}

		if( isView && dd < 0.0 ) {
			rtn = "<span class=\"minus\">" + rtn + "</span>";
		}

		return rtn;
	}
}
