/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.process;

import org.opengion.fukurou.db.ConnectionFactory;
import org.opengion.fukurou.util.Argument;
import org.opengion.fukurou.db.ApplicationInfo;
import org.opengion.fukurou.system.LogWriter;

import java.util.Set ;
import java.util.Map ;
import java.util.LinkedHashMap ;
import java.net.InetAddress;
import java.net.UnknownHostException;

import java.sql.Connection;

/**
 * Process_DBParam は、他のプロセスへ共通のデータベース接続を割り当てる為の、
 * ParamProcess インターフェースの実装クラスです。
 *
 * DB接続 が必要な Process (DBCountFilter、DBMerge、DBReader、DBWriterなど)を
 * 使用して処理する場合に、接続を指定することができます。
 * DBID(接続先) は、Process_DBParam の -configFile で指定する DBConfig.xml ファイルを使用します。
 *
 * @og.formSample
 *  Process_DBParam -infoUSER=C00000 -infoPGID=GE1234 -configFile=DBConfig.xml
 *
 *   [ -infoUSER=実行ユーザー       ] ： DB接続履歴取得用の実行ユーザー(例:C00000)
 *   [ -infoPGID=実行プログラムID   ] ： DB接続履歴取得用の実行プログラムID(例:GE1234)
 *   [ -configFile=実行プログラムID ] ： DB接続情報設定 XMLファイル(例:DBConfig.xml)
 *   [ -display=[false/true]        ] ： trueは、接続状況を詳細表示します(初期値:false)
 *
 * @og.rev 4.0.0.0 (2007/11/22) DBConfig.xml による DBID(接続先)指定に変更。
 * @og.rev 6.3.1.0 (2015/06/28) 履歴取得用パラメータの必須解除
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Process_DBParam extends AbstractProcess implements ParamProcess {
	/** 実行しているサーバーの名称 */
	private static final String HOST_NAME ;
	/** 実行しているサーバーのIPアドレス */
	private static final String HOST_ADRS ;

	private ApplicationInfo appInfo	;
	private boolean			display	;			// 表示しない

	// 5.3.4.0 (2011/04/01) bulkData 関係のメソッドを追加
	private Set<String> bulkData ;

	private static final Map<String,String> MUST_PROPARTY   ;	// ［プロパティ］必須チェック用 Map
	private static final Map<String,String> USABLE_PROPARTY ;	// ［プロパティ］整合性チェック Map

	static {
		String dmnHost ;
		String dnmAdrs ;
		try {
			final InetAddress address = InetAddress.getLocalHost();
			dmnHost = address.getHostName() ;
			dnmAdrs = address.getHostAddress() ;
		}
		catch( UnknownHostException ex ) {
			dmnHost = "Unknown";
			dnmAdrs = "Unknown";
		}
		HOST_NAME = dmnHost;
		HOST_ADRS = dnmAdrs;
	}

	static {
		MUST_PROPARTY = new LinkedHashMap<>();
		// 6.3.1.0 (2015/06/28) 必須から外します。
//		MUST_PROPARTY.put( "infoUSER",	"DB接続履歴取得用の実行ユーザー" );
//		MUST_PROPARTY.put( "infoPGID",	"DB接続履歴取得用の実行プログラムID" );
//		MUST_PROPARTY.put( "configFile",	"DB接続情報設定 XMLファイル" );

		USABLE_PROPARTY = new LinkedHashMap<>();
		USABLE_PROPARTY.put( "infoUSER"	, "DB接続履歴取得用の実行ユーザー" );
		USABLE_PROPARTY.put( "infoPGID"	, "DB接続履歴取得用の実行プログラムID" );
		USABLE_PROPARTY.put( "configFile", "DB接続情報設定 XMLファイル" );
		USABLE_PROPARTY.put( "display"	, "trueは、接続状況を詳細表示します(初期値:false)" );		// 6.3.1.0 (2015/06/28) 追加
	}

	/**
	 * デフォルトコンストラクター。
	 * このクラスは、動的作成されます。デフォルトコンストラクターで、
	 * super クラスに対して、必要な初期化を行っておきます。
	 *
	 */
	public Process_DBParam() {
		super( "org.opengion.fukurou.process.Process_DBParam",MUST_PROPARTY,USABLE_PROPARTY );
	}

	/**
	 * ApplicationInfoオブジェクトを登録します。
	 * これは、通常の初期処理ではなく、タグリブから起動される場合のみ
	 * 呼ばれるメソッドです。
	 * 初期処理メソッド(init)では、appInfo がセット済みの場合は、
	 * ConnectionFactoryの初期化を行いません。
	 *
	 * @og.rev 4.3.1.1 (2008/09/04) 新規追加(taglib呼出専用)
	 *
	 * @param   appInfo アプリ情報オブジェクト
	 */
	public void setAppInfo( final ApplicationInfo appInfo ) {
		this.appInfo = appInfo;
	}

	/**
	 * プロセスの初期化を行います。初めに一度だけ、呼び出されます。
	 * 初期処理(ファイルオープン、ＤＢオープン等)に使用します。
	 *
	 * @og.rev 4.3.1.1 (2008/09/04) taglib呼出時は、ConnectionFactoryの初期化を行わない
	 * @og.rev 6.3.1.0 (2015/06/28) display属性の追加
	 *
	 * @param   paramProcess データベースの接続先情報などを持っているオブジェクト
	 */
	@Override
	public void init( final ParamProcess paramProcess ) {
		final Argument arg = getArgument();			// 6.3.1.0 (2015/06/28) display属性の追加のため。

		// 4.3.1.1 (2008/09/04) taglib呼出時は、ConnectionFactoryの初期化を行わない
		if( appInfo == null ) {
//			final Argument arg = getArgument();

			final String infoUSER	= arg.getProparty( "infoUSER" );		// DB接続履歴取得用の実行ユーザー
			final String infoPGID	= arg.getProparty( "infoPGID" );		// DB接続履歴取得用の実行プログラムID
			final String configFile = arg.getProparty( "configFile" );		// DB接続情報設定 XMLファイル

			appInfo = new ApplicationInfo();
			// JavaVM 起動時のユーザーID,IPアドレス,ホスト名をセットします。
			appInfo.setClientInfo( infoUSER,HOST_ADRS,HOST_NAME );

			// 画面ID,操作,プログラムID
			appInfo.setModuleInfo( infoPGID,null,"fukurou" );

			// DBID接続情報の取得先の設定
			ConnectionFactory.init( null,configFile );
		}

		display = arg.getProparty( "display",display );		// 6.3.1.0 (2015/06/28)
	}

	/**
	 * 指定の 接続先ID に対する コネクションを返します。
	 *
	 * @param	key	接続先ID
	 *
	 * @return	コネクション
	 * @throws	RuntimeException DB接続先が未設定の場合
	 * @og.rtnNotNull
	 */
	@Override
	public Connection getConnection( final String key ) {
		return ConnectionFactory.connection( key,appInfo );
	}

	/**
	 * 検索した結果が設定された Set オブジェクトを設定します。
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) 新規追加
	 *
	 * @param	bulkData	検索した結果が設定されたSetオブジェクト
	 */
	@Override
	public void setBulkData( final Set<String> bulkData ) {
		this.bulkData = bulkData;
	}

	/**
	 * 検索した結果が設定された Set オブジェクトを返します。
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) 新規追加
	 *
	 * @return	検索した結果が設定された Setオブジェクト
	 */
	@Override
	public Set<String> getBulkData() {
		return bulkData ;
	}

	/**
	 * プロセスの終了を行います。最後に一度だけ、呼び出されます。
	 * 終了処理(ファイルクローズ、ＤＢクローズ等)に使用します。
	 *
	 * @og.rev 4.0.0.0 (2007/11/27) commit,rollback,remove 処理を追加
	 *
	 * @param   isOK トータルで、OKだったかどうか[true:成功/false:失敗]
	 */
	@Override
	public void end( final boolean isOK ) {
		// 何もありません。(PMD エラー回避)
	}

	/**
	 * プロセスの処理結果のレポート表現を返します。
	 * 処理プログラム名、入力件数、出力件数などの情報です。
	 * この文字列をそのまま、標準出力に出すことで、結果レポートと出来るような
	 * 形式で出してください。
	 *
	 * @return   処理結果のレポート
	 */
	@Override
	public String report() {
//		final String report = "[" + getClass().getName() + "]" + CR
//							+ ConnectionFactory.information();

		final String report = "[" + getClass().getName() + "]" + CR
							+ ConnectionFactory.information( display ) ;

		return report ;
	}

	/**
	 * このクラスの使用方法を返します。
	 *
	 * @return	このクラスの使用方法
	 * @og.rtnNotNull
	 */
	@Override
	public String usage() {
		final StringBuilder buf = new StringBuilder( BUFFER_LARGE )
			.append( "Process_DBParam は、他のプロセスへ共通のデータベース接続を割り当てる為の、"	).append( CR )
			.append( "ParamProcess インターフェースの実装クラスです。"								).append( CR )
			.append( CR )
			.append( "DB接続 が必要な Process (DBCountFilter、DBMerge、DBReader、DBWriterなど)を"	).append( CR )
			.append( "使用して処理する場合に、接続を指定することができます。"						).append( CR )
			.append( "DBID(接続先) は、-configFile で指定する DBConfig.xml ファイルを使用します。"	).append( CR )
			.append( CR )
			.append( "引数文字列中に空白を含む場合は、ダブルコーテーション(\"\") で括って下さい。"	).append( CR )
			.append( "引数文字列の 『=』の前後には、空白は挟めません。必ず、-key=value の様に"		).append( CR )
			.append( "繋げてください。"																).append( CR )
			.append( CR )
			.append( "[ -infoUSER=実行ユーザー       ] ： DB接続履歴取得用の実行ユーザー(例:C00000)"		).append( CR )
			.append( "[ -infoPGID=実行プログラムID   ] ： DB接続履歴取得用の実行プログラムID(例:GE1234)"	).append( CR )
			.append( "[ -configFile=実行プログラムID ] ： DB接続情報設定 XMLファイル(例:DBConfig.xml)"		).append( CR )
			.append( "[ -display=[false/true]        ] ： trueは、接続状況を詳細表示します(初期値:false)"	).append( CR )
			.append( CR ).append( CR )
			.append( getArgument().usage() ).append( CR );

		return buf.toString();
	}

	/**
	 * このクラスは、main メソッドから実行できません。
	 *
	 * @param	args	コマンド引数配列
	 */
	public static void main( final String[] args ) {
		LogWriter.log( new Process_DBParam().usage() );
	}
}
