/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.process;

import org.opengion.fukurou.system.OgRuntimeException ;		// 6.4.2.0 (2016/01/29)
import org.opengion.fukurou.util.Argument;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.system.Closer ;
import org.opengion.fukurou.system.LogWriter;
import org.opengion.fukurou.util.CommentLineParser;
import org.opengion.fukurou.util.FileInfo;							// 6.4.0.2 (2015/12/11)

import java.util.Map ;
import java.util.LinkedHashMap ;

import java.io.File;
import java.io.PrintWriter;
import java.io.BufferedReader;
import java.io.IOException;
import java.nio.charset.CharacterCodingException;					// 6.3.1.0 (2015/06/28)

/**
 * Process_FileCopy は、上流から受け取った FileLineModel を処理する、
 * ChainProcess インターフェースの実装クラスです。
 *
 * 上流から受け取った FileLineModel の ファイルから、inPath の共通パス
 * 以下のファイルを、outPath の共通パス以下にコピーします。
 * コピーの種類は、バイナリか、テキストで、テキストの場合は、エンコード
 * 変換も行うことが可能です。
 * inPath と outPath が同じ、または、outPath が未設定の場合は、入力と出力が
 * 同じですので、自分自身のエンコード変換処理を行うことになります。
 *
 * コピーされるファイルのファイル名は、入力ファイル名と同一です。保存される
 * フォルダが異なります。(同一にすることも可能です。)
 *
 * useOmitCmnt=true に設定すると、ファイル中のコメントを除外してコピーします。
 * ただし、使用できるのは、アスキーファイル(binary=false)の時だけです。
 *
 * 上流プロセスでは、Name 属性として、『File』を持ち、値は、Fileオブジェクト
 * である、Process_FileSearch を使用するのが、便利です。それ以外のクラスを
 * 使用する場合でも、Name属性と、File オブジェクトを持つ LineModel を受け渡し
 * できれば、使用可能です。
 *
 * 引数文字列中に空白を含む場合は、ダブルコーテーション("") で括って下さい。
 * 引数文字列の 『=』の前後には、空白は挟めません。必ず、-key=value の様に
 * 繋げてください。
 *
 * @og.formSample
 *  Process_FileCopy -inPath=入力共通パス -inEncode=Windows-31J -outPath=出力共通パス -outEncode=UTF-8
 *
 *     -inPath=入力共通パス         ：上流で検索されたファイルパスの共通部分
 *   [ -inEncode=入力エンコード   ] ：入力ファイルのエンコードタイプ
 *   [ -outPath=出力共通パス      ] ：出力するファイルパスの共通部分
 *   [ -outEncode=出力エンコード  ] ：出力ファイルのエンコードタイプ
 *   [ -binary=[false/true]       ] ：trueは、バイナリファイルのコピー(初期値:false)
 *   [ -changeCrLf=[false/true]   ] ：trueは、バイナリファイルのコピー時にCR+LFに変換します(初期値:false)
 *   [ -keepTimeStamp=[false/true]] ：trueは、コピー元のファイルのタイムスタンプで作成します(初期値:false)
 *   [ -useOmitCmnt=[false/true]  ] ：ファイル中のコメントを除外してコピーを行うかどうかを指定(初期値:false)
 *   [ -display=[false/true]      ] ：trueは、コピー状況を表示します(初期値:false)
 *   [ -debug=[false/true]        ] ：デバッグ情報を標準出力に表示する(true)かしない(false)か(初期値:false[表示しない])
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Process_FileCopy extends AbstractProcess implements ChainProcess {
	private File	tempFile		;

	private String	inPath			;
	private String	inEncode		;
	private String	outPath			;
	private String	outEncode		;
	private boolean	binary			;
	private boolean	changeCrLf		;			// 4.2.2.0 (2008/05/10)
	private boolean	keepTimeStamp	;			// 5.1.5.0 (2010/04/01)
	private boolean	useOmitCmnt		;			// 5.7.4.0 (2014/03/07)
	private boolean	display			;
	private boolean	debug			;			// 5.7.3.0 (2014/02/07) デバッグ情報

	private int		inPathLen	;
	private boolean	isEquals	;
	private int		inCount		;

	private static final Map<String,String> MUST_PROPARTY   ;		// ［プロパティ］必須チェック用 Map
	private static final Map<String,String> USABLE_PROPARTY ;		// ［プロパティ］整合性チェック Map

	static {
		MUST_PROPARTY = new LinkedHashMap<>();
		MUST_PROPARTY.put( "inPath",	"コピー元のファイル基準パス" );

		USABLE_PROPARTY = new LinkedHashMap<>();
		USABLE_PROPARTY.put( "inEncode",		"コピー元のファイルのエンコードタイプ" );
		USABLE_PROPARTY.put( "outPath",		"コピー先のファイル基準パス" );
		USABLE_PROPARTY.put( "outEncode",	"コピー先のファイルのエンコードタイプ" );
		USABLE_PROPARTY.put( "binary",		"trueは、バイナリファイルをコピーします(初期値:false)" );
		USABLE_PROPARTY.put( "changeCrLf",	"trueは、バイナリファイルのコピー時にCR+LFに変換します(初期値:false)" );		// 4.2.2.0 (2008/05/10)
		USABLE_PROPARTY.put( "keepTimeStamp","trueは、コピー元のファイルのタイムスタンプで作成します(初期値:false)" );	// 5.1.5.0 (2010/04/01)
		USABLE_PROPARTY.put( "useOmitCmnt" 	,"ファイル中のコメントを除外してコピーを行うかどうかを指定(初期値:false)" );		// 5.7.4.0 (2014/03/07)
		USABLE_PROPARTY.put( "display",		"trueは、コピー状況を表示します(初期値:false)" );
		USABLE_PROPARTY.put( "debug",		"デバッグ情報を標準出力に表示する(true)かしない(false)か" +
												CR + "(初期値:false:表示しない)" );		// 5.7.3.0 (2014/02/07) デバッグ情報
	}

	/**
	 * デフォルトコンストラクター。
	 * このクラスは、動的作成されます。デフォルトコンストラクターで、
	 * super クラスに対して、必要な初期化を行っておきます。
	 *
	 */
	public Process_FileCopy() {
		super( "org.opengion.fukurou.process.Process_FileCopy",MUST_PROPARTY,USABLE_PROPARTY );
	}

	/**
	 * プロセスの初期化を行います。初めに一度だけ、呼び出されます。
	 * 初期処理(ファイルオープン、ＤＢオープン等)に使用します。
	 *
	 * @og.rev 4.2.2.0 (2008/05/10) changeCrLf 属性対応
	 * @og.rev 5.1.5.0 (2010/04/01) keepTimeStamp 属性の追加
	 *
	 * @param   paramProcess データベースの接続先情報などを持っているオブジェクト
	 */
	public void init( final ParamProcess paramProcess ) {
		final Argument arg = getArgument();

		inPath			= arg.getProparty( "inPath"			);
		outPath			= arg.getProparty( "outPath"		);
		inEncode		= arg.getProparty( "inEncode"		,System.getProperty("file.encoding") );
		outEncode		= arg.getProparty( "outEncode"		,System.getProperty("file.encoding") );
		binary			= arg.getProparty( "binary"			,binary );
		changeCrLf		= arg.getProparty( "changeCrLf"		,changeCrLf );		// 4.2.2.0 (2008/05/10)
		keepTimeStamp	= arg.getProparty( "keepTimeStamp"	,keepTimeStamp );	// 5.1.5.0 (2010/04/01)
		useOmitCmnt		= arg.getProparty( "useOmitCmnt"	,useOmitCmnt );		// 5.7.4.0 (2014/03/07)
		display			= arg.getProparty( "display"		,display );
		debug			= arg.getProparty( "debug"			,debug );			// 5.7.3.0 (2014/02/07) デバッグ情報

		// 入力と出力が同じか？
		isEquals  = outPath == null || inPath.equalsIgnoreCase( outPath );
		inPathLen = inPath.length();

		if( binary ) {
			// 4.2.2.0 (2008/05/10) 判定ミスの修正
			if( ! inEncode.equalsIgnoreCase( outEncode ) ) {
				final String errMsg = "バイナリコピー時には、入出力のエンコードは同じ必要があります。" + CR
							+ " inEncode=[" + inEncode + "] , outEncode=[" + outEncode + "]" ;
				throw new OgRuntimeException( errMsg );
			}
			if( isEquals ) {
				final String errMsg = "入出力が同じファイルのバイナリコピーはできません。" + CR
							+ " inPath=[" + inPath + "] , outPath=[" + outPath + "]" ;
				throw new OgRuntimeException( errMsg );
			}
			// 5.7.4.0 (2014/03/07) コメント部分を削除する機能は、binary では使えません。
			if( useOmitCmnt ) {
				final String errMsg = "コメント部分を削除する機能(useOmitCmnt=true)は、バイナリコピーでは使えません。" + CR
							+ " inPath=[" + inPath + "] , outPath=[" + outPath + "]" ;
				throw new OgRuntimeException( errMsg );
			}
		}

		// 入力と出力が同じ場合は、中間ファイルを作成します。
		if( isEquals ) {
			try {
				tempFile = File.createTempFile( "X", ".tmp", new File( outPath ) );
				tempFile.deleteOnExit();
			}
			catch( IOException ex ) {
				final String errMsg = "中間ファイル作成でエラーが発生しました。" + CR
							+ " outPath=[" + outPath + "]" ;
				throw new OgRuntimeException( errMsg,ex );
			}
		}
	}

	/**
	 * プロセスの終了を行います。最後に一度だけ、呼び出されます。
	 * 終了処理(ファイルクローズ、ＤＢクローズ等)に使用します。
	 *
	 * @param   isOK トータルで、OKだったかどうか[true:成功/false:失敗]
	 */
	public void end( final boolean isOK ) {
		tempFile  = null;
	}

	/**
	 * 引数の LineModel を処理するメソッドです。
	 * 変換処理後の LineModel を返します。
	 * 後続処理を行わない場合(データのフィルタリングを行う場合)は、
	 * null データを返します。つまり、null データは、後続処理を行わない
	 * フラグの代わりにも使用しています。
	 * なお、変換処理後の LineModel と、オリジナルの LineModel が、
	 * 同一か、コピー(クローン)かは、各処理メソッド内で決めています。
	 * ドキュメントに明記されていない場合は、副作用が問題になる場合は、
	 * 各処理ごとに自分でコピー(クローン)して下さい。
	 *
	 * @og.rev 4.0.0.0 (2007/11/28) メソッドの戻り値をチェックします。
	 * @og.rev 4.2.2.0 (2008/05/10) changeCrLf 属性対応
	 * @og.rev 4.2.3.0 (2008/05/26) LineModel が FileLineModel でない場合の処理
	 * @og.rev 5.1.5.0 (2010/04/01) keepTimeStamp 属性の追加
	 * @og.rev 5.1.6.0 (2010/05/01) changeCrLf 属性が、.FileUtil#changeCrLfcopy メソッドへの移動に伴う対応
	 * @og.rev 5.7.2.2 (2014/01/24) エラー時にデータも出力します。
	 * @og.rev 6.3.1.0 (2015/06/28) nioを使用すると UTF-8とShuft-JISで、エラーになる。
	 * @og.rev 6.4.0.2 (2015/12/11) CommentLineParser 改造。
	 *
	 * @param	data	オリジナルのLineModel
	 *
	 * @return	処理変換後のLineModel
	 */
	public LineModel action( final LineModel data ) {
		inCount++ ;
		final FileLineModel fileData ;
		if( data instanceof FileLineModel ) {
			fileData = (FileLineModel)data ;
		}
		else {
			// LineModel が FileLineModel でない場合、オブジェクトを作成します。
			fileData = new FileLineModel( data );
		}

		if( debug ) { println( "Before:" + data.dataLine() ); }		// 5.1.2.0 (2010/01/01) display の条件変更

		final File inFile = fileData.getFile() ;
		if( ! inFile.isFile() ) {
			if( display ) { println( data.dataLine() ); }		// 5.1.2.0 (2010/01/01) display の条件変更
			return data;
		}

		// ファイル名を作成します。
		// ファイル名は、引数ファイル名 から、inPath を引き、outPath を加えます。
		final File outFile = new File( outPath, inFile.getAbsolutePath().substring( inPathLen ) );
		fileData.setFile( outFile );

		// 入出力が異なる場合
		if( !isEquals ) {
			tempFile = outFile;
			final File parent = outFile.getParentFile();
			if( parent != null && ! parent.exists() && !parent.mkdirs() ) {
				final String errMsg = "所定のフォルダが作成できませんでした。[" + parent + "]" + CR
							+ " inCount=[" + inCount + "]件" + CR
							+ " data=[" + data.dataLine() + "]" + CR ;		// 5.7.2.2 (2014/01/24) エラー時にデータも出力します。
				throw new OgRuntimeException( errMsg );
			}
		}

		if( binary ) {
			// 5.1.6.0 (2010/05/01) changeCrLfcopy 対応
			if( changeCrLf ) { FileUtil.changeCrLfcopy( inFile,tempFile ); }
			else             { FileUtil.copy( inFile,tempFile,keepTimeStamp ); }
		}
		else {
			final BufferedReader reader = FileUtil.getBufferedReader( inFile ,inEncode  );
			final PrintWriter    writer = FileUtil.getPrintWriter( tempFile  ,outEncode );

			try {
				String line1;
				if( useOmitCmnt ) {			// 5.7.4.0 (2014/03/07) コメント部分を削除してコピー
					// 6.4.0.2 (2015/12/11) CommentLineParser 改造
//					final CommentLineParser clp = new CommentLineParser();
					final CommentLineParser clp = new CommentLineParser( FileInfo.getSUFIX( inFile ) );
					while((line1 = reader.readLine()) != null) {
						line1 = clp.line( line1 );
						if( line1 != null ) {
							writer.println( line1 );
						}
					}
				}
				else {
					// 従来のコピー。ループ中で、if するのが嫌だったので、分離しました。
					while((line1 = reader.readLine()) != null) {
						writer.println( line1 );
					}
				}
			}
			// 6.3.1.0 (2015/06/28) nioを使用すると UTF-8とShuft-JISで、エラーになる。
			catch ( CharacterCodingException ex ) {
				final String errMsg = "文字のエンコード・エラーが発生しました。" + CR
									+	"  ファイルのエンコードが指定のエンコードと異なります。" + CR
									+	" [" + inFile.getPath() + "] , Encode=[" + inEncode + "]" ;
				throw new OgRuntimeException( errMsg,ex );
			}
			catch( IOException ex ) {
				final String errMsg = "ファイルコピー中に例外が発生しました。[" + data.getRowNo() + "]件目" + CR
							+ " inFile=[" + inFile + "] , tempFile=[" + tempFile + "]" + CR
							+ " data=[" + data.dataLine() + "]" + CR ;		// 5.7.2.2 (2014/01/24) エラー時にデータも出力します。
				throw new OgRuntimeException( errMsg,ex );
			}
			finally {
				Closer.ioClose( reader ) ;
				Closer.ioClose( writer ) ;
			}
		}

		if( isEquals ) {
			if( !outFile.delete() ) {
				final String errMsg = "所定のファイルを削除できませんでした。[" + outFile + "]" + CR
							+ " inCount=[" + inCount + "]件" + CR
							+ " data=[" + data.dataLine() + "]" + CR ;		// 5.7.2.2 (2014/01/24) エラー時にデータも出力します。
				throw new OgRuntimeException( errMsg );
			}

			if( !tempFile.renameTo( outFile ) ) {
				final String errMsg = "所定のファイルをリネームできませんでした。[" + tempFile + "]" + CR
							+ " inCount=[" + inCount + "]件" + CR
							+ " data=[" + data.dataLine() + "]" + CR ;		// 5.7.2.2 (2014/01/24) エラー時にデータも出力します。
				throw new OgRuntimeException( errMsg );
			}
		}

	 	// 5.1.5.0 (2010/04/01) keepTimeStamp 属性の追加
		// 6.0.0.1 (2014/04/25) These nested if statements could be combined
		if( keepTimeStamp && !outFile.setLastModified( inFile.lastModified() ) ) {
			final String errMsg = "lastModified 時間の設定が、できませんでした。[" + outFile + "]" + CR
						+ " inCount=[" + inCount + "]件" + CR
						+ " data=[" + data.dataLine() + "]" + CR ;		// 5.7.2.2 (2014/01/24) エラー時にデータも出力します。
			throw new OgRuntimeException( errMsg );
		}

		if( display ) { println( data.dataLine() ); }		// 5.1.2.0 (2010/01/01) display の条件変更
		return data ;
	}

	/**
	 * プロセスの処理結果のレポート表現を返します。
	 * 処理プログラム名、入力件数、出力件数などの情報です。
	 * この文字列をそのまま、標準出力に出すことで、結果レポートと出来るような
	 * 形式で出してください。
	 *
	 * @return   処理結果のレポート
	 */
	public String report() {
		final String report = "[" + getClass().getName() + "]" + CR
				+ TAB + "Copy Count : " + inCount   + CR
				+ TAB + "inPath     : " + inPath    + CR
				+ TAB + "inEncode   : " + inEncode  + CR
				+ TAB + "outPath    : " + outPath   + CR
				+ TAB + "outEncode  : " + outEncode + CR
				+ TAB + "binary     : " + binary ;

		return report ;
	}

	/**
	 * このクラスの使用方法を返します。
	 *
	 * @return	このクラスの使用方法
	 * @og.rtnNotNull
	 */
	public String usage() {
		final StringBuilder buf = new StringBuilder( BUFFER_LARGE )
			.append( "Process_FileCopy は、上流から受け取った FileLineModelを処理する、"			).append( CR )
			.append( "ChainProcess インターフェースの実装クラスです。"								).append( CR )
			.append( CR )
			.append( "上流から受け取った FileLineModel の ファイルから、inPath の共通パス" 			).append( CR )
			.append( "以下のファイルを、outPath の共通パス以下にコピーします。" 					).append( CR )
			.append( "コピーの種類は、バイナリか、テキストで、テキストの場合は、エンコード" 		).append( CR )
			.append( "変換も行うことが可能です。" 													).append( CR )
			.append( "inPath と outPath が同じ、または、outPath が未設定の場合は、入力と出力が" 	).append( CR )
			.append( "同じですので、自分自身のエンコード変換処理を行うことになります。" 			).append( CR )
			.append( CR )
			.append( "コピーされるファイルのファイル名は、入力ファイル名と同一です。保存される" 	).append( CR )
			.append( "フォルダが異なります。(同一にすることも可能です。)" 							).append( CR )
			.append( CR )
			.append( "上流プロセスでは、Name 属性として、『File』を持ち、値は、Fileオブジェクト" 	).append( CR )
			.append( "である、Process_FileSearch を使用するのが、便利です。それ以外のクラスを" 		).append( CR )
			.append( "使用する場合でも、Name属性と、File オブジェクトを持つ LineModel を受け渡し" 	).append( CR )
			.append( "できれば、使用可能です。" 													).append( CR )
			.append( CR )
			.append( "引数文字列中に空白を含む場合は、ダブルコーテーション(\"\") で括って下さい。" 	).append( CR )
			.append( "引数文字列の 『=』の前後には、空白は挟めません。必ず、-key=value の様に" 		).append( CR )
			.append( "繋げてください。" 															).append( CR )
			.append( CR ).append( CR )
			.append( getArgument().usage() ).append( CR );

		return buf.toString();
	}

	/**
	 * このクラスは、main メソッドから実行できません。
	 *
	 * @param	args	コマンド引数配列
	 */
	public static void main( final String[] args ) {
		LogWriter.log( new Process_FileCopy().usage() );
	}
}
