/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import org.opengion.fukurou.system.OgRuntimeException ;		// 6.4.2.0 (2016/01/29)
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.SystemManager;
import org.opengion.fukurou.util.Cleanable;

import java.util.Map;
// import java.util.HashMap;
import java.util.concurrent.ConcurrentHashMap;				// 6.4.3.1 (2016/02/12) refactoring
import java.util.Locale ;

/**
 * DBCellRenderer/DBCellEditor オブジェクトを取得する為に使用する，ファクトリクラスです。
 *
 *  DBCell オブジェクト の識別ID を元に、DBCellFactory.newInstance( String id )
 * メソッドで，DBCell オブジェクトを取得します。
 *
 * @og.group データ表示
 * @og.group データ編集
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class DBCellFactory {
	private static final String DEFAULT_RENDERER	= "LABEL";
	private static final String DEFAULT_EDITOR		= "TEXT";
//	private static final Map<String,CellRenderer>	REN_MAP	= new HashMap<>();		// 6.4.1.1 (2016/01/16) rMap → REN_MAP refactoring
//	private static final Map<String,CellEditor>		EDT_MAP	= new HashMap<>();		// 6.4.1.1 (2016/01/16) eMap → EDT_MAP refactoring
	/** 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。  */
	private static final Map<String,CellRenderer>	REN_MAP	= new ConcurrentHashMap<>();	// 6.4.1.1 (2016/01/16) rMap → REN_MAP refactoring
	/** 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。  */
	private static final Map<String,CellEditor>		EDT_MAP	= new ConcurrentHashMap<>();	// 6.4.1.1 (2016/01/16) eMap → EDT_MAP refactoring

	// 4.0.0 (2005/01/31) Cleanable インターフェースによる初期化処理
	static {
		final Cleanable clr = new Cleanable() {
			/**
			 * 初期化(クリア)します。
			 * 主に、キャッシュクリアで利用します。
			 */
			public void clear() {
				DBCellFactory.clear();
			}
		};

		SystemManager.addCleanable( clr );
	}

	/**
	 *  デフォルトコンストラクターをprivateにして、
	 *  オブジェクトの生成をさせないようにする。
	 *
	 */
	private DBCellFactory() {
	}

	/**
	 * 識別id に応じた DBCell オブジェクトを取得します。
	 *
	 * @og.rev 2.1.2.1 (2002/11/27) id が指定されていない場合の デフォルトを指定するように変更。
	 * @og.rev 3.1.1.1 (2003/04/03) DBCell のファクトリクラスに DBColumn オブジェクトを渡す。
	 * @og.rev 3.1.2.1 (2003/04/10) synchronized の方法を修正。
	 * @og.rev 3.5.6.0 (2004/06/18) 各種プラグイン関連付け設定を、システムパラメータ に記述します。
	 * @og.rev 4.0.0.0 (2005/01/31) キーの指定を、Renderer. から、Renderer_ に変更します。
	 * @og.rev 6.0.4.0 (2014/11/28) NullPointerException が発生するので、事前にチェックします。
	 * @og.rev 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。
	 *
	 * @param   id DBCell インターフェースを実装したサブクラスの識別id
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return  DBCellオブジェクト
	 * @og.rtnNotNull
	 */
	public static CellRenderer newRenderer( final String id,final DBColumn clm ) {
		final String type = ( id == null ) ? DEFAULT_RENDERER : id.toUpperCase(Locale.JAPAN) ;
		final String cls = HybsSystem.sys( "Renderer_" + type );		// 4.0.0 (2005/01/31)
		// NullPointerException が発生するので、事前にチェックします。
		if( cls == null ) {
			final String errMsg = "Renderer クラスが見つかりません。class=" + "Renderer_" + type
						+ " name=" + clm.getName();
			throw new OgRuntimeException( errMsg );
		}

		// ConcurrentMap#compute(K,BiFunction) で、あればmapの値を、無ければ新規作成します。
		final CellRenderer cell = REN_MAP.compute( type , (k, v) -> (v == null) ? (CellRenderer)HybsSystem.newInstance( cls ) : v );

//		CellRenderer cell;
//		synchronized( REN_MAP ) {
//			cell = REN_MAP.get( type );
//			if( cell == null ) {
//				cell = (CellRenderer)HybsSystem.newInstance( cls );	// 3.5.5.3 (2004/04/09)
//				REN_MAP.put( type,cell );
//			}
//		}
		return cell.newInstance( clm );
	}

	/**
	 * 識別id に応じた DBCell オブジェクトを取得します。
	 *
	 * @og.rev 2.1.2.1 (2002/11/27) id が指定されていない場合の デフォルトを指定するように変更。
	 * @og.rev 3.1.1.1 (2003/04/03) DBCell のファクトリクラスに DBColumn オブジェクトを渡す。
	 * @og.rev 3.1.2.1 (2003/04/10) synchronized の方法を修正。
	 * @og.rev 3.5.6.0 (2004/06/18) 各種プラグイン関連付け設定を、システムパラメータ に記述します。
	 * @og.rev 4.0.0.0 (2005/01/31) キーの指定を、Editor. から、Editor_ に変更します。
	 * @og.rev 6.0.4.0 (2014/11/28) NullPointerException が発生するので、事前にチェックします。
	 * @og.rev 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。
	 *
	 * @param   id DBCell インターフェースを実装したサブクラスの識別id
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return  DBCellオブジェクト
	 * @og.rtnNotNull
	 */
	public static CellEditor newEditor( final String id,final DBColumn clm ) {
		final String type = ( id == null ) ? DEFAULT_EDITOR : id.toUpperCase(Locale.JAPAN) ;
		final String cls = HybsSystem.sys( "Editor_" + type );		// 4.0.0 (2005/01/31)

		// NullPointerException が発生するので、事前にチェックします。
		if( cls == null ) {
			final String errMsg = "Editor クラスが見つかりません。class=" + "Editor_" + type
						+ " name=" + clm.getName();
			throw new OgRuntimeException( errMsg );
		}

		// ConcurrentMap#compute(K,BiFunction) で、あればmapの値を、無ければ新規作成します。
		final CellEditor cell = EDT_MAP.compute( type , (k, v) -> (v == null) ? (CellEditor)HybsSystem.newInstance( cls ) : v );

//		CellEditor cell;
//		synchronized( EDT_MAP ) {
//			cell = EDT_MAP.get( type );
//			if( cell == null ) {
//				cell = (CellEditor)HybsSystem.newInstance( cls );	// 3.5.5.3 (2004/04/09)
//				EDT_MAP.put( type,cell );
//			}
//		}
		return cell.newInstance( clm );
	}

	/**
	 * DBCell オブジェクトをプールからすべて削除します。
	 * システム全体を初期化するときや、動作が不安定になったときに行います。
	 * プールの方法自体が,一種のキャッシュ的な使いかたしかしていない為,
	 * 実行中でも、いつでもプールを初期化できます。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) キャッシュクリアメソッドを新規追加。
	 * @og.rev 3.1.2.1 (2003/04/10) synchronized の方法を修正。
	 * @og.rev 6.4.3.1 (2016/02/12) PMD refactoring. HashMap → ConcurrentHashMap に置き換え。
	 */
	public static void clear() {
//		synchronized( REN_MAP ) { REN_MAP.clear(); }
//		synchronized( EDT_MAP ) { EDT_MAP.clear(); }
		REN_MAP.clear();
		EDT_MAP.clear();
	}
}
