/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.servlet;

import org.opengion.fukurou.system.OgRuntimeException ;		// 6.4.2.0 (2016/01/29)
import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.lang.reflect.Field;
import java.net.URL;
import java.util.Enumeration;
import java.util.Map;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.opengion.fukurou.db.ConnectionFactory;
import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.HybsEntry;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.system.BuildNumber;			// 6.4.2.0 (2016/01/29) hayabusa.common.BuildNumber → fukurou.system.BuildNumber に移動
import org.opengion.fukurou.system.Closer ;							// 5.5.2.6 (2012/05/25)
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.SystemManager;
import org.opengion.hayabusa.common.SystemParameter;
import org.opengion.hayabusa.common.UserSummary;
// import org.opengion.hayabusa.filter.AccessStopFilter;				// 6.1.1.0 (2015/01/17) refactoring
import static org.opengion.fukurou.system.HybsConst.CR ;				// 6.1.0.0 (2014/12/26)
import static org.opengion.fukurou.system.HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring

import org.opengion.hayabusa.resource.ResourceFactory;				// 5.9.1.0 (2015/10/02)
import org.opengion.hayabusa.resource.UserInfo;						// 5.9.1.0 (2015/10/02)

// import javax.servlet.annotation.WebServlet;							// 6.3.4.0 (2015/08/01)
import javax.servlet.ServletRegistration;							// 6.3.4.0 (2015/08/01)
import javax.servlet.FilterRegistration;							// 6.3.4.0 (2015/08/01)

/**
 * サーバー管理情報を取得するAdminサーブレットです。
 *
 * 引数(URL)に応じて、サーバーの状態結果を返します。
 * 一般には、http://サーバー:ポート/システムID/jsp/admin?COMMAND=コマンド の
 * 形式のURL でアクセスします。
 *
 *  ・COMMAND=infomation
 *     【サーバー情報】
 *        ＯＳ情報      = Windows 7 Service Pack 1 x86 (32) 
 *        サーバー名    = 10374232-0004 ( 172.27.26.164 ) 
 *        サーブレット  = Apache Tomcat/8.0.22 
 *        TOMCAT_HOME   = C:\opengionV6\apps\tomcat8.0.22 
 *        JDKバージョン = Java HotSpot(TM) Server VM 1.8.0_45-b15 
 *        JAVA_HOME     = C:\opengionV6\apps\jdk180u45\jre 
 *     【実行環境】
 *        REAL_PATH     = C:/opengionV6/uap/webapps/gf/
 *        バージョンNo  = openGion 6.2.4.0 Release6 Builds (2015142)
 *        作成日時      = 2015/05/22 15:31:53
 *     【ログイン情報】
 *        ログイン人数  = 2 名 ( 明細情報 )
 *     【メモリ情報】
 *        空きメモリ    = 129 [MByte] 
 *        合計メモリ    = 209 [MByte] 
 *        使用率        = 38 [％] 
 *     【ディスク情報】
 *
 *  ・COMMAND=close
 *       リソース情報のキャッシュを全てクリアします。
 *
 *  ・COMMAND=loginUser
 *       現在のログインユーザーの明細情報を表示します。
 *       SORT=[JNAME,ID,ROLES,IPADDRESS,LOGINTIME] ソートキー
 *       DIREC=[true,false] true:昇順/false:降順
 *
 *  ・COMMAND=plugin
 *       現在のプラグインのバージョン情報を表示します。
 *
 *  ・COMMAND=systemResource
 *       現在のシステムリソースの設定情報を表示します。
 *
 *  ・COMMAND=AccessStop&stop=[true/false]
 *       アクセスストップフィルターの制御(停止、許可)を行います。
 *
 *  ・common/gamen/01_ADMIN/parameter.jsp
 *       現在のシステムリソースの設定情報を表示します。
 *
 *  ・common/gamen/01_ADMIN/systemInfo.jsp
 *       システム定数のシステム起動情報を表示します。
 *
 * @og.rev 3.5.3.0 (2003/10/27) Admin JSP を Servlet化して、エンジンと共に供給します。
 * @og.rev 4.0.0.0 (2005/08/31) プラグインのバージョン情報の表示機能を追加
 * @og.group その他機能
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
// @WebServlet(name="org.opengion.hayabusa.servlet.HybsAdmin",urlPatterns={"/jsp/admin"})
public final class HybsAdmin extends HttpServlet {
	private static final long serialVersionUID = 624120150522L ;	// 6.2.4.1 (2015/05/22)

	private static final String JSP = HybsSystem.sys( "JSP" );						// jspフォルダの正規パス

	private static final long MB = 1024*1024 ;						// 6.2.4.1 (2015/05/22)
	private static final long GB = 1024*1024*1024 ;					// 6.2.4.1 (2015/05/22)

	private static final String HEADER =
		"<?xml version=\"1.0\" encoding=\"UTF-8\" ?>"												+ CR +
		"<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\""							+ CR +
		"    \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd\">"							+ CR +
		"<html xmlns=\"http://www.w3.org/1999/xhtml\" >"											+ CR +
		"<head>"																					+ CR +
		"    <meta http-equiv=\"Content-Type\" content=\"text/html; charset=UTF-8\" />"				+ CR +
		"    <meta http-equiv=\"Content-Style-Type\" content=\"text/css\" />"						+ CR +
		"    <link rel=\"stylesheet\" href=\"" + JSP + "/common/default.css\" type=\"text/css\" />"	+ CR +
		"    <link rel=\"stylesheet\" href=\"" + JSP + "/custom/custom.css\" type=\"text/css\" />"	+ CR +
		"    <title>Hybs Admin</title>"																+ CR +
		"</head>"																					+ CR;

	// 3.5.3.1 (2003/10/31) User情報のテーブルの設定を、システムリソース より行う。
	private static final String TABLE_HEADER = getTableHeaderTag() ;

	private static final String OS_INFO      = HybsSystem.sys( "OS_INFO"      );		// Windows 7 Service Pack 1
	private static final String SERVER_INFO  = HybsSystem.sys( "SERVER_INFO"  );		// 10374232-0004 ( 200.1.50.239 )
	private static final String SERVLET_INFO = HybsSystem.sys( "SERVLET_INFO" );		// Apache Tomcat/7.0.39
	private static final String REAL_PATH    = HybsSystem.sys( "REAL_PATH"    );		// C:/opengion/uap/webapps/gf/
	private static final String TOMCAT_HOME  = HybsSystem.sys( "TOMCAT_HOME"  );		// C:/opengion/apps/tomcat5.5.17
	private static final String JDK_INFO     = HybsSystem.sys( "JDK_INFO"     );		// Java HotSpot(TM) Server VM 23.25-b01
	private static final String JAVA_HOME    = HybsSystem.sys( "JAVA_HOME"    );		// C:/opengion/apps/jdk170u25/jre
	private static final String ENGINE_INFO  = HybsSystem.sys( "ENGINE_INFO"  );		// 5.6.6.0 Release5 Builds (2013182)

	// 5.6.6.0 (2013/07/05) getLoginUser( String,boolean ) で、指定するキーを配列で持っておきます。
	// キーは、SystemManager.getRunningUserSummary 処理内で大文字化されるため、この配列は、表示用と兼用します。
	private static final String[] USER_KEYS = { "ID","Jname","Roles","IPAddress","LoginTime","LastAccess","LastGamenNm" };

	// 6.3.6.1 (2015/08/28) <table> 枠の指定と、</table> の static final String 化
	private static final String TABLE_WAKU  = "<table style=\"margin: 0px 0px 0px 20px;\" frame=\"box\" border=\"1px\" cellspacing=\"0px\" cellpadding=\"0px\" " ;
	private static final String TABLE_END   = "</table>" + CR ;

	/**
	 * GET メソッドが呼ばれたときに実行します。
	 *
	 * @og.rev 3.5.3.1 (2003/10/31) 機能ごとにメソッドを呼び出すように修正します。
	 * @og.rev 3.5.4.1 (2003/12/01) getAdminLink() メソッドを追加
	 * @og.rev 6.3.8.0 (2015/09/11) AccessStop を外部から制御できるように、コマンドと状態を受けるようにする。
	 * @og.rev 5.9.1.0 (2015/10/02) UserInfoセット機能追加
	 * @og.rev 6.3.8.3 (2015/10/03) Servlet/Filter 情報を作成。
	 * @og.rev 6.3.9.1 (2015/11/27) ３項演算子を || or &amp;&amp; で簡素化できる(PMD)。
	 *
	 * @param	req	HttpServletRequestオブジェクト
	 * @param	res	HttpServletResponseオブジェクト
	 *
	 * @throws ServletException サーブレット関係のエラーが発生した場合、throw されます。
	 * @throws IOException 入出力エラーが発生したとき
	 */
	@Override
	public void doGet( final HttpServletRequest req, final HttpServletResponse res)
								throws ServletException, IOException {

		res.setContentType( "text/html; charset=UTF-8" );
		final PrintWriter out = res.getWriter();

		final String command = req.getParameter( "COMMAND" );

		out.println( HEADER );
		out.println("<body>");

		out.print("<h2>");
		out.print( req.getServerName() );
		out.print( ":" );
		out.print( req.getServerPort() );
		out.print( req.getContextPath() );
		out.println("</h2>");

		if( "infomation".equalsIgnoreCase( command ) ) {
			out.print( getInfomation() );
		}
		else if( "close".equalsIgnoreCase( command ) ) {
			out.print( getCloseMessage( req.getSession() ) );
		}
		else if( "loginUser".equalsIgnoreCase( command ) ) {
			final String sort  = req.getParameter( "sort" );
			final String direc = req.getParameter( "direc" );
			// 6.3.9.1 (2015/11/27) ３項演算子を || or && で簡素化できる(PMD)。
//			final boolean dir = (direc == null) ? true :  Boolean.parseBoolean( direc );	// 6.1.0.0 (2014/12/26) refactoring
			final boolean dir = direc == null || Boolean.parseBoolean( direc );				// 6.3.9.1 (2015/11/27)

			out.print( getLoginUser(sort,dir) );
		}
		else if( "plugin".equalsIgnoreCase( command ) ) {
			out.print( getPlugInInfo() );
		}
		else if( "taglib".equalsIgnoreCase( command ) ) {
			out.print( getTaglibInfo() );
		}
		else if( "systemResource".equalsIgnoreCase( command ) ) {
			out.print( getSystemResource() );
		}
		// 6.3.8.0 (2015/09/11) AccessStop を外部から制御できるように、コマンドと状態を受けるようにする。
		else if( "AccessStop".equalsIgnoreCase( command ) ) {
//			out.print( getAccessStop() );
			out.print( getAccessStop( req.getParameter( "stop" ) ) );	// 6.3.8.0 (2015/09/11)
		}
		// 5.9.1.0 (2015/10/02) UserInfoセット機能追加
		else if( "UserInfo".equalsIgnoreCase( command ) ){ // 5.9.1.0 (2015/10/02)
			out.print( setUserinfo( req ) );
		}
		// 6.3.8.3 (2015/10/03) Servlet/Filter 情報を作成。
		else if( "registrationInfo".equalsIgnoreCase( command ) ) {
			out.print( getRegistrationMessage( req.getSession() ) );
		}
		else {
			out.print( getAdminLink() );		// 3.5.4.1 (2003/12/01) 追加
		}

		out.println("</body></html>");
	}

	/**
	 * infomation 情報を作成します。
	 *
	 * @og.rev 3.5.3.1 (2003/10/31) 機能ごとにメソッドを呼び出すように修正します。
	 * @og.rev 5.6.7.3 (2013/08/23) TOMCAT_WORKは、TOMCAT_HOME に置き換えます。
	 * @og.rev 6.2.4.1 (2015/05/22) ディスク容量を求めます。
	 * @og.rev 6.3.6.1 (2015/08/28) <table> 枠の指定と、</table> の static final String 化
	 * @og.rev 6.3.9.0 (2015/11/06) 整数値演算なので、Math.round 不要
	 *
	 * @return	infomation情報
	 * @og.rtnNotNull
	 */
	private String getInfomation() {
		// 4.0.0 (2005/01/31) ログイン数の取得方法の変更。
		final int loginCount = SystemManager.getRunningCount() ;

//		final long freeMemory     = Math.round( Runtime.getRuntime().freeMemory()/MB ) ;	// 6.2.4.1 (2015/05/22) 単位を MB に変更
//		final long totalMemory    = Math.round( Runtime.getRuntime().totalMemory()/MB );	// 6.2.4.1 (2015/05/22) 単位を MB に変更
//		final long useMemoryRatio = Math.round( (totalMemory-freeMemory)*100/totalMemory ) ;
		final long freeMemory     = Runtime.getRuntime().freeMemory()/MB ;					// 6.3.9.0 (2015/11/06) 整数値演算なので、Math.round 不要
		final long totalMemory    = Runtime.getRuntime().totalMemory()/MB;					// 6.3.9.0 (2015/11/06) 整数値演算なので、Math.round 不要
		final long useMemoryRatio = (totalMemory-freeMemory)*100/totalMemory ;				// 6.3.9.0 (2015/11/06) 整数値演算なので、Math.round 不要
		final String bgCls = useMemoryRatio <= 50 ? "OK" : ( useMemoryRatio <= 80 ? "CAUT" : "NG" ) ;

		final StringBuilder rtn = new StringBuilder( BUFFER_MIDDLE );

		rtn.append( "<table border = \"0px\" >" ).append( CR );
		tableTr1( rtn,"サーバー情報" );
		tableTr( rtn,"ＯＳ情報"		, OS_INFO		);
		tableTr( rtn,"サーバー名"	, SERVER_INFO	);
		tableTr( rtn,"サーブレット"	, SERVLET_INFO	);
		tableTr( rtn,"TOMCAT_HOME"	, TOMCAT_HOME	);		// 5.6.7.3 (2013/08/23)
		tableTr( rtn,"JDKバージョン", JDK_INFO		);
		tableTr( rtn,"JAVA_HOME"	, JAVA_HOME		);

		tableTr1( rtn,"実行環境" );
		tableTr( rtn,"REAL_PATH"	, REAL_PATH		);
		tableTr( rtn,"バージョンNo"	, ENGINE_INFO	);
		tableTr( rtn,"作成日時"		, BuildNumber.TIMESTAMP	);

		tableTr1( rtn,"ログイン情報" );
		tableTr( rtn,"ログイン人数"	, String.valueOf( loginCount )," 名 ","( <a href=\"admin?COMMAND=loginUser\">明細情報</a> )" );

		tableTr1( rtn,"メモリ情報" );
		tableTr( rtn,"空きメモリ"	, String.valueOf( freeMemory )		, " [MByte]" );	// 6.2.4.1 (2015/05/22) 単位を MB に変更
		tableTr( rtn,"合計メモリ"	, String.valueOf( totalMemory )		, " [MByte]" );	// 6.2.4.1 (2015/05/22) 単位を MB に変更
		tableTr( rtn,"使用率"		, String.valueOf( useMemoryRatio )	, " [％]"    );
//		rtn.append( "</table>" ).append( CR )
		rtn.append( TABLE_END )
			// 6.2.4.1 (2015/05/22) default.css で、memory、disc、discTD を定義
//			.append( "<table style=\"margin: 0px 0px 0px 20px;\" width=\"300px\" frame=\"box\" border=\"1px\" cellspacing=\"0px\" cellpadding=\"0px\" >" ).append( CR )
			.append( TABLE_WAKU ).append( "width=\"300px\" >" ).append( CR )
			.append( "\t<tr><td class=\"memory " ).append( bgCls ).append( "\" style=\"width:" )
			.append( useMemoryRatio ).append( "%;\" >" ).append( useMemoryRatio ).append( "%</td>" )
			.append( "<td class=\"BIT\">" ).append( 100-useMemoryRatio ).append( "%</td></tr>" ).append( CR )
//			.append( "</table>" ).append( CR ) ;
			.append( TABLE_END )

		// 6.2.4.1 (2015/05/22) ディスク容量を求めます。
			.append( "<table border=\"0px\" >" ).append( CR );
		tableTr1( rtn,"ディスク情報" );
//		rtn.append( "</table>" ).append( CR );
		rtn.append( TABLE_END )

//		rtn.append( "<table style=\"margin: 0px 0px 0px 20px;\" width=\"500px\" frame=\"box\" border=\"1px\" cellspacing=\"0px\" cellpadding=\"0px\" >" ).append( CR )
			.append( TABLE_WAKU ).append( "width=\"500px\" >" ).append( CR )
			.append( "<th>名前</th><th>合計サイズ</th><th>空き領域</th><th>使用率</th></tr>" ).append( CR );

		final File[] driv = File.listRoots();
		for( final File file : driv ) {
			// 6.3.9.0 (2015/11/06) 整数値演算なので、Math.round 不要
//			final long total	= Math.round( file.getTotalSpace()/GB  );		// パーティションのサイズ
//			final long usable	= Math.round( file.getUsableSpace()/GB );		// 仮想マシンが利用できるバイト数
			final long total	= file.getTotalSpace()/GB ;						// パーティションのサイズ
			final long usable	= file.getUsableSpace()/GB ;					// 仮想マシンが利用できるバイト数
			tableTr2( rtn, file.getPath() , total , usable );					// ディスク容量専用3
		}
//		rtn.append( "</table>" ).append( CR );
		rtn.append( TABLE_END )

		// information
			.append( "<hr />" ).append( CR )
			.append( "<pre>" ).append( CR )
			.append( ConnectionFactory.information() ).append( CR )
			.append( "</pre>" ).append( CR );

		return rtn.toString();
	}

	/**
	 * infomation 情報を作成します。
	 *
	 * @og.rev 5.6.6.0 (2013/07/05) &lt;/td&gt;&lt;/tr&gt;漏れ追加
	 * @og.rev 5.6.7.3 (2013/08/23) 前後に、【】を付けます。
	 *
	 * @param	buf	情報登録用のStringBuilder
	 * @param	key	キー
	 *
	 * @return	infomation情報(入力bufと同じオブジェクト)
	 * @og.rtnNotNull
	 */
	private StringBuilder tableTr1( final StringBuilder buf, final String key ) {
		buf.append( "\t<tr><td colspan=\"4\"><b>【" ).append( key ).append( "】</b></td></tr>" ).append( CR );	// 【】追加
		return buf ;
	}

	/**
	 * infomation 情報を作成します。
	 *
	 * @og.rev 5.6.6.0 (2013/07/05) 最初の td に、全角スペース２個 追加
	 *
	 * @param	buf	情報登録用のStringBuilder
	 * @param	key	キー
	 * @param	val	値の可変長引数
	 *
	 * @return	infomation情報(入力bufと同じオブジェクト)
	 * @og.rtnNotNull
	 */
	private StringBuilder tableTr( final StringBuilder buf, final String key, final String... val ) {
		buf.append( "\t<tr><td width=\"20px\">　</td><td>" ).append( key ).append( "</td><td> = </td><td>" );			// 段を作成する為に、width指定 追加
		for( int i=0; i<val.length; i++ ) {
			buf.append( val[i] );
		}
		buf.append( "</td></tr>" ).append( CR );
		return buf ;
	}

	/**
	 * infomation 情報を作成します。
	 *
	 * @og.rev 6.2.4.1 (2015/05/22) ディスク容量専用
	 * @og.rev 6.2.4.2 (2015/05/29) total が 0 の場合の対応
	 *
	 * @param	buf	情報登録用のStringBuilder
	 * @param	name	ドライブ名
	 * @param	total	パーティションのサイズ(GB)
	 * @param	usable	仮想マシンが利用できるサイズ(GB)
	 *
	 * @return	infomation情報(入力bufと同じオブジェクト)
	 * @og.rtnNotNull
	 */
	private StringBuilder tableTr2( final StringBuilder buf, final String name, final long total, final long usable ) {
//		final long useRatio = total == 0L ? 0L : Math.round( (total-usable)*100/total ) ;
		final long useRatio = total == 0L ? 0L : (total-usable)*100/total ;			// 6.3.9.0 (2015/11/06) findBugs

		final String bgCls = useRatio <= 50 ? "OK" : ( useRatio <= 80 ? "CAUT" : ( useRatio <= 95 ? "WARN" : "NG" ) ) ;

		buf.append( "\t<tr><td class=\"BIT\">" ).append( name ).append( "</td>" )
			.append( "<td class=\"S9\" >" ).append( total  ).append( " GB</td>" )
			.append( "<td class=\"S9\" >" ).append( usable ).append( " GB</td>" ).append( CR )
			.append( "\t\t<td class=\"discTD\" ><span class=\"disc " ).append( bgCls )
			.append( "\" style=\"width:" ).append( useRatio ).append( "%;\" >" )
			.append( useRatio ).append( "%</span>　</td></tr>" ).append( CR );
		return buf ;
	}

	/**
	 * close 情報を作成します。
	 *
	 * @og.rev 3.5.3.1 (2003/10/31) 機能ごとにメソッドを呼び出すように修正します。
	 * @og.rev 3.6.0.0 (2004/09/17) CalendarFactory.clear() を追加します。
	 * @og.rev 4.0.0.0 (2005/01/31) Cleanable インターフェースによる初期化処理
	 * @og.rev 4.1.0.2 (2008/01/29) UserInfoをsessionから消去する(超暫定対応)
	 * @og.rev 5.6.6.0 (2013/07/05) UserSummary の削除処理は、SystemManager から行う。
	 * @og.rev 6.3.7.0 (2015/09/04) ファイナライズを呼び出します。
	 *
	 * @param	session	HttpSessionオブジェクト
	 *
	 * @return	close情報
	 * @og.rtnNotNull
	 */
	private String getCloseMessage( final HttpSession session ) {

		// 4.0.0 (2005/01/31) Cleanable インターフェースによる初期化処理
		SystemManager.allClear( false ) ;

		final ServletContext context = session.getServletContext();
		final Map<String,String> param = SystemParameter.makeSystemParameter( context );
		HybsSystem.setInitialData( param );			// 4.0.0 (2005/01/31)

		// 5.6.6.0 (2013/07/05) UserSummary の削除処理は、SystemManager から行う。
		SystemManager.removeSession( session ) ;

		// 6.3.7.0 (2015/09/04) ファイナライズを呼び出します。
		System.runFinalization();
	//	System.gc();					// 取りあえず、GCは呼びません。

		final String rtn = "<pre>"
					+ "キャッシュ情報をクリアしました。"
					+ CR
					+ "ユーザー情報の初期化に関しては、ブラウザを閉じて、再ログインが必要です。"
					+ CR
					+ ConnectionFactory.information()
					+ "</pre>" ;
		return rtn ;
	}

	/**
	 * loginUser 情報を作成します。
	 *
	 * @og.rev 3.5.3.1 (2003/10/31) 機能ごとにメソッドを呼び出すように修正します。
	 * @og.rev 3.8.5.3 (2006/08/07) ユーザー情報をソートするためのキー情報を追加
	 * @og.rev 3.8.7.0 (2006/12/15) USER.LASTACCESS情報を追加します。
	 * @og.rev 4.0.0.0 (2005/01/31) DBColumn の 属性(CLS_NM)から、DBTYPEに変更
	 * @og.rev 4.4.0.1 (2009/08/08) 最終ログイン画面名称を追加
	 * @og.rev 5.6.6.0 (2013/07/05) table作成処理を、変更します。
	 * @og.rev 6.3.6.1 (2015/08/28) <table> 枠の指定と、</table> の static final String 化
	 * @og.rev 5.9.1.2 (2015/10/23) 自己終了警告対応
	 *
	 * @param	sort 	ソートするキー項目を指定
	 * @param	direc	ソートする方向 [true:昇順/false:降順]
	 *
	 * @return	loginUser情報
	 * @og.rtnNotNull
	 */
	private String getLoginUser( final String sort,final boolean direc ) {
		// 4.0.0 (2005/01/31) ログイン数の取得方法の変更。
		final int loginCount = SystemManager.getRunningCount() ;

		// 4.0.0 (2005/01/31)
		final UserSummary[] userInfos = SystemManager.getRunningUserSummary( sort,direc );

		final StringBuilder rtn = new StringBuilder( BUFFER_MIDDLE )
			.append( "現在 " ).append( loginCount ).append( " 名の方がログイン中です。" )
			.append( CR )
			.append( "<table " ).append( TABLE_HEADER ).append( " >" ).append( CR )
			.append( " <thead><tr><th>No</th>" );

		// 5.6.6.0 (2013/07/05) table作成処理を、変更します。
		for( final String sortKey : USER_KEYS ) {
			rtn.append( "  <th><a href=\"?COMMAND=loginUser&sort=" ).append( sortKey )
				.append( "&direc=" ).append( !direc )
				.append( "\">" ).append( sortKey ).append( "</a></th>" );
		}

		rtn.append( " </tr></thead>" ).append( CR )
			// 5.9.1.2 (2015/10/23) 自己終了警告対応
//			.append( " <colgroup class=\"S9\" />" ).append( CR )
//			.append( " <colgroup class=\"X\" span=\"6\" />" ).append( CR )
			.append( " <colgroup class=\"S9\" ><!-- --></colgroup>" ).append( CR )
			.append( " <colgroup class=\"X\" span=\"6\" ><!-- --></colgroup>" ).append( CR )
			.append( CR );

		for( int i=0; i<userInfos.length; i++ ) {
			final UserSummary userInfo = userInfos[i] ;
			rtn.append( " <tr class=\"row_" ).append( i%2 ).append( "\" >" ).append( CR )
				.append( "  <td>" ).append( String.valueOf( i+1 )   ).append( "</td>" ).append( CR )
				.append( "  <td>" ).append( userInfo.getUserID()    ).append( "</td>" ).append( CR )
				.append( "  <td>" ).append( userInfo.getJname()     ).append( "</td>" ).append( CR )
				.append( "  <td>" ).append( userInfo.getRoles()     ).append( "</td>" ).append( CR )
				.append( "  <td>" ).append( userInfo.getIPAddress() ).append( "</td>" ).append( CR )
				.append( "  <td>" ).append( HybsSystem.getDate( userInfo.getLoginTime() ) ).append( "</td>" ).append( CR )
				.append( "  <td>" ).append( userInfo.getAttribute( "LASTACCESS") ).append( "</td>" ).append( CR )
				.append( "  <td>" ).append( StringUtil.nval( userInfo.getAttribute( "LASTGAMENNM"), "" ) ).append( "</td>" ).append( CR )  // 4.4.0.1 (2009/08/08)
				.append( " </tr>" ).append( CR );
		}
//		rtn.append( "</table>" ).append( CR );
		rtn.append( TABLE_END );

		return rtn.toString() ;
	}

	/**
	 * PlugIn 情報を作成します。
	 *
	 * @og.rev 4.0.0.0 (2005/08/31) 新規作成
	 * @og.rev 5.6.6.0 (2013/07/05) "DBConstValue","Daemon","JspCreate" を、追加します。
	 *
	 * @return	PlugIn情報
	 * @og.rtnNotNull
	 */
	private String getPlugInInfo() {

		final String[] pluginType = new String[] {
								"Query","Renderer","Editor","DBType","ViewForm",
								"TableReader","TableWriter","TableFilter","ChartWriter","CalendarQuery",
								"DBConstValue","Daemon","JspCreate"					// 5.6.6.0 (2013/07/05) 追加
							} ;

		final ClassInfo info = new ClassInfo();

		for( int j=0; j<pluginType.length; j++ ) {
			final String type = pluginType[j] ;
			final HybsEntry[] names = HybsSystem.sysEntry( type + "_" );
			for( int i=0; i<names.length; i++ ) {
				final String key		= names[i].getKey().substring( type.length()+1 );
				final String clsName	= names[i].getValue();
				info.addLine( type,key,clsName );
			}
		}
		return info.getClassInfoData() ;
	}

	/**
	 * Taglib 情報を作成します。
	 *
	 * @og.rev 4.0.0.0 (2006/01/31) 新規作成
	 * @og.rev 5.3.6.0 (2011/06/01) Taglib クラス名の求め方を変更します。(jar版のみ)
	 * @og.rev 5.5.2.6 (2012/05/25) JarFile を、Closer#zipClose( ZipFile ) メソッドを利用して、close します。
	 *
	 * @return	Taglib情報
	 * @og.rtnNotNull
	 */
	private String getTaglibInfo() {
		final ClassInfo info = new ClassInfo();

		// 5.5.2.6 (2012/05/25) findbugs対応
		JarFile jarFile = null;
		try {
			final ClassLoader loader = Thread.currentThread().getContextClassLoader();
			final Enumeration<URL> enume = loader.getResources( "org/opengion/hayabusa/taglib/" );		// 4.3.3.6 (2008/11/15) Generics警告対応
			while( enume.hasMoreElements() ) {
				final URL url = enume.nextElement();		// 4.3.3.6 (2008/11/15) Generics警告対応
				// jar:file:/実ディレクトリ または、file:/実ディレクトリ
				final String dir = url.getFile();
				if( "jar".equals( url.getProtocol() ) ) {
					// dir = file:/G:/webapps/gf/WEB-INF/lib/hayabusa4.0.0.jar!/org/opengion/hayabusa/taglib 形式です。
					final String jar = dir.substring(dir.indexOf( ':' )+1,dir.lastIndexOf( '!' ));
					// jar = /G:/webapps/gf/WEB-INF/lib/hayabusa4.0.0.jar 形式に切り出します。
					jarFile = new JarFile( jar );
					final Enumeration<JarEntry> en = jarFile.entries() ;		// 4.3.3.6 (2008/11/15) Generics警告対応
					while( en.hasMoreElements() ) {
						final JarEntry ent = en.nextElement();		// 4.3.3.6 (2008/11/15) Generics警告対応
						final String file = ent.getName();
						if( ! ent.isDirectory() && file.endsWith( "Tag.class" ) ) {
							final String type		= "Taglib_jar";
							// 5.3.6.0 (2011/06/01) Taglib クラス名の求め方を変更します。(jar版のみ)
							final String key		= file.substring( file.lastIndexOf( '/' )+1,file.length()-6 );	// -6 は、.class 分
							final String clsName	= file.replace( '/','.' ).substring( 0,file.length()-6 );
							info.addLine( type,key,clsName );
						}
					}
					Closer.zipClose( jarFile );		// 5.5.2.6 (2012/05/25) findbugs対応
					jarFile = null;					// 正常終了時に、close() が２回呼ばれるのを防ぐため。
				}
				else {
					// dir = /G:/webapps/gf/WEB-INF/classes/org/opengion/hayabusa/taglib/ 形式です。
//					final File fileObj = new File( dir );
//					final File[] list = fileObj.listFiles();
					final File[] list = new File( dir ).listFiles();
					// 6.3.9.0 (2015/11/06) null になっている可能性がある(findbugs)
					if( list != null ) {
						for( final File file : list ) {
							final String name = file.getName() ;
							if( file.isFile() && name.endsWith( "Tag.class" ) ) {
								final String type		= "Taglib_file";
								final String key		= name.substring( 0,name.length()-6 );
								final String clsName	= "org.opengion.hayabusa.taglib." + key ;
								info.addLine( type,key,clsName );
							}
						}
					}
//					for( int i=0; i<list.length; i++ ) {
//						final String file = list[i].getName() ;
//						if( list[i].isFile() && file.endsWith( "Tag.class" ) ) {
//							final String type		= "Taglib";
//							final String key		= file.substring( 0,file.length()-6 );
//							final String clsName	= "org.opengion.hayabusa.taglib." + key ;
//							info.addLine( type,key,clsName );
//						}
//					}
				}
			}
		}
		catch( IOException ex ) {
			final String errMsg = "taglibファイル読み取りストリームを失敗しました。"
					+ CR + ex.getMessage();
			throw new OgRuntimeException( errMsg,ex );
		}
		finally {
			Closer.zipClose( jarFile );		// 5.5.2.6 (2012/05/25) findbugs対応
		}

		return info.getClassInfoData() ;
	}

	/**
	 * クラス情報を表示するためのデータを管理します。
	 * ここでは、引数に渡された、分類(Classify)、名称(Key Name)、クラス名(Class Name)、
	 * バージョン(Version)情報をテーブル形式で表示します。
	 * バージョン情報は、クラス名から、インスタンスを作成して、private static final String VERSION
	 * フィールドの値を読み取ります。
	 *
	 * @og.rev 4.0.0.0 (2006/01/31) 新規作成
	 */
	private static final class ClassInfo {
		private final StringBuilder rtn = new StringBuilder( BUFFER_MIDDLE );
		private int cnt;

		/**
		 * コンストラクター
		 *
		 * @og.rev 4.0.0.0 (2006/01/31) 新規作成
		 * @og.rev 5.9.1.2 (2015/10/23) 自己終了警告対応
		 */
		public ClassInfo() {
//			rtn.append( "<table " ).append( TABLE_HEADER ).append( " >" ).append( CR );
//			rtn.append( "	<thead><tr><th>No</th><th>Classify</th><th>Key Name</th><th>Class Name</th><th>Version</th></tr></thead>" ).append( CR );
//			rtn.append( "	<colgroup class=\"S9\" />" ).append( CR );
//			rtn.append( "	<colgroup class=\"X\" />"  ).append( CR );
//			rtn.append( "	<colgroup class=\"X\" />"  ).append( CR );
//			rtn.append( "	<colgroup class=\"X\" />"  ).append( CR );
//			rtn.append( "	<colgroup class=\"X\" />"  ).append( CR );
//			rtn.append( "	<colgroup class=\"X\" />"  ).append( CR );
//			rtn.append( CR );

			// 5.9.1.2 (2015/10/23) 自己終了警告対応。ついでに、append連結しておきます。
			rtn.append( "<table " ).append( TABLE_HEADER ).append( " >" ).append( CR )
				.append( "	<thead><tr><th>No</th><th>Classify</th><th>Key Name</th><th>Class Name</th><th>Version</th></tr></thead>" ).append( CR )
				.append( "	<colgroup class=\"S9\" ><!-- --></colgroup>" ).append( CR )
				.append( "	<colgroup class=\"X\" ><!-- --></colgroup>"  ).append( CR )
				.append( "	<colgroup class=\"X\" ><!-- --></colgroup>"  ).append( CR )
				.append( "	<colgroup class=\"X\" ><!-- --></colgroup>"  ).append( CR )
				.append( "	<colgroup class=\"X\" ><!-- --></colgroup>"  ).append( CR )
				.append( "	<colgroup class=\"X\" ><!-- --></colgroup>"  ).append( CR )
				.append( CR );
		}

		/**
		 * テーブル表示用のデータを追加します。
		 *
		 * @og.rev 4.0.0.0 (2006/01/31) 新規作成
		 *
		 * @param	type	タイプ属性
		 * @param	key 	キー属性
		 * @param	clsName	クラス名(このクラス名からインスタンス化します。)
		 */
		public void addLine( final String type, final String key, final String clsName ) {
			final String version	= getFieldValue( clsName );

			final boolean isCustom = version.compareTo( BuildNumber.VERSION_NO ) > 0
								|| version.indexOf( "Pache"   ) >= 0
								|| version.indexOf( "Nightly" ) >= 0 ;

			final String trType = isCustom ? "warning" : String.valueOf( cnt%2 );

			rtn.append( "	<tr class=\"row_" ).append( trType ).append( "\" >" ).append( CR )
				.append( "		<td>" ).append( cnt++    ).append( "</td>" ).append( CR )
				.append( "		<td>" ).append( type     ).append( "</td>" ).append( CR )
				.append( "		<td>" ).append( key      ).append( "</td>" ).append( CR )
				.append( "		<td>" ).append( clsName  ).append( "</td>" ).append( CR )
				.append( "		<td>" ).append( version  ).append( "</td>" ).append( CR )
				.append( "	</tr>" ).append( CR );
		}

		/**
		 * すべての内部のデータを文字列化して返します。
		 *
		 * @og.rev 4.0.0.0 (2006/01/31) 新規作成
		 * @og.rev 6.3.6.1 (2015/08/28) <table> 枠の指定と、</table> の static final String 化
		 *
		 * @return	作成されたテーブルデータ
		 * @og.rtnNotNull
		 */
		public String getClassInfoData() {
//			rtn.append( "</table>" ).append( CR );
			rtn.append( TABLE_END );
			return rtn.toString() ;
		}

		/**
		 * 指定のオブジェクトの  VERSION staticフィールドの値を取得します。
		 *
		 * @og.rev 4.0.0.0 (2005/08/31) 新規作成
		 *
		 * @param	clsName	指定のクラスを表す名称
		 * @return	VERSIONフィールドの値(エラー時は、そのメッセージ)
		 */
		private String getFieldValue( final String clsName ) {
			String rtn ;
			try {
				final Object obj = HybsSystem.newInstance( clsName );
				final Field fld = obj.getClass().getDeclaredField( "VERSION" ) ;
				// privateフィールドの取得には、accessibleフラグを trueにする必要があります。
				fld.setAccessible( true );

				rtn = (String)fld.get( null );
			}
			catch( Throwable ex ) {
				rtn = ex.getMessage();
			}
			return rtn ;
		}
	}

	/**
	 * systemResource 情報を作成します。
	 *
	 * @og.rev 3.5.3.1 (2003/10/31) 機能ごとにメソッドを呼び出すように修正します。
	 * @og.rev 4.0.0.0 (2005/01/31) DBColumn の 属性(CLS_NM)から、DBTYPEに変更
	 * @og.rev 6.3.6.1 (2015/08/28) <table> 枠の指定と、</table> の static final String 化
	 * @og.rev 5.9.1.2 (2015/10/23) 自己終了警告対応
	 *
	 * @return	systemResource情報
	 * @og.rtnNotNull
	 */
	private String getSystemResource() {
		final StringBuilder rtn = new StringBuilder( BUFFER_MIDDLE )
			.append( "<table " ).append( TABLE_HEADER ).append( " >" ).append( CR )
			.append( "	<thead><tr><th>No</th><th>Key</th><th>Value</th></tr></thead>" ).append( CR )
			// 5.9.1.2 (2015/10/23) 自己終了警告対応
//			.append( "	<colgroup class=\"S9\" />" ).append( CR )
//			.append( "	<colgroup class=\"X\" span=\"2\" />" ).append( CR )
			.append( "	<colgroup class=\"S9\" ><!-- --></colgroup>" ).append( CR )
			.append( "	<colgroup class=\"X\" span=\"2\" ><!-- --></colgroup>" ).append( CR )
			.append( CR );

		final String[][] str = HybsSystem.getSystemResourceData();
		for( int i=0; i<str[0].length; i++ ) {
			rtn.append( "	<tr class=\"row_" ).append( i%2 ).append( "\" >" ).append( CR )
				.append( "		<td>" ).append( String.valueOf( i+1 ) ).append( "</td>" ).append( CR )
				.append( "		<td>" ).append( str[0][i] ).append( "</td>" ).append( CR )
				.append( "		<td>" ).append( str[1][i] ).append( "</td>" ).append( CR )
				.append( "	</tr>" ).append( CR );
		}
//		rtn.append( "</table>" ).append( CR );
		rtn.append( TABLE_END );

		return rtn.toString();
	}

	/**
	 * AccessStop 情報を作成します。
	 *
	 * @og.rev 3.5.3.1 (2003/10/31) 機能ごとにメソッドを呼び出すように修正します。
	 * @og.rev 4.0.0.0 (2007/11/29) AccessStopFilter#getStopFilter() ⇒ isStopFilter() に変更
	 * @og.rev 6.1.1.0 (2015/01/17) パッケージを import するのと、ロジック見直し。
	 * @og.rev 6.3.8.0 (2015/09/11) AccessStop を外部から制御できるように、コマンドと状態を受けるようにする。
	 *
	 * @param	isStop		true:停止/false:実行
	 * @return	AccessStop情報
	 * @og.rtnNotNull
	 */
//	private String getAccessStop() {
	private String getAccessStop( final String isStop ) {

//		final boolean flag = !AccessStopFilter.isStopFilter();
		final boolean flag = Boolean.parseBoolean( isStop );		// 6.3.8.0 (2015/09/11)
	//	AccessStopFilter.setStopFilter( flag );

		return flag ? "Webアプリケーションのサービスを停止します。"
					: "Webアプリケーションのサービスを開始します。";
	}

	/**
	 * Servlet/Filter 情報を作成します。
	 *
	 * @og.rev 6.3.8.3 (2015/10/03) Servlet/Filter 情報を作成。
	 * @og.rev 5.9.1.2 (2015/10/23) 自己終了警告対応
	 *
	 * @param	session	HttpSessionオブジェクト
	 *
	 * @return	Servlet/Filter情報
	 * @og.rtnNotNull
	 */
	private String getRegistrationMessage( final HttpSession session ) {
		final ServletContext context = session.getServletContext();

		final StringBuilder rtn = new StringBuilder( BUFFER_MIDDLE )
			.append( "<table " ).append( TABLE_HEADER ).append( " >" ).append( CR )
			.append( "	<thead><tr><th>No</th><th>Type</th><th>Key</th><th>Class</th><th>Mappings</th></tr></thead>" ).append( CR )
			// 5.9.1.2 (2015/10/23) 自己終了警告対応
//			.append( "	<colgroup class=\"S9\" />" ).append( CR )
//			.append( "	<colgroup class=\"X\" />" ).append( CR )
//			.append( "	<colgroup class=\"X\" />" ).append( CR )
//			.append( "	<colgroup class=\"X\" />" ).append( CR )
//			.append( "	<colgroup class=\"X\" />" ).append( CR )
			.append( "	<colgroup class=\"S9\" ><!-- --></colgroup>" ).append( CR )
			.append( "	<colgroup class=\"X\" ><!-- --></colgroup>" ).append( CR )
			.append( "	<colgroup class=\"X\" ><!-- --></colgroup>" ).append( CR )
			.append( "	<colgroup class=\"X\" ><!-- --></colgroup>" ).append( CR )
			.append( "	<colgroup class=\"X\" ><!-- --></colgroup>" ).append( CR )
			.append( CR );

		int no = 0;
		for( final ServletRegistration reg : context.getServletRegistrations().values() ) {
			rtn.append( "	<tr class=\"row_" ).append( no%2 ).append( "\" >" ).append( CR )
				.append( "		<td>" ).append( String.valueOf( ++no )	).append( "</td>" ).append( CR )
				.append( "		<td>Servlet</td>" ).append( CR )
				.append( "		<td>" ).append( reg.getName()			).append( "</td>" ).append( CR )
				.append( "		<td>" ).append( reg.getClassName()		).append( "</td>" ).append( CR )
				.append( "		<td>" ).append( reg.getMappings()		).append( "</td>" ).append( CR )
				.append( "	</tr>" ).append( CR );
		}

		for( final FilterRegistration reg : context.getFilterRegistrations().values() ) {
			rtn.append( "	<tr class=\"row_" ).append( no%2 ).append( "\" >" ).append( CR )
				.append( "		<td>" ).append( String.valueOf( ++no )		).append( "</td>" ).append( CR )
				.append( "		<td>Filter</td>" ).append( CR )
				.append( "		<td>" ).append( reg.getName()				).append( "</td>" ).append( CR )
				.append( "		<td>" ).append( reg.getClassName()			).append( "</td>" ).append( CR )
				.append( "		<td>" ).append( reg.getUrlPatternMappings() ).append( "</td>" ).append( CR )
				.append( "	</tr>" ).append( CR );
		}

		rtn.append( TABLE_END );

		return rtn.toString();
	}

	/**
	 * admin リンク情報を作成します。
	 * 簡易メソッドなので、国際化対応していません。
	 *
	 * @og.rev 3.5.4.1 (2003/12/01) 新規作成
	 * @og.rev 5.1.1.2 (2009/12/10) 画面IDを変更
	 * @og.rev 5.6.3.4 (2013/04/26) ｸｲｯｸﾘﾌｧﾚﾝｽ 画面を追加
	 * @og.rev 6.3.6.1 (2015/08/28) <table> 枠の指定と、</table> の static final String 化
	 * @og.rev 6.3.8.0 (2015/09/11) AccessStop を外部から制御できるように、コマンドと状態を受けるようにする。
	 *
	 * @return	アドミンリンク情報
	 * @og.rtnNotNull
	 */
	private String getAdminLink() {
		final StringBuilder rtn = new StringBuilder( BUFFER_MIDDLE )
			.append( "<table><tr>" ).append( CR )
			.append( "<td width=\"10px\"/>" ).append( CR )
			.append( "<td>[<a href=\"admin?COMMAND=infomation\" target=\"RESULT\" >状況表示</a>]</td>" ).append( CR )
			.append( "<td width=\"10px\"/>" ).append( CR )
			.append( "<td>[<a href=\"admin?COMMAND=close\" target=\"RESULT\" >ﾌﾟｰﾙ削除</a>]</td>" ).append( CR )
			.append( "<td width=\"10px\"/><td>" ).append( CR )
			.append( "[<a href=\"admin?COMMAND=loginUser\" target=\"RESULT\" >ﾛｸﾞｲﾝﾕｰｻﾞｰ</a>]</td>" ).append( CR )
			.append( "<td width=\"10px\"/><td>" ).append( CR )
			.append( "[<a href=\"admin?COMMAND=plugin\" target=\"RESULT\" >ﾌﾟﾗｸﾞｲﾝ情報</a>]</td>" ).append( CR )
			.append( "<td width=\"10px\"/><td>" ).append( CR )
			.append( "[<a href=\"admin?COMMAND=taglib\" target=\"RESULT\" >ﾀｸﾞﾘﾌﾞ情報</a>]</td>" ).append( CR )
			.append( "<td width=\"10px\"/><td>" ).append( CR )
			.append( "[<a href=\"admin?COMMAND=registrationInfo\" target=\"RESULT\" >Servlet/Filter情報</a>]</td>" ).append( CR )
			.append( "<td width=\"10px\"/><td>" ).append( CR )
			.append( "[<a href=\"common/quickReference.html\" target=\"RESULT\" >ｸｲｯｸﾘﾌｧﾚﾝｽ</a>]</td>" ).append( CR )	// 5.6.3.4 (2013/04/26)
			.append( "<td width=\"10px\"/><td>" ).append( CR )
			.append( "[<a href=\"admin?COMMAND=systemResource\" target=\"RESULT\" >ｼｽﾃﾑﾘｿｰｽ</a>]</td>" ).append( CR )
			.append( "<td width=\"10px\"/><td>" ).append( CR )
			// 6.3.8.0 (2015/09/11) AccessStop を外部から制御できるように、コマンドと状態を受けるようにする。
//			.append( "[<a href=\"admin?COMMAND=AccessStop\" target=\"RESULT\" >ｱｸｾｽ制限</a>]</td>" ).append( CR )
			.append( "ｱｸｾｽ制限[<a href=\"admin?COMMAND=AccessStop&stop=true\"  target=\"RESULT\" >停止</a>]/" )
			.append( "[<a href=\"admin?COMMAND=AccessStop&stop=false\" target=\"RESULT\" >開始</a>]</td>" ).append( CR )
			.append( "<td width=\"10px\"/><td>" ).append( CR )
			.append( "[<a href=\"common/gamen/01_ADMIN/parameter.jsp?GAMENID=01_ADMIN\" target=\"RESULT\" >ﾊﾟﾗﾒｰﾀ</a>]</td>" ).append( CR )
			.append( "<td width=\"10px\"/><td>" ).append( CR )
			.append( "[<a href=\"common/gamen/01_ADMIN/systemInfo.jsp?GAMENID=01_ADMIN\" target=\"RESULT\" >ｼｽﾃﾑ状況</a>]</td>" ).append( CR )
//			.append( "</tr></table>" ).append( CR );
			.append( "</tr>" ).append( TABLE_END );

		return rtn.toString();
	}

	/**
	 * DBTableModel から テーブルのタグ文字列を作成して返します。
	 *
	 * @og.rev 3.5.3.1 (2003/10/31) User情報のテーブルの設定を、システムリソース より行う。
	 * @og.rev 5.2.2.0 (2010/11/01) SystemData 見直し漏れの対応。
	 *
	 * @return	テーブルのタグ文字列
	 * @og.rtnNotNull
	 */
	private static String getTableHeaderTag() {
		// 6.1.1.0 (2015/01/17) Attributesの連結記述
		return new Attributes()
					.set( "id"		,"viewTable"	)		// 3.6.0.5 (2004/10/18)
					.set( "summary"	,"layout"		)		// サマリー
					.getAttribute();
	}

	/**
	 * userInfo 情報をセットします。
	 * 現状、このメソッドではLANGのみ設定可能です。
	 * （予約語以外のセットはuserInfoタグで行う）
	 * 
	 * LANGではuserInfoを切り替えた後にユーザのGUIInfoを再ロードします。
	 *
	 * @og.rev 5.9.1.0 (2015/10/02) 新規作成
	 *
	 * @param	req	HttpServletRequestオブジェクト
	 * @return	結果情報
	 */
	private String setUserinfo( final HttpServletRequest req ) {
		final HttpSession session = req.getSession();
		final UserInfo userInfo = (UserInfo)(session.getAttribute( HybsSystem.USERINFO_KEY ));

		final String infoLang = req.getParameter( "LANG" );

		String rtn = null ;
		if( userInfo != null && infoLang != null && infoLang.length() > 0) {
			userInfo.setLang( infoLang );
			ResourceFactory.newInstance( userInfo.getLang() ).makeGUIInfos( userInfo );
			rtn = "User language : " + infoLang;
		}

		if( rtn == null || rtn.length() == 0 ) {
			rtn = "Error";
		}

		return rtn ;
	}
}
