/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.util;

import static org.opengion.fukurou.system.HybsConst.CR;				// 6.1.0.0 (2014/12/26) refactoring
import static org.opengion.fukurou.system.HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring

import java.util.List;
import java.util.ArrayList;

/**
 * Options.java は、String 型リストをプールするクラスです。
 *
 * HTMLのOptionタグのように、複数の文字列をキープしておき、
 * すべてを連結して出力するような場合に利用できます。
 *
 * この実装は同期化されません。
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class Options {
	private final List<String> option = new ArrayList<>( 100 );

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public Options() {
		super();		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。
	}

	/**
	 * すべての要素をリストから削除します 。
	 * この呼び出しからの復帰後、リストは空になります 。
	 * (例外をスローした場合を除く)。
	 *
	 */
	public void clear()  {
		option.clear() ;
	}

	/**
	 * リストの末尾に指定の文字列を追加します。
	 * value が null の場合は､追加されません。
	 *
	 * @param    value リストに追加される文字列
	 */
	public void add( final String value ) {
		if( value != null ) { option.add( value ) ; }
	}

	/**
	 * リスト内の指定された位置にある要素を返します。
	 * ただし、value が null の場合は、追加されていませんので、
	 * index の順番と 登録データの順番が異なる場合もありますので､
	 * 注意する必要があります。
	 *
	 * @param    index 返される要素のインデックス
	 *
	 * @return   リスト内の指定された位置にある要素
	 */
	public String get( final int index ) {
		return option.get( index ) ;
	}

	/**
	 * 登録されているオプションの数を返します。
	 *
	 * @return   インタフェース List 内の size
	 *
	 */
	public int size() {
		return option.size() ;
	}

	/**
	 * リスト内のすべての要素を正しい順序で保持する配列を返します。
	 *
	 * @return   リスト内のすべての要素の配列
	 * @og.rtnNotNull
	 */
	public String[] toArray() {
		return option.toArray( new String[option.size()] ) ;
	}

	/**
	 * リストに含まれているデータを オプションタグ形式で返します。
	 * 各オプションタグは整形します。(各リスト毎に改行を入れます。)
	 *
	 * @return  Optionタグの形式で返します
	 * @og.rtnNotNull
	 */
	public String getOption() {
		return getOption( true );
	}

	/**
	 * リストに含まれているデータを オプションタグ形式で返します。
	 * 各オプションタグの整形をする／しないは、パラメータで指定します。
	 *
	 * @param	flag  整形する(true)／整形しない(false)
	 *
	 * @return	Optionタグの形式で返します
	 * @og.rtnNotNull
	 */
	public String getOption( final boolean flag ) {
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );

		final String crlf = flag ? CR : " ";

		for( int i=0; i<option.size(); i++ ) {
			buf.append( option.get(i) ).append( crlf );
		}
		return buf.toString();
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return	オブジェクトの文字列表現
	 */
	@Override
	public String toString() {
		return getOption( false ) ;
	}
}
