/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import org.opengion.fukurou.system.OgBuilder;									// 6.4.5.0 (2016/04/08)
import org.opengion.fukurou.model.NativeType;
import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.TagBuffer;
import org.opengion.fukurou.util.StringUtil;									// 6.4.4.2 (2016/04/01)
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.resource.CodeData;
import org.opengion.hayabusa.resource.ColumnData;
import org.opengion.hayabusa.resource.LabelData;
import org.opengion.hayabusa.resource.RoleMode;

/**
 * DBType インターフェースを継承した Abstractクラスです。
 * getRendererValue( String value ) 、getEditorValue( String value ) 、
 * isValueChack( String ) メソッドを、サブクラスで実装する必要があります。
 *
 * @og.rev 3.4.0.0 (2003/09/01) 表示パラメータ、編集パラメータ、文字パラメータの追加。
 * @og.group テーブル管理
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class DBColumn {

	private final ColumnData columnData ;
	private final LabelData  labelData ;
	private final CodeData   codeData ;

	private final CellRenderer	cellRenderer	;		// 表示用レンデラー
	private final CellEditor	cellEditor		;		// 編集用エディター
	private final DBType		dbType			;		// データのタイプ

	private final String		lang			;		// 言語
	private final boolean		writable		;		// カラムが書き込み可能かどうか
	private final String		defValue		;		// データのデフォルト値
	private final Attributes	rendAttri		;		// 表示用レンデラー追加用属性
	private final Attributes	editAttri		;		// 編集用エディター追加用属性
	private final boolean		addNoValue 		;		// メニューに空の選択リストを追加するかどうか	// 3.5.5.7 (2004/05/10)
	private final String		addKeyLabel		;		// 6.0.4.0 (2014/11/28) キー:ラベル形式で表示するかどうか
	private final boolean		writeKeyLabel	;		// 6.2.3.0 (2015/05/01)
	private final String		dbid			;		// データベース接続先ID
	private final boolean		official	 	;		// カラムリソースから作成されたかどうか		// 3.6.0.7 (2004/11/06)

	private final int			checkLevel		;		// DBColumn の 整合性チェックを行うレベルを規定します。

	// 6.0.4.0 (2014/11/28) selection は、Column から取得するのではなく、Factory で作成する。

	private final String		eventColumn		;	// 4.3.6.0 (2009/04/01) イベントカラム
	private final String		eventValue		;	// 6.3.3.0 (2015/07/25) eventValue 追加
	private final String		rawEditParameter;	// 4.3.6.0 (2009/04/01) {@XXXX}の変換がされていない生パラメータ
	private final String		rawRendParameter;	// 5.1.7.0 (2009/04/01) {@XXXX}の変換がされていない生パラメータ
	private final String		eventURL		;	// 4.3.6.0 (2009/04/01) イベントカラムで利用するURL

	private final String		useSLabel		;	// 5.5.1.0 (2012/04/03) MENUのベース表示の切り替え
	private final String		noDisplayVal	;	// 5.6.2.3 (2013/03/22) 非表示文字の設定

	private final boolean		stringOutput	;	// 5.7.6.3 (2013/05/23) ファイルレンデラ出力時のフラグ

	// ※ 内部の各属性から作り出す値です。CustomTable等では、毎回呼ばれるため、キャッシュします。
	private String				className		;	// 6.4.5.0 (2016/04/08) キャッシュします。

	/**
	 * DBColumnConfig オブジェクトより作成されるコンストラクター
	 * すべての情報は、インスタンス作成時に設定します。
	 * このオブジェクトは、１度作成されると変更されることはありません。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規追加
	 * @og.rev 4.0.0.0 (2007/11/07) Selectionオブジェクトをキャッシュする
	 * @og.rev 4.3.6.0 (2009/04/01) eventColumnの対応
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 * @og.rev 5.5.1.0 (2012/04/03) useSLabel対応
	 * @og.rev 5.6.2.3 (2013/03/22) 非表示文字の設定。noDisplayVal追加
	 * @og.rev 5.7.3.0 (2014/02/07) SelectionFactory 対応
	 * @og.rev 5.7.6.3 (2014/05/23) stringOutput対応
	 * @og.rev 6.0.4.0 (2014/11/28) selection は、Column から取得するのではなく、Factory で作成する。
	 * @og.rev 6.0.4.0 (2014/11/28) addKeyLabel 属性を追加
	 * @og.rev 6.2.3.0 (2015/05/01) writeKeyLabel 属性を追加
	 * @og.rev 6.3.3.0 (2015/07/25) eventValue 追加
	 *
	 * @param   lang    言語
	 * @param   clmData カラムデータオブジェクト
	 * @param   lblData ラベルデータオブジェクト
	 * @param   cdData  コードデータオブジェクト
	 */
	public DBColumn( final String 	  lang,
					 final ColumnData clmData ,
					 final LabelData  lblData ,
					 final CodeData   cdData ) {
		this.lang       = lang ;
		this.columnData = clmData ;
		this.labelData  = lblData ;
		this.codeData   = cdData  ;

		writable  = true ;

		try {
			dbType = DBTypeFactory.newInstance( columnData.getDbType() );
		}
		catch( RuntimeException ex ) {
			final String errMsg = "dbType の作成に失敗しました。"
						+ " name=[" + columnData.getName() + "]"
						+ " dbType=[" + columnData.getDbType() + "] "
						+ ex.getMessage();
			throw new HybsSystemException( errMsg,ex );
		}

		rendAttri		= null	;
		editAttri		= null	;
		addNoValue		= false	;
		addKeyLabel		= null	;		// 6.0.4.0 (2014/11/28) キー:ラベル形式で表示するかどうか
		writeKeyLabel	= false	;		// 6.2.3.0 (2015/05/01) writeKeyLabel 属性を追加
		official		= true	;		// 3.6.0.7 (2004/11/06)
		checkLevel		= -1;
		dbid			= null	;		// 標準から作成されるカラムオブジェクトは、DEFAULT 接続先を設定する。

		eventColumn		= null; 		// 4.3.6.0 (2009/04/01)
		eventValue		= null;			// 6.3.3.0 (2015/07/25) eventValue 追加
		rawEditParameter = clmData.getEditorParam(); // 4.3.6.0 (2009/04/01)
		rawRendParameter = clmData.getRendererParam(); // 5.1.7.0 (2010/06/01)
		eventURL		= null; 		// 4.3.6.0 (2009/04/01)

		useSLabel	= "auto";		// 5.5.1.0
		noDisplayVal= null;			// 5.6.2.3 (2013/03/22) 非表示文字の設定

		stringOutput = HybsSystem.sysBool( "USE_STRING_EXCEL_OUTPUT" );		// 5.7.6.3 (2014/05/23)

		final String def = columnData.getDefault();
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		defValue = def == null ? dbType.getDefault() : def;

//		if( def != null ) { defValue = def; }
//		else {              defValue = dbType.getDefault() ; }

		// 5.7.3.0 (2014/02/07) SelectionFactory 対応
		// 6.0.4.0 (2014/11/28) selection は、Column から取得するのではなく、Factory で作成する。

		try {
			cellRenderer = DBCellFactory.newRenderer( columnData.getRenderer(),this );
			cellEditor   = DBCellFactory.newEditor(   columnData.getEditor(),  this );
		}
		catch( RuntimeException ex ) {
			final String errMsg = "Renderer,Editor の作成に失敗しました。"
						+ " name=["		+ columnData.getName()		+ "]"
						+ " Renderer=[" + columnData.getRenderer()	+ "]"
						+ " Editor=["	+ columnData.getEditor()	+ "]"
						+ ex.getMessage();
			throw new HybsSystemException( errMsg,ex );
		}
	}

	/**
	 * DBColumnConfig オブジェクトより作成されるコンストラクター
	 * すべての情報は、インスタンス作成時に設定します。
	 * このオブジェクトは、１度作成されると変更されることはありません。
	 *
	 * @og.rev 4.0.0.0 (2007/11/07) Selectionオブジェクトをキャッシュする
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 * @og.rev 5.1.8.0 (2010/07/01) メソッド名変更(getDefValue ⇒ getDefault)
	 * @og.rev 5.5.1.0 (2012/04/03) useSLabel対応
	 * @og.rev 5.6.2.3 (2013/03/22) 非表示文字の設定。noDisplayVal追加
	 * @og.rev 5.6.6.0 (2013/07/05) codeKeyVal 属性を使用した、Selection_KEYVAL オブジェクトの作成
	 * @og.rev 5.7.3.0 (2014/02/07) SelectionFactory 対応
	 * @og.rev 5.7.6.3 (2014/05/23) stringOutput対応
	 * @og.rev 6.0.4.0 (2014/11/28) selection は、Column から取得するのではなく、Factory で作成する。
	 * @og.rev 6.0.4.0 (2014/11/28) addKeyLabel 属性を追加
	 * @og.rev 6.2.3.0 (2015/05/01) writeKeyLabel 属性を追加
	 * @og.rev 6.3.3.0 (2015/07/25) eventValue 追加
	 *
	 * @param   config DBColumnConfigオブジェクト
	 */
	public DBColumn( final DBColumnConfig config ) {
		lang			= config.getLang()		;

		columnData		= config.getColumnData();
		labelData		= config.getLabelData();
		codeData		= config.getCodeData();

		writable		= config.isWritable();
		dbType			= DBTypeFactory.newInstance( columnData.getDbType() );
		rendAttri		= config.getRendererAttributes();
		editAttri		= config.getEditorAttributes();
		addNoValue		= config.isAddNoValue();
		addKeyLabel		= config.getAddKeyLabel();		// 6.0.4.0 (2014/11/28) キー:ラベル形式で表示するかどうか
		writeKeyLabel	= config.isWriteKeyLabel();		// 6.2.3.0 (2015/05/01) writeKeyLabel 属性を追加
		official		= config.isOfficial();			// 3.6.0.7 (2004/11/06)
		dbid			= config.getDbid();

		eventColumn  	= config.getEventColumn(); 		// 4.3.6.0 (2009/04/01)
		eventValue		= config.getEventValue();		// 6.3.3.0 (2015/07/25) eventValue 追加
		rawEditParameter = config.getRawEditParameter(); // 4.3.6.0 (2009/04/01)
		rawRendParameter = config.getRawRendParameter(); // 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
		eventURL	 	 = config.getEventURL();		// 4.3.6.0 (2009/04/01)

		useSLabel		 = config.getUseSLabel();		// 5.5.1.0 (2012/04/03)
		noDisplayVal	 = config.getNoDisplayVal();	// 5.6.2.3 (2013/03/22) 非表示文字の設定

		stringOutput	 = config.isStringOutput();		// 5.7.6.3 (2014/05/23)

		// DBColumn の 整合性チェックを行うレベルを規定します。
		final String CHECK_LEVEL = HybsSystem.sys( "DB_OFFICIAL_COLUMN_CHECK_LEVEL" );
		if( !official && CHECK_LEVEL != null && CHECK_LEVEL.length() > 0 ) {
			checkLevel = Integer.parseInt( CHECK_LEVEL );
		}
		else {
			checkLevel = -1;
		}

		final String def = config.getDefault();
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		defValue = def == null ? dbType.getDefault() : def;

//		if( def != null ) { defValue = def; }
//		else {              defValue = dbType.getDefault() ; }

		// 5.6.6.0 (2013/07/05) codeKeyVal 属性を使用した、Selection_KEYVAL オブジェクトの作成
		// 優先順位は、codeData オブジェクトで、codeKeyVal を利用したSelectionは、DBColumnConfig からのみ作成可能
		// 6.0.4.0 (2014/11/28) selection は、Column から取得するのではなく、Factory で作成する。

		cellRenderer = DBCellFactory.newRenderer( columnData.getRenderer(),this );
		cellEditor   = DBCellFactory.newEditor(   columnData.getEditor(),  this );
	}

	/**
	 * 言語を返します。
	 *
	 * @return  言語
	 */
	public String getLang() {
		return lang;
	}

	/**
	 * カラム名を返します。
	 *
	 * @return  カラム名
	 */
	public String getName() {
		return columnData.getName();
	}

	/**
	 * カラムのラベル名を返します。
	 *
	 * @return  カラムのラベル名
	 */
	public String getLabel() {
		return labelData.getLabel();
	}

	/**
	 * カラムのラベル名を返します。
	 *
	 * @return  カラムのラベル名(名前(短))
	 */
	public String getShortLabel() {
		return labelData.getShortLabel();
	}

	/**
	 * カラムのラベル名を返します。
	 *
	 * @return  カラムのラベル名(名前(長))
	 */
	public String getLongLabel() {
		return labelData.getLongLabel();
	}

	/**
	 * カラムのクラスを文字列にした名称を返します。
	 * 内容的には、カラムの className の値と、dbTypeの値をマージした値になります。
	 * さらに、editorAttributesの "class"キーワードの値もマージします。
	 * ただし、この値には、must属性も設定されているため、それだけは除外します。
	 * 各種スタイルを表現するのに使用します。
	 *
	 * @og.rev 6.4.4.2 (2016/04/01) editAttri に clazz属性が設定されている場合は、スペースで連結して返します。
	 * @og.rev 6.4.5.0 (2016/04/08) className , dbType , editAttriのclass属性(除くmust)をスペースで連結して返します。
	 *
	 * @return  カラムのクラスを文字列にした名称(X9,X,など)
	 * @see		#getDbType()
	 */
	public String getClassName() {
		if( className == null ) {
			final String dbTyp = columnData.getDbType();

			final OgBuilder clsNmBuf = new OgBuilder()
										.append( columnData.getClassName() )
										.delete( "VARCHAR2" , "NUMBER" );

			final OgBuilder edtAttBuf = new OgBuilder();
			if( editAttri != null ) {
				edtAttBuf.append( editAttri.get( "class" ) )
							.delete( "mustAny" , "must" );			// mustAnyが先
			}

			className = new OgBuilder()
							.join( " " , dbTyp , clsNmBuf , edtAttBuf )
							.toString();
		}

		return className ;
	}

	/**
	 * フィールドのデータ長を返します。
	 * 通常は、整数型の文字列とデータ長は同じですが,小数点を表すデータ長は
	 * x,y 形式の場合、x + 2 桁で表されます。(マイナス記号と小数点記号)
	 * 7,3 は、 xxxx,yyy のフォーマットで、データ長は、9 になります。
	 *
	 * @og.rev 2.1.1.2 (2002/11/21) 最大桁数入力時の桁数チェックの間違いを訂正。
	 * @og.rev 4.0.0.0 (2005/01/31) メソッド名変更 getMaxlength() → getTotalSize()
	 *
	 * @return  データ長定義文字列
	 */
	public int getTotalSize() {
		return columnData.getTotalSize();
	}

	/**
	 * フィールドの使用桁数を返します。
	 * 小数指定の場合は、"7,3" のようなカンマで整数部、小数部を区切った書式になります。
	 * 7,3 は、 xxxx,yyy のフォーマットで、整数部４桁、小数部３桁を意味します。
	 *
	 * @return  使用桁数
	 */
	public String getMaxlength() {
		return columnData.getMaxlength();
	}

	/**
	 * フィールドの整数部のデータ長を返します。
	 * 通常は、整数型の文字列とデータ長は同じですが,小数点を表すデータ長は
	 * x,y 形式の場合、x - y 桁で表されます。(マイナス記号含まず)
	 * 7,3 は、 xxxx,yyy のフォーマットで、データ長は、4 になります。
	 *
	 * @return  データ長定義文字列
	 */
	public int getSizeX() {
		return columnData.getSizeX() ;
	}

	/**
	 * フィールドの小数部のデータ長を返します。
	 * 通常は、整数型の文字列では、0 になりますが,小数点を表すデータ長は
	 * x,y 形式の場合、y 桁で表されます。
	 * 7,3 は、 xxxx,yyy のフォーマットで、データ長は、3 になります。
	 *
	 * @return  データ長定義文字列
	 */
	public int getSizeY() {
		return columnData.getSizeY() ;
	}

	/**
	 * カラムの表示桁数を返します。
	 * viewLength は、設定した場合のみ、使用できます。通常は、null が返ります。
	 *
	 * @og.rev 3.5.5.5 (2004/04/23) 新規追加
	 * @og.rev 4.0.0.0 (2005/01/31) メソッド名変更 getSize() → getViewLength()
	 *
	 * @return	カラムの文字桁数
	 */
	public String getViewLength() {
		return columnData.getViewLength();
	}

	/**
	 * カラムが書き込み可能かどうかを返します。
	 *
	 * @return  カラムが書き込み可能かどうか
	 */
	public boolean isWritable() {
		return writable;
	}

	/**
	 * データの値そのものではなく、その値のラベル文字を返します。
	 *
	 * @og.rev 3.5.5.4 (2004/04/15) value が null の場合は、""(ゼロストリング)にする。
	 *
	 * @param   value	入力値
	 *
	 * @return  データ表示用の文字列
	 * @og.rtnNotNull
	 */
	public String getRendererValue( final String value ) {
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		return cellRenderer.getValue( value == null ? "" : value );
//		return cellRenderer.getValue( (value != null ) ? value : "" );
	}

	/**
	 * データ表示用のHTML文字列を作成します。
	 * 行番号毎に異なる値を返すことの出来る DBCellRenderer を使用することが出来ます。
	 *
	 * @og.rev 3.5.5.7 (2004/05/10) 行番号に無関係に、値を返すように変更します。
	 * @og.rev 4.0.0.0 (2005/11/30) 行番号に対応した値を返すように変更します。
	 *
	 * @param   row		行番号
	 * @param   value	入力値
	 *
	 * @return  データ編集用の文字列
	 * @og.rtnNotNull
	 */
	public String getRendererValue( final int row,final String value ) {
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		return cellRenderer.getValue( row,value == null ? "" : value );
//		return cellRenderer.getValue( row,(value != null ) ? value : "" );
	}

	/**
	 * データ出力用の文字列を作成します。
	 * ファイル等に出力する形式を想定しますので、HTMLタグを含まない
	 * データを返します。
	 *
	 * writeKeyLabel == true かつ addKeyLabel == "true" の場合は、":" で分離して、
	 * ﾗﾍﾞﾙのみを返します。
	 * IO側で、カラムを分けて出力するので、VAL:LBL ではなく、LBL だけ出力します。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) データ出力用のレンデラー
	 * @og.rev 6.2.3.0 (2015/05/01) writeKeyLabel 属性を追加
	 *
	 * @param   value	入力値
	 *
	 * @return  データ出力用の文字列
	 * @og.rtnNotNull
	 */
	public String getWriteValue( final String value ) {

		String rtnStr = cellRenderer.getWriteValue( value );

		// writeKeyLabel == true かつ addKeyLabel == "true" の場合は、":" で分離して、ﾗﾍﾞﾙのみを返します。
		if( rtnStr != null && isWriteKeyLabel() && "true".equalsIgnoreCase( addKeyLabel ) ) {
			final int ad = rtnStr.indexOf( ':' );
			if( ad >= 0 ) { rtnStr = rtnStr.substring( ad+1 ); }
		}

		return rtnStr ;
	}

	/**
	 * データ入力用の文字列を作成します。
	 * ファイル等から、読み取る場合に、このメソッド経由で、必要な形式変換を行います。
	 * 
	 * 現時点では、KEY:VAL形式の Selection オブジェクト経由でデータを取り込む場合、
	 * KEY に分解処理します。
	 *
	 * @og.rev 6.2.2.0 (2015/03/27) SelectionCellEditor I/Fを追加
	 *
	 * @param   value	入力値
	 *
	 * @return  データ入力用の文字列
	 * @og.rtnNotNull
	 */
	public String getReaderValue( final String value ) {
		String rtnVal = value;
		if( cellEditor instanceof SelectionCellEditor ) {
			rtnVal = ((SelectionCellEditor)cellEditor).getReaderValue( rtnVal );
		}

		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		return rtnVal == null ? "" : rtnVal;
//		return rtnVal != null ? rtnVal : "";
	}

	/**
	 * データ編集用のHTML文字列を作成します。
	 *
	 * @og.rev 3.5.5.4 (2004/04/15) value が null の場合は、""(ゼロストリング)にする。
	 *
	 * @param   value	入力値
	 *
	 * @return  データ編集用の文字列
	 * @og.rtnNotNull
	 */
	public String getEditorValue( final String value ) {
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		return cellEditor.getValue( value == null ? "" : value );
//		return cellEditor.getValue( (value != null ) ? value : "" );
	}

	/**
	 * データ編集用のHTML文字列を作成します。
	 * 行番号付の編集データを作成します。名前_行番号 で登録する為、
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * @param   row		行番号
	 * @param   value	入力値
	 *
	 * @return  データ編集用の文字列
	 * @og.rtnNotNull
	 */
	public String getEditorValue( final int row,final String value ) {
		// 6.4.1.1 (2016/01/16) PMD refactoring. Avoid if (x != y) ..; else ..;
		return cellEditor.getValue( row, value == null ? "" : value );
//		return cellEditor.getValue( row,(value != null ) ? value : "" );
	}

	/**
	 * データ出力用の固定長文字列を作成します。
	 * HOST送信用桁数がセットされていれば、そちらを優先します。
	 *
	 * @og.rev 3.5.4.5 (2004/01/23) エンコード指定に変更します。
	 * @og.rev 6.0.4.0 (2014/11/28) データ出力用のレンデラーと名称がかぶるので、変更します。
	 *
	 * @param   value	対象の値
	 * @param   encode	固定長で変換する文字エンコード
	 *
	 * @return  データ編集用の文字列
	 */
	public String getFixedValue( final String value,final String encode ) {
		return dbType.valueFill( value ,columnData.getSizeX(), columnData.getSizeY(), encode );
	}

	/**
	 * データの表示用レンデラーを返します。
	 *
	 * @og.rev 3.8.0.2 (2005/07/11) 新規追加
	 *
	 * @return	データの表示用レンデラー
	 */
	public String getRenderer() {
		return columnData.getRenderer() ;
	}

	/**
	 * データの編集用エディターを返します。
	 *
	 * @og.rev 3.8.0.2 (2005/07/11) 新規追加
	 *
	 * @return	データの編集用エディター
	 */
	public String getEditor() {
		return columnData.getEditor() ;
	}

	/**
	 * 文字種別名を返します。
	 * カラムの文字種別名名称を返します。
	 * これは，HTML上の各種タグに，データベース定義に応じたクラスを
	 * セットし,CSS(Cascading Style Sheet)の class="xxxxx" とする事により
	 * 各種スタイルを表現するのに使用します。
	 *
	 * ここでは, カラムリソースの DBTYPE 属性で指定の文字列(X,S9,KXなど)を返します。
	 *
	 * @return  データの文字種別(X,KX,S9 など)
	 */
	public String getDbType() {
		return columnData.getDbType() ;
	}

	/**
	 * データのNATIVEの型の識別コードを返します。
	 *
	 * @og.rev 4.1.1.2 (2008/02/28) Enum型(fukurou.model.NativeType)に変更
	 *
	 * @return  NATIVEの型の識別コード(DBType で規定)
	 * @see org.opengion.fukurou.model.NativeType
	 */
	public NativeType getNativeType() {
		return dbType.getNativeType();
	}

	/**
	 * そのカラムの,デフォルト値の値を返します。
	 *
	 * カラムリソースに デフォルト情報が登録されている場合は,その値を返します。
	 * デフォルト値が設定されていない場合は, null を返します。
	 *
	 * @return  デフォルト値(無ければ null)
	 */
	public String getDefault() {
		return defValue;
	}

	/**
	 * 表示用レンデラーのパラメータを取得します。
	 *
	 * @og.rev 3.4.0.0 (2003/09/01) 新規追加
	 *
	 * @return	表示用レンデラーのパラメータ
	 */
	public String getRendererParam() {
		return columnData.getRendererParam();
	}

	/**
	 * 編集用エディターのパラメータを取得します。
	 *
	 * @og.rev 3.4.0.0 (2003/09/01) 新規追加
	 *
	 * @return	編集用エディターのパラメータ
	 */
	public String getEditorParam() {
		return columnData.getEditorParam();
	}

	/**
	 * データタイプのパラメータを取得します。
	 *
	 * @og.rev 3.4.0.0 (2003/09/01) 新規追加
	 *
	 * @return	データタイプのパラメータ
	 */
	public String getDbTypeParam() {
		return columnData.getDbTypeParam();
	}

	/**
	 * カラムロールを取得します。
	 *
	 * @og.rev 4.0.0.0 (2005/11/30) 新規追加
	 *
	 * @return	カラムロール
	 */
	public String getRoles() {
		return columnData.getRoles();
	}

	/**
	 * カラムオブジェクトのロールモードを返します。
	 *
	 * @og.rev 4.3.0.0 (2008/07/04) ロールモードマルチ対応
	 *
	 * @return カラムオブジェクトのロールモード
	 */
	public RoleMode getRoleMode() {
		return columnData.getRoleMode();
	}

	/**
	 * 接続先IDを返します。
	 *
	 * @return  接続先ID
	 */
	public String getDbid() {
		return dbid;
	}

	/**
	 * String引数の文字列を＋１した文字列を返します。
	 * これは、英字の場合(A,B,C など)は、B,C,D のように,最終桁の文字コードを
	 * ＋１ します。
	 * 文字列が数字タイプの場合は, 数字に変換して、+1 します。(桁上がりもあり)
	 * 混在タイプの場合は,最後の桁だけを確認して ＋１します。
	 * 引数が null の場合と、ゼロ文字列("")の場合は,引数を,そのまま返します。
	 *
	 * ※ 機能拡張：第２引数に指定の文字列(数字、日付等)を指定する事で、
	 *    引数の文字列に、任意の値を加算できるようにします。
	 *    ただし、すべての DBTypeではなく、ある程度特定します。
	 *    対象外の DBTypeで、第２引数が null 出ない場合は、Exception を Throwsします。
	 *    第２引数が、null の場合は、従来と同じ＋１します。
	 *
	 * @og.rev 5.6.0.3 (2012/01/24) ADD に、引数の値を加算する機能を追加します。
	 *
	 * @param   value  引数の文字列
	 * @param   add    加算する文字列(null の場合は、従来と同じ、+1 します。)
	 *
	 * @return  引数の文字列を＋１した文字列。または、任意の値を加算した文字列。
	 */
	public String valueAdd( final String value,final String add ) {
		// DBType の実装の関係で、旧メソッドは残しておきます。
		// 6.4.1.1 (2016/01/16) PMD refactoring. A method should have only one exit point, and that should be the last statement in the method
		return add == null || add.isEmpty() ? dbType.valueAdd( value ) : dbType.valueAdd( value , add );

//		if( add == null || add.isEmpty()  ) {
//			return dbType.valueAdd( value );
//		}
//		else {
//			return dbType.valueAdd( value , add );
//		}
	}

	/**
	 * データが登録可能かどうかをチェックします。
	 * データがエラーの場合は、そのエラー内容を返します。
	 *
	 * @og.rev 3.6.0.0 (2004/09/22) dbType パラメータを引数に追加
	 * @og.rev 3.6.0.7 (2004/11/06) official属性がfalseの場合、チェックレベルに対応したチェックを行う。
	 * @og.rev 3.8.0.8 (2005/10/03) 企画改善 61200-050921-02 ラベルからHTMLタグを削除する。
	 * @og.rev 5.2.2.0 (2010/11/01) 内部処理を、#valueCheck( value , true ) に委譲。
	 *
	 * @param   value	チェック対象の値
	 *
	 * @return  エラー内容  正常時は null
	 * @see		#valueCheck( String , boolean )
	 */
	public ErrorMessage valueCheck( final String value ) {
		return valueCheck( value , true );
	}

	/**
	 * データが登録可能かどうかをチェックします。
	 * データがエラーの場合は、そのエラー内容を返します。
	 *
	 * @og.rev 3.6.0.0 (2004/09/22) dbType パラメータを引数に追加
	 * @og.rev 3.6.0.7 (2004/11/06) official属性がfalseの場合、チェックレベルに対応したチェックを行う。
	 * @og.rev 3.8.0.8 (2005/10/03) 企画改善 61200-050921-02 ラベルからHTMLタグを削除する。
	 * @og.rev 5.2.2.0 (2010/11/01) 厳密にチェック(isStrict=true)するフラグを追加
	 *
	 * @param   value		チェック対象の値
	 * @param   isStrict	厳密にチェック(isStrict=true)するかどうか
	 *
	 * @return  エラー内容  正常時は null
	 */
	public ErrorMessage valueCheck( final String value , final boolean isStrict ) {
		// 3.8.0.8 (2005/10/03) ラベルからHTMLタグを削除する。
		String lbl = labelData.getLabel() ;
		if( lbl.indexOf( '<' ) >= 0 ) {
			lbl = lbl.replaceAll( "<[^>]*>","" );
		}

		// 5.2.2.0 (2010/11/01) 厳密にチェック(isStrict=true)するフラグを追加
		final ErrorMessage errMsg = dbType.valueCheck( lbl,value,columnData.getSizeX(), columnData.getSizeY(),columnData.getDbTypeParam() , isStrict );
		if( checkLevel >= 0 ) {
			// ERR0034:指定のカラムオブジェクトには、カラムリソースが存在しません。name={0} label={1}
			errMsg.addMessage( 0,checkLevel,"ERR0034", columnData.getName(),labelData.getLabel() );
		}
		return errMsg ;
	}

	/**
	 * エディターで編集されたデータを登録する場合に、データそのものを
	 * 変換して、実登録データを作成します。
	 * 例えば,大文字のみのフィールドなら、大文字化します。
	 * 実登録データの作成は、DBType オブジェクトを利用しますので,
	 * これと DBCellEditor とがアンマッチの場合は、うまくデータ変換
	 * されない可能性がありますので、注意願います。
	 *
	 * @param   value 一般に編集データとして登録されたデータ
	 *
	 * @return  修正後の文字列(一般にデータベースに登録するデータ)
	 */
	public String valueSet( final String value ) {
		return dbType.valueSet( value );
	}

	/**
	 * action で指定されたコマンドを実行して、値の変換を行います。
	 * oldValue(旧データ)は、元のDBTableModelに設定されていた値です。通常は、
	 * この値を使用してカラム毎に変換を行います。newValue(新データ)は、引数で
	 * 指定された新しい値です。この値には、パラメータを指定して変換方法を
	 * 制御することも可能です。
	 * 指定のアクションがカラムで処理できない場合は、エラーになります。
	 *
	 * @param   action アクションコマンド
	 * @param   oldValue 入力データ(旧データ)
	 * @param   newValue 入力データ(新データ)
	 *
	 * @return	実行後のデータ
	 */
	public String valueAction( final String action,final String oldValue,final String newValue ) {
		return dbType.valueAction( action,oldValue,newValue );
	}

	/**
	 * 内部の設定情報オブジェクトを返します。
	 * このオブジェクトを ローカルで書き換えて、DBColumn を作るようにします。
	 *
	 * @og.rev 3.1.0.0 (2003/03/20) DBColumnConfig 関係の見直し。
	 * @og.rev 3.4.0.0 (2003/09/01) 表示パラメータ、編集パラメータ、文字パラメータの追加。
	 * @og.rev 3.5.5.5 (2004/04/23) size 属性の意味を変更、maxlength 属性を追加。
	 * @og.rev 3.5.5.8 (2004/05/20) codeName 属性を追加。
	 * @og.rev 3.6.0.7 (2004/11/06) カラムリソースから作成されたかどうか(official属性追加)
	 * @og.rev 5.1.8.0 (2010/07/01) すべての属性を設定するようにします。
	 * @og.rev 5.5.1.0 (2012/04/03) useSLabel対応
	 * @og.rev 5.6.2.3 (2013/03/22) 非表示文字の設定。noDisplayVal追加
	 * @og.rev 5.7.6.3 (2014/05/23) stringOutput対応
	 * @og.rev 6.2.0.0 (2015/02/27) フィールドサイズ 追加(VIEW_LENGTHと分離して、役割を明確にする)
	 * @og.rev 6.2.3.0 (2015/05/01) writeKeyLabel 属性を追加
	 * @og.rev 6.3.3.0 (2015/07/25) eventValue 追加
	 *
	 * @return	設定情報オブジェクト
	 */
	public DBColumnConfig getConfig() {
		// 互換性確保のため、DBColumnConfig の コンストラクタは修正していません。

		final DBColumnConfig config =
			new DBColumnConfig(
						lang							,
						columnData.getName()			,
						labelData						,
						columnData.getClassName()		,
						columnData.getFieldSize()		,		// 6.2.0.0 (2015/02/27)
						columnData.getViewLength()		,		// 6.2.0.0 (2015/02/27)
						columnData.getMaxlength()		,
						String.valueOf( writable )		,
						columnData.getRenderer()		,
						columnData.getEditor()			,
						codeData						,
						columnData.getDbType()			,
						defValue						,
						columnData.getRendererParam()	,
						columnData.getEditorParam()		,
						columnData.getDbTypeParam()		,
						columnData.getRoles()			,		// 4.0.0 (2005/11/30)
						official						,		// 3.6.0.7 (2004/11/06)
						dbid	 ) ;

		// 5.1.8.0 (2010/07/01) すべての属性を設定するようにします。
		// コンストラクタは修正していないため、セッターメソッド経由で渡します。
		config.setRendererAttributes( rendAttri );
		config.setEditorAttributes( editAttri );
		config.setAddNoValue( addNoValue );
		config.setAddKeyLabel( addKeyLabel );		// 6.0.4.0 (2014/11/28) キー:ラベル形式で表示するかどうか
		config.setWriteKeyLabel( writeKeyLabel	);	// 6.2.3.0 (2015/05/01)
		config.setEventColumn( eventColumn );
		config.setEventValue( eventValue );			// 6.3.3.0 (2015/07/25) eventValue 追加
		config.setRawEditParameter( rawEditParameter );
		config.setRawRendParameter( rawRendParameter );
		config.setEventURL( eventURL );
		config.setUseSLabel( useSLabel );			// 5.5.1.0 (2012/04/03)
		config.setNoDisplayVal( noDisplayVal );		// 5.6.2.3 (2013/03/22) 非表示文字の設定
		config.setStringOutput( stringOutput );		// 5.7.6.3 (2014/05/23) 

		return config ;
	}

	/**
	 * 表示用レンデラーの追加属性を返します。
	 *
	 * @og.rev 3.1.0.0 (2003/03/20) DBColumnConfig オブジェクトの内部キープを廃止
	 *
	 * @return   属性リスト
	 */
	public Attributes getRendererAttributes() {
		return rendAttri ;
	}

	/**
	 * 編集用エディター用の追加属性を返します。
	 *
	 * @og.rev 3.1.0.0 (2003/03/20) DBColumnConfig オブジェクトの内部キープを廃止
	 *
	 * @return   属性リスト
	 */
	public Attributes getEditorAttributes() {
		return editAttri ;
	}

	/**
	 * メニューに空の選択リストを追加するかどうかを取得します。
	 *
	 * @og.rev 3.5.5.7 (2004/05/10) 新規追加
	 *
	 * @return	空の選択リストを追加するかどうか(true:追加する/false:追加しない)
	 */
	public boolean isAddNoValue() {
		return addNoValue ;
	}

	/**
	 * セレクトメニューの場合、キー:ラベル形式で表示するかどうか[true/false/null]を返します。
	 *
	 * これは、上位入れ子のタグの OptionTag で、addKeyLabel を取り出して、
	 * true であれば、キー:ラベル形式 のオプションを、#addOption( String ) で
	 * 登録させます。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) キー:ラベル形式で表示するかどうか。新規追加
	 *
	 * @return	true:キー:ラベル形式/false:ラベルのみ/null:指定通り
	 */
	public String getAddKeyLabel() {
		return addKeyLabel;
	}

	/**
	 * セレクトメニューの場合、キーとラベルを分けて出力するかどうか[true/false]を取得します。
	 *
	 * カラムがプルダウンメニュー形式の場合、ファイル出力時に、キーとラベルを分離して出力する事ができます。
	 * この属性と、addKeyLabel とは、微妙に関連していて、addKeyLabel="true" のみの場合は、
	 * キー:ラベル形式で、出力されますが、writeKeyLabel="true" も指定されている場合は、こちらが優先されます。
	 * なお、この指定が有効なのは、セレクトメニューのカラムのみです。
	 *
	 * @og.rev 6.2.3.0 (2015/05/01) キー／ラベルで分離して出力するかどうか。
	 *
	 * @return	キー／ラベル分離出力するかどうか[true:分離出力/false:通常]
	 */
	public boolean isWriteKeyLabel() {
		return writeKeyLabel ;
	}

	/**
	 * このカラムオブジェクトがカラムリソースより正式に作られたかどうかを取得します。
	 * カラムリソースが無い場合は、仮オブジェクトかデータベースメタデータより作成されます。
	 * その場合は、チェック機能が緩くなるため、正式なカラムオブジェクトと区別する為に
	 * この属性を持ちます。
	 *
	 * @og.rev 3.6.0.7 (2004/11/06) 新規追加
	 *
	 * @return	正式に作られたかどうか(true:正式/false:暫定)
	 */
	public boolean isOfficial() {
		return official ;
	}

	/**
	 * カラムのラベルデータオブジェクトを返します。
	 *
	 * @return  カラムのラベルデータオブジェクト
	 */
	public LabelData getLabelData() {
		return labelData;
	}

	/**
	 * カラムのコードデータオブジェクトを返します。
	 * コードデータが存在しない場合は、null を返します。
	 * 受け取り側で、null かどうか判定してから使用してください。
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) codeData が null でも、そのまま返します。
	 *
	 * @return  カラムのコードデータオブジェクト
	 */
	public CodeData getCodeData() {
		return codeData;
	}

	/**
	 * フィールドの入力枠サイズを返します。
	 *
	 * これは、設定された値そのものを返しますので、未設定の時は、null が返ります。
	 *
	 * テキストフィールドのサイズに該当します。
	 * 何も指定しない場合は、null が返ります。
	 * その場合の、入力枠サイズは、maxlength が使用されます。
	 * ただし、桁数が大きい場合は、システム定数の HTML_COLUMNS_MAXSIZE や、
	 * HTML_VIEW_COLUMNS_MAXSIZE で指定された値が使われます。
	 * それらの値よりも、ここで取得 した fieldSize が優先されます。
	 *
	 * @og.rev 6.2.0.0 (2015/02/27) フィールドサイズ 追加(VIEW_LENGTHと分離して、役割を明確にする)
	 *
	 * @return  入力枠サイズ
	 */
	public String getFieldSize() {
		return columnData.getFieldSize();
	}

	/**
	 * 引数の最大入力サイズより、実際のフィールドのサイズを求めます。
	 *
	 * 計算方法
	 *    ① fieldSize があれば、その値を返します。
	 *    ② ColumnData#getTotalSize() を求め、あれば、その値を返す。
	 *    ③ totalSizeが、引数の最大入力サイズより大きい場合は、引数の最大入力サイズを返す。
	 *
	 * 各呼び出すメソッドは、引数の最大入力サイズ のみ指定します。それ以外の値は、内部の値を使います。
	 * 最大入力サイズは、システム定数の、HTML_COLUMNS_MAXSIZE か、HTML_VIEW_COLUMNS_MAXSIZE が
	 * 一般的です。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) getFieldSize メソッドを XHTMLTag から DBColumn へ移動
	 * @og.rev 4.0.0.0 (2007/02/05) ADJUSTMENT 処理を廃止します。(CSSにて対応)
	 * @og.rev 6.2.0.0 (2015/02/27) フィールドサイズ 追加(VIEW_LENGTHと分離して、役割を明確にする)
	 * @og.rev 6.2.3.0 (2015/05/01) maxlength 変数は、0 の場合は、無制限になります。
	 *
	 * @param maxlength	  最大入力サイズ
	 *
	 * @return 表示すべきサイズ
	 * @see #getFieldSize()
	 */
	public int getFieldSize( final int maxlength ) {

		// ① fieldSize があれば、その値を返します。
		final String fixSize = columnData.getFieldSize();
		if( fixSize != null && !fixSize.isEmpty() ) { 
			return Integer.parseInt( fixSize );
		}

		// ② ColumnData#getTotalSize() を求め、あれば、その値を返す。
		int size = columnData.getTotalSize();

		// ③ totalSizeが、引数の最大入力サイズより大きい場合は、引数の最大入力サイズを返す。
		if( maxlength > 0 && ( size <= 0 || size > maxlength ) ) {
			size = maxlength;
		}

		return size;
	}

	/**
	 * イベントカラム(親カラム)。
	 *
	 * @og.rev 4.3.6.0 (2009/04/01)
	 *
	 * @return イベントカラム
	 */
	public String getEventColumn() {
		return eventColumn ;
	}

	/**
	 * イベントカラムの子カラムの値を出力するためのSQL文を返します。
	 *
	 * @og.rev 6.3.3.0 (2015/07/25) eventValue 追加
	 *
	 * @return イベントカラムの値SQL文
	 */
	public String getEventValue() {
		return eventValue ;
	}

	/**
	 * {&#064;XXXX}を変換していない状態の編集パラメータを返します。
	 *
	 * @og.rev 4.3.6.0 (2009/04/01)
	 *
	 * @return 生編集パラメータ
	 */
	public String getRawEditParam() {
		return rawEditParameter ;
	}

	/**
	 * {&#064;XXXX}を変換していない状態の編集パラメータを返します。
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 *
	 * @return 生表示パラメータ
	 */
	public String getRawRendParam() {
		return rawRendParameter ;
	}

	/**
	 * eventColumn利用時にJSで利用するURL。
	 *
	 * @og.rev 4.3.6.0 (2009/04/01)
	 *
	 * @return イベントURL
	 */
	public String getEventURL() {
		return eventURL ;
	}

	/**
	 * 引数のタグ文字列に対して、イベントカラムの実行に必要なspanタグを付加した
	 * 文字列を返します。
	 *
	 * @param tag タグ文字列
	 * @param initVal 子カラムの初期値
	 * @param writable タグの要素が書き込み可能かどうか
	 *
	 * @return spanタグを付加したタグ文字列
	 */
	public String getEventColumnTag( final String tag, final String initVal, final boolean writable ) {
		return getEventColumnTag( tag, initVal, -1, writable );
	}

	/**
	 * 引数のタグ文字列に対して、イベントカラムの実行に必要なspanタグを付加した
	 * 文字列を返します。
	 *
	 * @param tag タグ文字列
	 * @param initVal 子カラムの初期値
	 * @param row 行番号
	 * @param writable タグの要素が書き込み可能かどうか
	 *
	 * @return spanタグを付加したタグ文字列
	 * @og.rtnNotNull
	 */
	public String getEventColumnTag( final String tag, final String initVal, final int row, final boolean writable ) {
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		final String name = columnData.getName()
							+ ( row < 0 ? "" : HybsSystem.JOINT_STRING + row ) ;

		return new TagBuffer( "span" )
					.add( "class"							, HybsSystem.EVENT_COLUMN_CLASS )
					.add( HybsSystem.EVENT_COLUMN_ID		, name )
					.add( HybsSystem.EVENT_COLUMN_INITVAL	, initVal )
					.add( HybsSystem.EVENT_COLUMN_WRITABLE	, String.valueOf( writable ) )
					.addBody( tag )
					.makeTag();

	}

	/**
	 * セットされている表示パラメータ、編集パラメータに"{&#064;XXXX}"が含まれているか(パラメーターのパースが必要か)を
	 * 返します。
	 *
	 * @og.rev 6.0.0.1 (2014/04/25) 内部処理変更
	 *
	 * @return "{&#064;XXXX}"が含まれているか(含まれている場合true)
	 */
	public boolean isNeedsParamParse() {
		return	( rawRendParameter != null && rawRendParameter.indexOf( "{@" ) >= 0 )
			||	( rawEditParameter != null && rawEditParameter.indexOf( "{@" ) >= 0 );
	}

	/**
	 * ラベル短ベースのメニューにするかどうか。
	 *
	 * @og.rev 5.5.1.0 (2012/04/03)
	 *
	 * @return イベントカラム
	 */
	public String getUseSLabel() {
		return useSLabel ;
	}

	/**
	 * 非表示文字列を返します。
	 *
	 * これは、例えば、数字型の場合は、0 がデフォルトなどの場合、
	 * あえて表示したくないなどのケースに使います。
	 * そのような状況が設定されていない場合は、null が返されます。
	 *
	 * @og.rev 5.6.2.3 (2013/03/22) 新規追加
	 *
	 * @return 非表示文字
	 */
	public String getNoDisplayVal() {
		return noDisplayVal ;
	}

	/**
	 * レンデラー利用ファイル出力時に数値等も文字タイプに固定するかどうか
	 *		(初期値:USE_STRING_EXCEL_OUTPUT[={@og.value SystemData#USE_STRING_EXCEL_OUTPUT}])。
	 *
	 * これは、レンデラーでのExcel出力時のセルタイプに影響します。
	 * trueの場合は数値も全て文字として出力します。
	 * falseの場合はDBTYPEに従った出力になりるため、数値型の DBTYPE では、doubleに一度変換されます。
	 * (初期値:USE_STRING_EXCEL_OUTPUT[={@og.value SystemData#USE_STRING_EXCEL_OUTPUT}])。
	 *
	 * @og.rev 5.7.6.3 (2014/05/23) 新規追加
	 *
	 * @return レンデラー利用時の出力方法 (true:String出力/false:DBTYPEに従う)
	 */
	public boolean isStringOutput() {
		return stringOutput ;
	}

	/**
	 * マルチ・キーセレクトを使用するかどうかを返します。
	 * true：使用する。false:使用しない です。
	 * ただし、実際に使用するかどうかは、HTML出力時に決めることが出来ます。
	 * ここでは、USE_MULTI_KEY_SELECT が true で、USE_SIZE(=20)以上の場合に
	 * true を返します。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) Selection 側から移動。
	 * @og.rev 6.3.9.1 (2015/11/27) ３項演算子を || or &amp;&amp; で簡素化できる(PMD)。
	 *
	 * @return  選択リストで、マルチ・キーセレクトを使用するかどうか(true:使用する)
	 */
	public boolean useMultiSelect() {
//		return codeData == null ? false : codeData.useMultiSelect() ;
		return codeData != null && codeData.useMultiSelect() ;
	}

	/**
	 * オブジェクトの文字列表現を返します。
	 *
	 * n=Name,l=Label,c=ClassName,r=Renderer,e=Editor,d=DbType という形式を作成します。
	 *
	 * @og.rev 6.2.2.0 (2015/03/27) DBColumnオブジェクトの文字列表現(#toString())を実装します。
	 *
	 * @return	文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return "n="	+ columnData.getName()
			+ ",l=" + labelData.getLabel()
			+ ",c=" + columnData.getClassName()
			+ ",r=" + columnData.getRenderer()
			+ ",e=" + columnData.getEditor()
			+ ",d=" + columnData.getDbType() ;
	}
}
