/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.resource.CodeData;
import static org.opengion.fukurou.system.HybsConst.CR ;				// 6.1.0.0 (2014/12/26)
import static org.opengion.fukurou.system.HybsConst.BUFFER_LARGE;		// 6.1.0.0 (2014/12/26) refactoring

/**
 * データのコード情報を取り扱うクラスです。
 *
 * コードのキーとラベルの情報から、HTMLのメニューやリストを作成するための オプション
 * タグを作成したり、与えられたキーをもとに、チェック済みのオプションタグを作成したり
 * します。
 *
 * @og.group 選択データ制御
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Selection_RADIO extends Selection_NULL {
	private final CodeData codeData ;

	/**
	 * コンストラクター
	 *
	 * @param	cdData	コードデータオブジェクト
	 *
	 */
	public Selection_RADIO( final CodeData cdData ) {
		super();		// 6.4.1.1 (2016/01/16) PMD refactoring. It is a good practice to call super() in a constructor
		if( cdData == null ) {
			final String errMsg = "コードリソースが定義されていません。" + CR ;
			throw new HybsSystemException( errMsg );
		}

		codeData = cdData ;
	}

	/**
	 * 初期値が選択済みの 選択肢(オプション)を返します。
	 * このオプションは、引数の値を初期値とするオプションタグを返します。
	 * ※ このクラスでは実装されていません。
	 *
	 * @og.rev 5.1.3.0 (2010/02/01) 追加
	 *
	 * @param	selectValue	選択されている値
	 * @param	seqFlag	シーケンスアクセス機能の指定
	 * @param	useShortLabel	短ラベルの指定
	 *
	 * @return  オプションタグ
	 */
	@Override
	public String getOption( final String selectValue,final boolean seqFlag, final boolean useShortLabel ) {
		final String errMsg = "このクラスでは実装されていません。";
		throw new UnsupportedOperationException( errMsg );
	}

	/**
	 * 初期値が選択済みの 選択肢(オプション)を返します。
	 * このオプションは、引数の値を初期値とするオプションタグを返します。
	 *
	 * @og.rev 2.1.0.1 (2002/10/17) 選択リストを、正方向にしか選べないようにする sequenceFlag を導入する
	 * @og.rev 3.5.6.3 (2004/07/12) キャッシュを利用せず毎回タグを作成します。
	 * @og.rev 3.8.6.0 (2006/09/29) useLabel 属性 追加
	 * @og.rev 6.2.2.4 (2015/04/24) メソッド変更。旧 #getRadio( String , String , boolean )
	 *
	 * @param   name         ラジオの name
	 * @param   selectValue  選択されている値
	 * @param   useLabel     ラベル表示の有無 [true:有/false:無]
	 *
	 * @return  オプションタグ
	 * @og.rtnNotNull
	 */
	@Override
	public String getOption( final String name,final String selectValue,final boolean useLabel ) {
		final String inputTag = "<input type=\"radio\" name=\"" + name + "\" value=\"" ;
		final StringBuilder buf = new StringBuilder( BUFFER_LARGE );
		final int size = codeData.getSize();
		for( int i=0; i<size; i++ ) {
			final String value = codeData.getCodeKey(i);
			if( useLabel ) { buf.append( "<label>" ); }
			buf.append( inputTag ).append( value ).append( '"' );		// 6.0.2.5 (2014/10/31) char を append する。
			if( value.equals( selectValue ) ) {
				buf.append( " checked=\"checked\"" );
			}
			buf.append( "/>" );
			if( useLabel ) { buf.append( codeData.getShortLabel(i) ).append( "</label>" ); }
		}
		return buf.toString();
	}

	/**
	 * 選択肢(value)に対するラベルを返します。
	 * 選択肢(value)が、存在しなかった場合は、選択肢そのものを返します。
	 * このメソッドでは、短縮ラベルを返すかどうかを指定するフラグを指定します。
	 * getValueLabel( XX,false ) は、getValueLabel( XX ) と同じです。
	 *
	 * @og.rev 4.0.0.0 (2005/11/30) を追加
	 *
	 * @param	selectValue	選択肢の値
	 * @param	isSLbl	短縮ラベルを使用する [true:使用する/false:しない]
	 *
	 * @return  選択肢のラベル
	 * @see     #getValueLabel( String )
	 */
	@Override
	public String getValueLabel( final String selectValue,final boolean isSLbl ) {
		// マッチするアドレスを探す。
		final int selected = codeData.getAddress( selectValue );

		// 6.4.1.1 (2016/01/16) PMD refactoring. A method should have only one exit point, and that should be the last statement in the method
		return selected < 0
					? selectValue			// マッチしなければ、選択肢そのものを返す。
					: isSLbl
						? codeData.getShortLabel(selected)
						: codeData.getLongLabel(selected);

//		if( selected < 0 ) {
//			// マッチしなければ、選択肢そのものを返す。
//			return selectValue;
//		}
//		else {
//			if( flag ) {
//				return codeData.getShortLabel(selected);
//			}
//			else {
//				return codeData.getLongLabel(selected);
//			}
//		}
	}

}
