/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.fukurou.system.OgBuilder ;				// 6.4.4.1 (2016/03/18)
import org.opengion.fukurou.db.DBUtil;
import org.opengion.fukurou.db.ApplicationInfo;
import org.opengion.fukurou.util.StringFormat;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.AbstractRenderer;
import org.opengion.hayabusa.db.CellRenderer;
import org.opengion.hayabusa.db.DBColumn;

/**
 * MULTIQUERY レンデラーは、表示パラメータで指定された SQL文の実行結果を表示するクラスです。
 *
 * 複数件のデータが取得された場合、各データは、spanタグで囲われて返されます。
 *  <span>値1</span><span>値2</span><span>値3</span>
 *
 * 第2カラムを指定した場合、その値は、spanタグのclass属性として付加されます。
 *
 * また、SQL文の指定には、元のValue を、$1 として使用可能です。
 * ($Cで自身のカラム名を参照することも可能です)
 *
 * カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.rev 5.1.8.0 (2010/07/01) 新規追加
 * @og.group データ表示
 *
 * @version  5.0
 * @author   Hiroki Nakamura
 * @since    JDK5.0,
 */
public class Renderer_MULTIQUERY extends AbstractRenderer {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.4.1 (2016/03/18)" ;

	private final String query;
	private final String dbid;
	private final String name;

	/** コネクションにアプリケーション情報を追記するかどうか指定 */
	public static final boolean USE_DB_APPLICATION_INFO  = HybsSystem.sysBool( "USE_DB_APPLICATION_INFO" ) ;

	private final ApplicationInfo appInfo;
	private static final String SYSTEM_ID  = HybsSystem.sys( "SYSTEM_ID" ) ;

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 */
	public Renderer_MULTIQUERY() {
		super();		// 6.4.1.1 (2016/01/16) PMD refactoring. It is a good practice to call super() in a constructor
		query   = null;
		dbid	= null;
		appInfo = makeApplicationInfo( null );
		name	= null;
	}

	/**
	 * デフォルトコンストラクター。
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	private Renderer_MULTIQUERY( final DBColumn clm ) {
		super();		// 6.4.1.1 (2016/01/16) PMD refactoring. It is a good practice to call super() in a constructor
		query = clm.getRendererParam();
		dbid  = clm.getDbid();
		appInfo = makeApplicationInfo( clm.getName() );
		name  = clm.getName();
	}

	/**
	 * アクセスログ取得の為,ApplicationInfoオブジェクトを作成します。
	 *
	 * @param	name	オブジェクト
	 *
	 * @return	ApplicationInfoオブジェクト
	 */
	private ApplicationInfo makeApplicationInfo( final String name ) {
		// 6.3.9.1 (2015/11/27) Found 'DD'-anomaly for variable(PMD)
//		ApplicationInfo infoTemp = null;
		final ApplicationInfo infoTemp ;
		if( USE_DB_APPLICATION_INFO ) {
			infoTemp = new ApplicationInfo();
			// ユーザーID,IPアドレス,ホスト名
			infoTemp.setClientInfo( SYSTEM_ID,HybsSystem.HOST_ADRS,HybsSystem.HOST_NAME );
			// 画面ID,操作,プログラムID
			infoTemp.setModuleInfo( "Renderer_MULTIQUERY",null,name );
		}
		else {
			infoTemp = null;			// 6.3.9.1 (2015/11/27)
		}

		return infoTemp ;
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellRendererオブジェクト
	 * @og.rtnNotNull
	 */
	public CellRenderer newInstance( final DBColumn clm ) {
		return new Renderer_MULTIQUERY( clm );
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * ここでは、AAA:BBB:CCC:DDD という値を、$1,$2,$3,$4 に割り当てなおして、
	 * QUERYを実行します。また、$1 は、本来の値として、メニューの初期値設定等に
	 * 使用します。上記の例では、AAA が値で、それ以降は、引数になります。
	 * さらに、元の文字列"AAA:BBB:CCC:DDD"は、$0 に割り当てられます。割り当てがない
	 * 変数は、""(ゼロ文字列)として、扱われます。
	 * 又、$Cには自分自身のカラム名を割り当てます。
	 *
	 * 複数件のデータが取得された場合、各データは、spanタグで囲われて返されます。
	 *  <span>値1</span><span>値2</span><span>値3</span>
	 *  第2カラムを指定した場合、その値は、spanタグのclass属性として付加されます
	 *
	 * @og.rev 5.7.9.0 (2014/08/08) DBUtil.dbExecute 実行時エラーを回避
	 * @og.rev 6.4.4.1 (2016/03/18) StringBuilderの代わりに、OgBuilderを使用する。
	 *
	 * @param   value 入力値
	 *
	 * @return  データの表示用文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final String value ) {
		final StringFormat format = new StringFormat( query, value, name );
		final String str = format.format();

		// 5.7.9.0 (2014/08/08) DBUtil.dbExecute 実行時エラーを回避
		String rtnVal = (value == null) ? "" : value ;		// 本来、QUERYで変換すべきだが、元の値を設定する。
		try {
			final String[][] rtn = DBUtil.dbExecute( str, null, appInfo, dbid );

			if( rtn != null && rtn.length > 0 && rtn[0] != null && rtn[0].length > 0 ) {
				// 6.3.9.1 (2015/11/27) Found 'DD'-anomaly for variable(PMD)
//				boolean isUseClass = false;
//				if( rtn[0].length > 1 ) {
//					isUseClass = true;
//				}
				final boolean isUseClass = rtn[0].length > 1 ;

//				final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
				final OgBuilder buf = new OgBuilder();
				for( int i=0; i<rtn.length; i++ ) {
					buf.append( "<span" )
						.appendIf( isUseClass , " class=\"" , rtn[i][1] )
						.append( "\">" , rtn[i][0] , "</span>" );
//					if( isUseClass ) {
//						buf.append( " class=\"" ).append( rtn[i][1] ).append( '"' );	// 6.0.2.5 (2014/10/31) char を append する。
//					}
//					buf.append( '>' ).append( rtn[i][0] ).append( "</span>" );			// 6.0.2.5 (2014/10/31) char を append する。
				}
				rtnVal = buf.toString();
			}
		}
		catch( RuntimeException ex ) {
			final String errMsg = "SQL文の処理に失敗しました。" + CR
								+ "query=" + query + " , inValue=" + value + " , name=" + name
								+ CR
								+ ex.getMessage() ;
			System.err.println( errMsg );
		}

		return rtnVal;
	}

	/**
	 * データ出力用の文字列を作成します。
	 * ファイル等に出力する形式を想定しますので、HTMLタグを含まない
	 * データを返します。
	 * 基本は、#getValue( String ) をそのまま返します。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) データ出力用のレンデラー
	 *
	 * @param   value 入力値
	 *
	 * @return  データ出力用の文字列
	 * @og.rtnNotNull
	 * @see		#getValue( String )
	 */
	@Override
	public String getWriteValue( final String value ) {
		return ( value == null ) ? "" : value;
	}
}
