/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.transfer;

import org.opengion.fukurou.system.OgRuntimeException ;		// 6.4.2.0 (2016/01/29)
import org.opengion.fukurou.util.StringUtil;

/**
 * 伝送要求に対してのHTTP経由で旧伝送DB(CB01)への登録を行います。
 *
 * 実行対象は、(データコード) (送り先) (テキスト種別) ([プロトコル]://[ホスト名]:[ポート番号]/[コンテキスト名]/)
 * の形式で指定します。
 *
 * 例)実行対象 : 3 D9 B991 http://localhost:8824/gf/
 *
 * 要求を受けた際の動作は_CB01と同じです。
 * _CB01同様にCB01テーブル以外に次のテーブル及びシーケンスが必要です
 *  テーブル：CB02
 *  シーケンス：CB010001,CB010002
 *
 * <del>6.3.9.0 (2015/11/06)その他の処理内容については、{@link org.opengion.fukurou.transfer.AbstractTransferExecHTTP}及び</del>
 * その他の処理内容については、{@link org.opengion.fukurou.transfer.TransferExec_HTTP}及び
 * {@link org.opengion.fukurou.transfer.TransferExec_CB01}のJavaDocを参照して下さい。
 *
 * @og.rev 5.4.4.1 (2012/02/02) HybsSystemExceptionは利用しない
 * @og.rev 6.3.9.0 (2015/11/06) extendsクラス名を、Abstract付に変更する。
 * @og.group 伝送システム
 *
 * @version  5.0
 * @author   Hiroki.Nakamura
 * @since    JDK1.6
 */
// public class TransferExec_HTTP_CB01 extends TransferExec_HTTP {
public class TransferExec_HTTP_CB01 extends AbstractTransferExecHTTP {

	private String remoteHost		;	// リモート接続先URL
	private String remoteExecObj	;	// リモート接続先の実行対象

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public TransferExec_HTTP_CB01() {
		super();		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。
	}

	/**
	 * ローカルの実行対象を、リモート接続先の実行対象とリモート接続先URLに分解します。
	 *
	 * @param localExecObj ローカルの実行対象
	 */
	@Override
	protected void splitExecObj( final String localExecObj ) {
		final String[] obj = StringUtil.csv2Array( localExecObj, ' ' );
		if( obj.length < 4 ) {
			final String errMsg = "実行対象は、(データコード) (送り先) (テキスト種別) ([プロトコル]://[ホスト名]:[ポート番号]/[コンテキスト名]/) の形式で指定して下さい。[EXECOBJ=" + localExecObj + "]";
			// throw new HybsSystemException( errMsg );
			throw new OgRuntimeException( errMsg ); // 5.4.4.1 (2012/02/02)
		}
		final String hcdd = obj[0];
		final String hto = obj[1];
		final String hsyu = obj[2];
		remoteHost = obj[3];
		if( hcdd == null || hcdd.isEmpty()
		 || hto  == null || hto.isEmpty()
		 || hsyu == null || hsyu.isEmpty()
		 || remoteHost == null || remoteHost.isEmpty() ) {
			final String errMsg = "実行対象は、(データコード) (送り先) (テキスト種別) ([プロトコル]://[ホスト名]:[ポート番号]/[コンテキスト名]/) は必須です。[EXECOBJ=" + localExecObj + "]";
			throw new OgRuntimeException( errMsg );
		}

		remoteExecObj = hcdd + " " + hto + " " + hsyu;
	}

	/**
	 * リモート接続先URLを返します。
	 * このメソッドは、{@link #splitExecObj(String)}の後に呼び出しする必要があります。
	 *
	 * @return リモート接続先URL
	 */
	@Override
	public String getRemoteHost() {
		if( remoteHost == null || remoteHost.isEmpty() ) {
			final String errMsg = "先に#splitExecObj(String)を実行して下さい";
			throw new OgRuntimeException( errMsg );
		}
		return remoteHost;
	}

	/**
	 * リモート接続先の実行対象を返します。
	 * このメソッドは、{@link #splitExecObj(String)}の後に呼び出しする必要があります。
	 *
	 * @return 接続URL
	 */
	@Override
	public String getRemoteExecObj() {
		if( remoteExecObj == null || remoteExecObj.isEmpty() ) {
			final String errMsg = "先に#splitExecObj(String)を実行して下さい";
			throw new OgRuntimeException( errMsg );
		}
		return remoteExecObj;
	}
}
